/*******************************************************************************
* Copyright (C) 2021 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*   Content : Intel(R) oneAPI Math Kernel Library (oneMKL) Sparse BLAS C OpenMP
*             offload example for mkl_sparse_s_mm
*
********************************************************************************
*
* Consider the matrix A (see 'Sparse Storage Formats for Sparse BLAS Level 2
* and Level 3 in the  Intel oneMKL Reference Manual')
*
*                 |   1       -1      0   -3     0   |
*                 |  -2        5      0    0     0   |
*   A    =        |   0        0      4    6     4   |,
*                 |  -4        0      2    7     0   |
*                 |   0        8      0    0    -5   |
*
*  The matrix A is represented in a zero-based compressed sparse row (CSR) storage
*  scheme with three arrays (see 'Sparse Matrix Storage Schemes' in the
*   Intel oneMKL Reference Manual) as follows:
*
*         values  =  ( 1 -1 -3 -2  5  4  6  4 -4  2  7  8 -5 )
*         columns =  ( 0  1  3  0  1  2  3  4  0  2  3  1  4 )
*         row_index = ( 0        3     5        8       11    13 )
*
*  The test computes the following operations :
*
*       A*X = Y using mkl_sparse_s_mm omp offload
*       where A is an MxN general sparse matrix and X and Y are dense matrices.
*
********************************************************************************
*/
#include <assert.h>
#include <omp.h>
#include <stdio.h>
#include <stdlib.h>

#include "common_for_sparse_examples.h"
#include "mkl.h"
#include "mkl_omp_offload.h"

int main()
{
//*******************************************************************************
//     Declaration and initialization of parameters for sparse representation of
//     the matrix A in the CSR format:
//     Assume A is M-by-N matrix.
//*******************************************************************************
#define M     5 // nRows of A
#define N     5 // nCols of A
#define NRHS  2
#define NNZ   13
#define LDX   NRHS
#define LDY   NRHS
#define ALIGN 64

    //*******************************************************************************
    //    Declaration of local variables:
    //*******************************************************************************
    // Descriptor of main sparse matrix properties
    struct matrix_descr descrA;
    // Create matrix descriptor
    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;

    // Structure with sparse matrix stored in CSR format
    sparse_matrix_t csrA = NULL;

    sparse_index_base_t indexing = SPARSE_INDEX_BASE_ZERO;
    sparse_operation_t opA       = SPARSE_OPERATION_NON_TRANSPOSE;
    sparse_layout_t layout       = SPARSE_LAYOUT_ROW_MAJOR;

    const MKL_INT ldx = LDX;
    const MKL_INT ldy = LDY;

    float alpha = 1.0f, beta = 0.0f;
    MKL_INT i;

    float *values      = (float *)mkl_malloc(sizeof(float) * NNZ, ALIGN);
    MKL_INT *columns   = (MKL_INT *)mkl_malloc(sizeof(MKL_INT) * NNZ, ALIGN);
    MKL_INT *row_index = (MKL_INT *)mkl_malloc(sizeof(MKL_INT) * (M + 1), ALIGN);
    float *x           = (float *)mkl_malloc(sizeof(float) * N * ldx, ALIGN);
    float *w           = (float *)mkl_malloc(sizeof(float) * N * ldx, ALIGN);

    float *y  = (float *)mkl_malloc(sizeof(float) * M * ldy, ALIGN);
    float *z1 = (float *)mkl_malloc(sizeof(float) * M * ldy, ALIGN);
    float *z2 = (float *)mkl_malloc(sizeof(float) * M * ldy, ALIGN);

    const int num_pointers = 8;
    void *pointer_array[num_pointers];
    pointer_array[0] = values;
    pointer_array[1] = columns;
    pointer_array[2] = row_index;
    pointer_array[3] = x;
    pointer_array[4] = w;
    pointer_array[5] = y;
    pointer_array[6] = z1;
    pointer_array[7] = z2;

    if (!values || !columns || !row_index || !x || !y || !z1 || !z2 || !w) {
        free_allocated_memories(pointer_array, num_pointers);
        return 1;
    }

    //*******************************************************************************
    //    Sparse representation of the matrix A
    //*******************************************************************************
    float init_values[NNZ] = {1.0f, -1.0f, -3.0f, -2.0f, 5.0f, 4.0f, 6.0f,
                              4.0f, -4.0f, 2.0f,  7.0f,  8.0f, -5.0f};
    MKL_INT init_columns[NNZ]     = {0, 1, 3, 0, 1, 2, 3, 4, 0, 2, 3, 1, 4};
    MKL_INT init_row_index[M + 1] = {0, 3, 5, 8, 11, 13};
    float init_x[N * LDX]         = {1.0f, 5.0f, 3.0f, 4.0f, 2.0f, 2.0f, 10.0f, 6.0f, 8.0f, 4.0f};

    for (i = 0; i < NNZ; i++) {
        values[i]  = init_values[i];
        columns[i] = init_columns[i];
    }
    for (i = 0; i < M + 1; i++) {
        row_index[i] = init_row_index[i];
    }

    for (i = 0; i < N * ldx; i++) {
        x[i] = init_x[i];
        w[i] = init_x[i];
    }

    for (i = 0; i < M * ldy; i++) {
        y[i]  = 0.0f;
        z1[i] = 0.0f;
        z2[i] = 0.0f;
    }

    //  Task: Obtain matrix-matrix multiply A*x --> y
    //          A - zero-based indexing,
    //          x - row major ordering
    printf("\n EXAMPLE PROGRAM FOR mkl_sparse_s_mm omp_offload async execution. \n");
    printf("--------------------------------------------------------------------\n");
    printf("\n");
    printf("   INPUT DATA FOR mkl_sparse_s_mm omp offload async execution\n");
    printf("   WITH GENERAL SPARSE MATRIX     \n");
    printf("   ALPHA = %4.1f  BETA = %4.1f    \n", alpha, beta);
    printf("   SPARSE_OPERATION_NON_TRANSPOSE \n");
    printf("   Input vector                   \n");
    for (i = 0; i < N; i++) {
        printf("%7.1f, %7.1f\n", x[2 * i], x[2 * i + 1]);
    }

    sparse_status_t ie_status;

    // Create handle with matrix stored in CSR format
    ie_status = mkl_sparse_s_create_csr(&csrA, indexing,
                                        M, // number of rows
                                        N, // number of cols
                                        row_index, row_index + 1, columns, values);
    if (ie_status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_s_create_csr: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    ie_status = mkl_sparse_s_mm(opA, alpha, csrA, descrA, layout, x, NRHS, ldx, beta, y, ldy);
    if (ie_status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_s_mm: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    // Release matrix handle and deallocate matrix
    ie_status = mkl_sparse_destroy(csrA);
    if (ie_status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_destroy: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    printf("\n");
    printf("   OUTPUT DATA FOR mkl_sparse_d_mm \n");
    for (i = 0; i < M; i++) {
        printf("%7.1f, %7.1f\n", y[2 * i], y[2 * i + 1]);
    }

    printf("-------------------------------------------------------\n");
    fflush(stdout);

    const int devNum = 0;

    sparse_matrix_t csrA_gpu = NULL;

    sparse_status_t status_create;
    sparse_status_t status_mm1;
    sparse_status_t status_mm2;
    sparse_status_t status_destroy;

// call create_csr/mm/destroy via omp_offload.
#pragma omp target data map(to:row_index[0:M+1],columns[0:NNZ],values[0:NNZ],x[0:N*ldx],w[0:N*ldx]) \
                        map(tofrom:z1[0:M*ldy],z2[0:M*ldy]) device(devNum)
    {
        printf("Create CSR matrix via omp_offload\n");

#pragma omp dispatch device(devNum)
        status_create = mkl_sparse_s_create_csr(&csrA_gpu, indexing, M, N,
                                                row_index, row_index + 1, columns, values);

        printf("Compute mkl_sparse_s_mm via omp_offload\n");

#pragma omp dispatch device(devNum) nowait
        status_mm1 = mkl_sparse_s_mm(opA, alpha, csrA_gpu, descrA,
                                     layout, x, NRHS, ldx, beta, z1, ldy);

#pragma omp dispatch device(devNum) nowait
        status_mm2 = mkl_sparse_s_mm(opA, alpha, csrA_gpu, descrA,
                                     layout, w, NRHS, ldx, beta, z2, ldy);
#pragma omp taskwait
        printf("Destroy the CSR matrix via omp_offload\n");

#pragma omp dispatch device(devNum)
        status_destroy = mkl_sparse_destroy(csrA_gpu);
    }

    int flps_per_value = 2 * ((NNZ / M) + 1);
    int status1        = 0;
    int status2        = 0;

    int status_offload = status_create | status_mm1 | status_mm2 | status_destroy;
    if (status_offload != 0) {
        printf("\tERROR: status_create = %d, status_mm1 = %d, status_mm2 = %d, status_destroy = %d\n",
               status_create, status_mm1, status_mm2, status_destroy);
        goto cleanup;
    }

    printf("   OUTPUT DATA FOR mkl_sparse_s_mm_omp_offload async execution\n");
    for (i = 0; i < M; i++) {
        printf("%7.1f, %7.1f\n", z1[2 * i], z1[2 * i + 1]);
    }
    printf("---------------------------------------------------\n");
    fflush(stdout);

    status1 = validation_result_float(y, z1, M * ldy, flps_per_value);
    status2 = validation_result_float(y, z2, M * ldy, flps_per_value);

cleanup:
    free_allocated_memories(pointer_array, num_pointers);

    const int status_all = status1 | status2 | status_offload;
    printf("Test %s\n", status_all == 0 ? "PASSED" : "FAILED");
    fflush(stdout);

    return status_all;
}
