# Copyright (C) 2025 Intel Corporation
# SPDX-License-Identifier: MIT

import argparse
from typing import List, Dict

from mpp.core.devices import DeviceType


class DefaultFileIds:
    METRIC_CHART_FILE_PATH = DeviceType.CORE
    INPUT_DATA_FILE_PATH = 'input_data_file_path'
    DIRECTORY = 'directory'


class MultiFileSpecFormatter:
    """
    Formatter for command line options that accept file names/paths and may include
    a prepended core type specifier. Handles formatting file paths for both single and hybrid core platforms.
    This class focuses purely on formatting logic, not validation.
    """

    class MultiFileSpecError(Exception):
        def __init__(self, message):
            super().__init__(message)

    def __init__(self, default_file_id=None):
        """
        Initialize the formatter.
        
        :param default_file_id: Default file ID for non-hybrid platforms
        """
        self.__file_id: str = default_file_id or DefaultFileIds.METRIC_CHART_FILE_PATH
        self.__file: str = ''

    @property
    def file_id(self):
        return self.__file_id

    @file_id.setter
    def file_id(self, file_id: str):
        self.__file_id = file_id

    def __call__(self, file_spec: str):
        """
        Format a file specification string into a dictionary.
        Note: This method assumes the input has already been validated.
        
        :param file_spec: A string that represents a file with path. For hybrid files, the string is
                prepended with <core_type>=
                Non-hybrid file format: <file name with path>
                hybrid file format: <core_type>=<file name with path>
        :return: a single element dictionary where key:value is <core_type>:<file name with path>,
                 { <core_type>: <file name with path> }, for non-hybrid entries, the core_type is the
                 default core type.
        """
        self._parse_file_spec(file_spec)
        return self._construct_file_arg()

    def _construct_file_arg(self):
        """Construct the formatted file argument dictionary."""
        return {self.__file_id: self.__file}

    def _parse_file_spec(self, file_spec: str):
        """
        Parse the file specification for hybrid and non-hybrid entries
        :param file_spec: a single or hybrid core file specification
        """
        self.__file = file_spec
        hybrid_arg_separator: str = "="
        if hybrid_arg_separator in file_spec:
            args = file_spec.split(hybrid_arg_separator)
            # Basic parsing check - detailed validation should be done elsewhere
            if len(args) == 2 and args[0] and args[1]:
                core_type = args[0].lower()
                # Check if the core type is the default - this should not be used as hybrid
                if core_type == DeviceType.CORE:
                    raise argparse.ArgumentTypeError(f'Cannot use default core type "{DeviceType.CORE}" as hybrid specification.')
                self.__file_id = core_type
                self.__file = args[1]
            else:
                raise argparse.ArgumentTypeError(f'Invalid format: {file_spec}. '
                                                 f'Expected: <core type>=<file name with path>.')

    @staticmethod
    def reformat_multi_file_args(file_spec_list):
        """
        Reformat a list of file specification dictionaries into a single dictionary.
        This is pure formatting logic - no validation.
        
        :param file_spec_list: list of dictionaries, where each dictionary has one key:value pair
        :return: single dictionary where all file specifications are merged
        """
        if not file_spec_list:
            return {}
        
        if isinstance(file_spec_list, bool):
            return {}
        
        if not isinstance(file_spec_list, list):
            raise argparse.ArgumentTypeError(f'Expected list, got {type(file_spec_list)}')
        
        file_dict = {}
        for file_spec_dict in file_spec_list:
            if not isinstance(file_spec_dict, dict):
                raise argparse.ArgumentTypeError(f'Expected dict in list, got {type(file_spec_dict)}')
            
            if len(file_spec_dict) != 1:
                raise argparse.ArgumentTypeError(f'Each file specification dict must have exactly one key:value pair, got {len(file_spec_dict)}')
            
            file_dict.update(file_spec_dict)
        
        return file_dict


class IntegerRangeListFormatter:
    """
    Formatter for command line options that accept a list of integers in a range format.
    The format is expected to be a comma-separated list of integers or ranges (e.g., "1,2,3,5-7").
    """

    def __init__(self, name):
        """
        Initialize the formatter with an optional initial value.

        :param name: The name of the argument
        """
        self.name = name

    def format(self, value: str) -> Dict[str, List[int]]:
        """
        Format a string of integers and ranges into a list of integers.

        :param value: A string representing a list of integers and ranges
        :return: A list of integers parsed from the input string
        """
        if not value:
            return []

        result = []
        for part in value.split(','):
            if '-' in part:
                start, end = map(int, part.split('-'))
                result.extend(range(start, end + 1))
            else:
                result.append(int(part))
        return {self.name: result}
