/*******************************************************************************
* Copyright (C) 2025 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/



/* Reads matrix from a Matrix Market file format.
* Input_file should be already opened prior to calling this function
* also, the user should close the file outside the call to the function
* This is a modified version of mm_read_unsymmetric_sparse found in mmio.c
* file by National Institute of Standards and Technology (NIST).
* The modification include capability to read single and double precision
* values as well as long long int indices.
*/
MKL_INT x_mm_read_unsymmetric_sparse_mkl(FILE *input_file, MKL_INT *M_pt, MKL_INT *N_pt, MKL_INT **ax_ptr, MKL_INT **ay_ptr, DATA_TYPE **a_ptr, MKL_INT *nnz_pt, MKL_INT input_indexing, MKL_INT output_indexing, MKL_INT *is_symm)
{
    MKL_INT retcode = 0;
    MM_typecode matcode;
    MKL_INT m, n, nnz, i;
    MKL_INT *ax = NULL, *ay = NULL;
    DATA_TYPE  *a = NULL;
    FILE *file = NULL;

    file = input_file;
    if (file == NULL)
    {
        printf("Error:NULL input argument as a file for read_matrix_mtx\n");
        return -1;
    }

    if (mm_read_banner(file, &matcode) != 0)
    {
        printf("Could not process Matrix Market banner.\n");
        return -1;
    }

    if(mm_is_symmetric(matcode))
    {
        (*is_symm) = 1;
    }
    if(mm_is_general(matcode))
    {
        (*is_symm) = 0;
    }

    if( !mm_is_matrix(matcode) || !mm_is_sparse(matcode) || mm_is_pattern(matcode) )
    {
        printf("Sorry, matrix reader does not support ");
        char *matcode_str = mm_typecode_to_str(matcode);
        if(matcode_str != NULL)
        {
            fprintf(stderr, "Matrix Market type: [%s]\n", matcode_str);
            free(matcode_str);
        }
        else
        {
            fprintf(stderr, "Matrix Market type: unknown\n");
        }
        fflush(0);
        return -1;
    }
    /*  This is how one can screen matrix types if their application */
    /*  only supports a subset of the Matrix Market data types.      */
    if (mm_is_complex(matcode))
    {
        printf("Sorry, matrix reader does not currently support complex valued Matrix Market files\n"); fflush(0);
        return -1;
    }

    /* find out size of sparse matrix .... */
    if ((retcode = mm_read_mtx_crd_size_mkl(file, &m, &n, &nnz)) !=0)
    {
        printf("Error in reading matrix dimensions.\n");
        return retcode;
    }

    if( ((m<=0) || (n<=0)) || (nnz<=0) )
    {
        printf("The file seem to be corrupted. Dimensions read: nrows=%lld, ncol=%lld, nnz=%lld\n", (long long int) m, (long long int) n, (long long int) nnz);
        return -1;
    }


    ax = (MKL_INT  *)mkl_malloc(sizeof(MKL_INT  )*nnz, 4096);
    if(ax == NULL)
    {
        printf("Error in allocating memory\n");
        return -1;
    }
    ay = (MKL_INT  *)mkl_malloc(sizeof(MKL_INT  )*nnz, 4096);
    if(ay == NULL)
    {
        mkl_free(ax);
        printf("Error in allocating memory\n");
        return -1;
    }
    a  = (DATA_TYPE*)mkl_malloc(sizeof(DATA_TYPE)*nnz, 4096);
    if(a == NULL)
    {
        mkl_free(ax);
        mkl_free(ay);
        printf("Error in allocating memory\n");
        return -1;
    }

    int num_items_read;
    for(i=0;i<nnz;i++) {
#ifdef SINGLE_PRECISION
#ifdef MKL_ILP64
        num_items_read = fscanf(file,"%lld %lld %f\n",&ax[i],&ay[i], &a[i]);
#else
        num_items_read = fscanf(file,"%d %d %f\n",&ax[i],&ay[i], &a[i]);
#endif
#else
#ifdef MKL_ILP64
        num_items_read = fscanf(file,"%lld %lld %lf\n",&ax[i],&ay[i], &a[i]);
#else
        num_items_read = fscanf(file,"%d %d %lf\n",&ax[i],&ay[i], &a[i]);
#endif
#endif

        if(num_items_read != 3)
        {
            printf("Error reading matrix at nonzero %lld\n", (long long int) i);
            fflush(0);
            mkl_free(ax);
            mkl_free(ay);
            mkl_free(a);
            return -1;
        }

#ifdef DEBUG_
        printf("ax[%lld] = %lld ay[%lld] = %lld a[%lld] = %e \n", (long long int)i, ax[i], (long long int)i, ay[i], (long long int)i, a[i]);
#endif
        if (input_indexing == 1 && output_indexing == 0)
        {
            ax[i]--;
            ay[i]--;
        }
        else if (input_indexing == 0 && output_indexing == 1)
        {
            ax[i]++;
            ay[i]++;
        }
        if ((ax[i]<output_indexing)||(ax[i]>n + (output_indexing - 1))||(ay[i]<output_indexing)||(ay[i]>n + (output_indexing-1)))
        {
            printf("Error reading matrix at nonzero %lld. Index (%lld, %lld) out of bound.\n", (long long int) i, (long long int)(ax[i]), (long long int)(ay[i]));
            fflush(0);
            mkl_free(ax);
            mkl_free(ay);
            mkl_free(a);
            return -1;
        }
    }

    *ax_ptr = ax;
    *ay_ptr = ay;
    *a_ptr = a;
    *M_pt = m;
    *N_pt = n;
    *nnz_pt = nnz;

    return 0;
}

//This is the interface function
//It reads a matrix from file and returns a sparse_matrix_t in CSR fomrat
//is_symm is set to 1 if matrix is symmetric, else 0
sparse_matrix_t x_matrix_reader(char *matrix_filename, const MKL_INT indexing, const sparse_operation_t operation, MKL_INT *is_symm)
{
    printf("Reading matrix from %s \n", matrix_filename);

    FILE *matrix_file = NULL;
    matrix_file = fopen(matrix_filename,"rt");
    if (!matrix_file)
    {
        printf("Cannot open file with the matrix: %s\n", matrix_filename);
        return NULL;
    }

    MKL_INT nrows = 0, ncols = 0, nnz=0;
    MKL_INT input_indexing=1; //MatrixMarket has 1-based indexing
    MKL_INT *ax = NULL, *ay = NULL;
    DATA_TYPE  *acoo=NULL;
    MKL_INT retcode;
    retcode = x_mm_read_unsymmetric_sparse_mkl(matrix_file, &nrows, &ncols, &ax, &ay, &acoo, &nnz, input_indexing, indexing, is_symm);
    if(retcode != 0)
    {
        //Error output already thrown
        fclose(matrix_file);
        return NULL;
    }

    fclose(matrix_file);
    //--------------------------------------------------------------------------
    // create the matrix
    //--------------------------------------------------------------------------
    sparse_matrix_t A_coo = NULL, A_csr = NULL;
    MKL_INT stat;
    sparse_index_base_t one_or_zero = (indexing==0)?SPARSE_INDEX_BASE_ZERO: SPARSE_INDEX_BASE_ONE;

    stat = mkl_sparse_x_create_coo ( &A_coo,
            one_or_zero,
            nrows, ncols,
            nnz,
            ax, ay, acoo );
    if(stat!=SPARSE_STATUS_SUCCESS)
    {
        printf("Error in mkl_sparse_x_create_coo.\n");
        mkl_free(ax);
        mkl_free(ay);
        mkl_free(acoo);
        return NULL;
    }

    stat = mkl_sparse_convert_csr ( A_coo,
            operation,
            &A_csr );
    if(stat!=SPARSE_STATUS_SUCCESS)
    {
        printf("Error in mkl_sparse_convert_csr.\n");
        mkl_free(ax);
        mkl_free(ay);
        mkl_free(acoo);
        return NULL;
    }

    stat = mkl_sparse_destroy(A_coo);
    if(stat!=SPARSE_STATUS_SUCCESS)
    {
        printf("Error in mkl_sparse_convert_csr.\n");
        mkl_free(ax);
        mkl_free(ay);
        mkl_free(acoo);
        return NULL;
    }

    if(ax)
    {
        mkl_free(ax);
    }
    if(ay)
    {
        mkl_free(ay);
    }
    if(acoo)
    {
        mkl_free(acoo);
    }
    return A_csr;
}
