/*******************************************************************************
* Copyright (C) 2021 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*   Content : Intel(R) oneAPI Math Kernel Library (Intel(R) oneMKL) Sparse BLAS
*   C example using Compressed Sparse Column (CSC) Matrix Format
*
********************************************************************************
*
* Example program for using Intel oneMKL Inspector-Executor Sparse BLAS routines
* for matrices represented in the compressed sparse column (CSC) sparse storage
* format.
*
* The following Inspector Executor Sparse Blas routines are used in the example:
*
*   Initialization/Destruction stage:
*          mkl_sparse_d_create_csc
*          mkl_sparse_destroy
*
*   Inspector stage:
*          mkl_sparse_set_mv_hint  mkl_sparse_set_sv_hint
*          mkl_sparse_set_mm_hint  mkl_sparse_set_sm_hint
*          mkl_sparse_optimize
*
*   Executor stage:
*          mkl_sparse_d_mv         mkl_sparse_d_trsv
*          mkl_sparse_d_mm         mkl_sparse_d_trsm
*
* Consider the matrix A (see Appendix 'Sparse Storage Formats for Sparse Blas
* level 2-3')
*
*                 |   1       -1     -3    0     0   |
*                 |  -2        5      0    0     0   |
*   A    =        |   0        0      4    6     4   |,
*                 |  -4        0      2    7     0   |
*                 |   0        8      0    0    -5   |
*
*
* decomposed as
*
*                      A = L + D + U,
*
*  where L is the strict  lower triangle of A, U is the strictly  upper triangle
*  of A, D is the main diagonal. Namely
*
*        |   0    0   0    0     0   |       |  0   -1   -3    0   0   |
*        |  -2    0   0    0     0   |       |  0    0    0    0   0   |
*   L  = |   0    0   0    0     0   |,  U=  |  0    0    0    6   4   |
*        |  -4    0   2    0     0   |       |  0    0    0    0   0   |
*        |   0    8   0    0     0   |       |  0    0    0    0   0   |
*
*
*           |   1  0  0   0   0   |
*           |   0  5  0   0   0   |
*   D    =  |   0  0  4   0   0   |.
*           |   0  0  0   7   0   |
*           |   0  0  0   0  -5   |
*
*  The matrix A given above is represented in the compressed sparse row  storage scheme
*  with the help of three arrays, two of length nnz=13 and one of length nrows+1
*  (see Appendix 'Sparse Storage Formats for Sparse Blas level 2-3'
*
*          colPtr  = ( 0        3      6     9  11    13)
*          rowInd  = ( 0  1  3  0 1 4  0 2 3 2 3 2  4 )
*          values  = ( 1 -2 -4 -1 5 8 -3 4 2 6 7 4 -5 )
*
*  In what follows the symbol ' means transposition of object preceding the symbol.
*
*  Using notation A = (L + D + U) for provided matrix elements and I for identity
*  matrix, the test performs the following operations :
*
*       1. The example computes (L+D)'*x_m_original = y_m using mkl_sparse_d_mm  where x_m_original is a known 5 by 2
*          matrix and then the example solves the system (L+D)'*x_m_calculated = y_m with the help of
*          mkl_sparse_d_trsm. It's evident that x_m_calculated should be equal to x_m_original.
*
*       2. The example computes (U+I)*x_v_calculated = y_v using mkl_sparse_d_mv where x_v_calculated is a vector
*          and then the example calls mkl_sparse_d_trsv which  solves the system
*          (U+I)*x_v_calculated = y_v with the single right hand side. It's evident that x_v_calculated
*          should be equal to x_v_original.
*
*       3. The example computes D*x_v_original = y_v using mkl_sparse_d_mv  where x_v_original is a vector
*          and then the example solves the system D*x_v_calculated = y_v with the single right hand side.
*          It's evident that x_v_calculated should be equal to x_v_original.
*
*       4. The next step is the computation (L+D+L') x_v_original = y_v using mkl_sparse_d_mv
*          where x_v_original is a vector. It is easy to see that L+D+L' is a symmetric matrix.
*
*       5. The next step is the computation A'* x_v_original = y_v using mkl_sparse_mv where x_v_original
*          is a vector.
*
* The code given below uses only one sparse representation for the all operations.
********************************************************************************
*/
#include <stdio.h>
#include <math.h>
#include "mkl_types.h"
#include "mkl_spblas.h"

#ifdef MKL_ILP64
#define INT_PRINT_FORMAT "%lld"
#else
#define INT_PRINT_FORMAT "%d"
#endif

#define ALMOST_EQUAL(a, b) (fabs((a)-(b)) < 1e-10 ? 1 : 0)

int main () {
//*******************************************************************************
//     Definition arrays for sparse representation of the matrix A in
//     the coordinate format:
//*******************************************************************************
#define M 5    /* nrows = ncols = M */
#define NNZ 13
#define NRHS 2
    MKL_INT m = M, nrhs = NRHS;

    MKL_INT colIndex[M+1] = {   0,               3,              6,              9,        11,       13};
    MKL_INT rows[NNZ]     = {   0,   1,    3,    0,   1,   4,    0,   2,   3,    2,   3,    2,   4};
    double values[NNZ]    = { 1.0, -2.0, -4.0, -1.0, 5.0, 8.0, -3.0, 4.0, 2.0,  6.0, 7.0, 4.0, -5.0};

//*******************************************************************************
//    Declaration of local variables :
//*******************************************************************************

    double   x_m_original[M*NRHS]   = {1.0, 5.0,
                                       1.0, 4.0,
                                       1.0, 3.0,
                                       1.0, 2.0,
                                       1.0, 1.0};

    double   y_m[M*NRHS]            = {0.0, 0.0,
                                       0.0, 0.0,
                                       0.0, 0.0,
                                       0.0, 0.0,
                                       0.0, 0.0};

    double   x_m_calculated[M*NRHS] = {0.0, 0.0,
                                       0.0, 0.0,
                                       0.0, 0.0,
                                       0.0, 0.0,
                                       0.0, 0.0};

    double      x_v_original[M]   = {1.0, 1.0, 1.0, 1.0, 1.0};
    double      y_v[M]            = {0.0, 0.0, 0.0, 0.0, 0.0};
    double      x_v_calculated[M] = {0.0, 0.0, 0.0, 0.0, 0.0};

    double      alpha = 1.0, beta = 0.0;
    MKL_INT     i, j;
    struct matrix_descr descrA;
    sparse_matrix_t cscA;

    sparse_status_t status;
    int exit_status = 0;

    printf( "\n EXAMPLE PROGRAM FOR CSC format routines from IE Sparse BLAS\n" );
    printf( "-------------------------------------------------------\n" );

//*******************************************************************************
//   Create CSC sparse matrix handle and analyze step
//*******************************************************************************

    status = mkl_sparse_d_create_csc( &cscA,
                                      SPARSE_INDEX_BASE_ZERO,
                                      m,    // number of rows
                                      m,    // number of cols
                                      colIndex,
                                      colIndex+1,
                                      rows,
                                      values );

    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_d_create_csc: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //*******************************************************************************
    // First we set hints for the different operations before calling the
    // mkl_sparse_optimize() api which actually does the analyze step.  Not all
    // configurations have optimized steps, so the hint apis may return status
    // MKL_SPARSE_STATUS_NOT_SUPPORTED (=6) if no analysis stage is actually available
    // for that configuration.
    //*******************************************************************************

    //*******************************************************************************
    // Set hints for Task 1: Lower triangular transpose MM and SM solve with
    // non-unit diagonal and row-major format
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_LOWER;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_set_mm_hint(cscA, SPARSE_OPERATION_TRANSPOSE, descrA,
                                    SPARSE_LAYOUT_ROW_MAJOR, nrhs, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 1: mkl_sparse_set_mm_hint: %d \n", status);
    }

    status = mkl_sparse_set_sm_hint(cscA, SPARSE_OPERATION_TRANSPOSE, descrA,
                                    SPARSE_LAYOUT_ROW_MAJOR, nrhs, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 1: mkl_sparse_set_sm_hint: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //*******************************************************************************
    // Set hints for Task 2: Upper triangular transpose MV and SV solve
    // with unit diagonal
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_UPPER;
    descrA.diag = SPARSE_DIAG_UNIT;

    status = mkl_sparse_set_mv_hint(cscA, SPARSE_OPERATION_NON_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 2: mkl_sparse_set_mv_hint: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    status = mkl_sparse_set_sv_hint(cscA, SPARSE_OPERATION_NON_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 2: mkl_sparse_set_sv_hint: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //*******************************************************************************
    // Set hints for Task 3: Diagonal MV and SV
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_DIAGONAL;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_set_mv_hint(cscA, SPARSE_OPERATION_NON_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 3: mkl_sparse_set_mv_hint: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    status = mkl_sparse_set_sv_hint(cscA, SPARSE_OPERATION_NON_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 3: mkl_sparse_set_sv_hint: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //*******************************************************************************
    // Set hints for Task 4: Lower symmetric MV with non-unit diagonal
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_SYMMETRIC;
    descrA.mode = SPARSE_FILL_MODE_LOWER;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_set_mv_hint(cscA, SPARSE_OPERATION_NON_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 4: mkl_sparse_set_mv_hint: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //*******************************************************************************
    // Set hints for Task 5: General transpose MV with non-unit diagonal
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_set_mv_hint(cscA, SPARSE_OPERATION_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 5: mkl_sparse_set_mv_hint: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //*******************************************************************************
    // Analyze sparse matrix; choose proper kernels and workload balancing strategy
    //*******************************************************************************
    status = mkl_sparse_optimize ( cscA );
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_optimize: %d \n", status);
        exit_status = 1;
        goto exit;
    }

//*******************************************************************************
//    Task 1.  Obtain Triangular matrix-matrix multiply (L+D)' *x_m_original --> y_m
//    and solve triangular system   (L+D)' *x_m_calculated = y_m with multiple right
//    hand sides. Array x_m_calculated must be equal to the array x_m_original
//*******************************************************************************
    printf("                                  \n");
    printf("   TASK 1:                        \n");
    printf("   INPUT DATA FOR mkl_sparse_d_mm \n");
    printf("   WITH LOWER TRIANGULAR MATRIX   \n");
    printf("     m = " INT_PRINT_FORMAT "   nrhs = " INT_PRINT_FORMAT "\n", m, nrhs);
    printf("     ALPHA = %4.1f  BETA = %4.1f  \n", alpha, beta);
    printf("     SPARSE_OPERATION_TRANSPOSE   \n" );
    printf("   Input matrix                   \n");
    for (i = 0; i < m; i++) {
        for (j = 0; j < nrhs; j++) {
            printf("%7.1f", x_m_original[i*nrhs+j]);
        };
        printf("\n");
    };

    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_LOWER;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_d_mm( SPARSE_OPERATION_TRANSPOSE, alpha, cscA, descrA,
                              SPARSE_LAYOUT_ROW_MAJOR, x_m_original, nrhs, nrhs, beta,
                              y_m, nrhs);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 1 mkl_sparse_d_mm: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("                                   \n");
    printf("   OUTPUT DATA FOR mkl_sparse_d_mm \n");
    printf("   WITH TRIANGULAR MATRIX          \n");
    for (i = 0; i < m; i++) {
        for (j = 0; j < nrhs; j++) {
            printf("%7.1f", y_m[i*nrhs+j]);
        };
        printf("\n");
    };

    printf("-----------------------------------------------\n");
    printf("   Solve triangular system   \n");
    printf("   with obtained             \n");
    printf("   right hand side           \n");
    printf("                             \n");

    status =  mkl_sparse_d_trsm( SPARSE_OPERATION_TRANSPOSE, alpha, cscA, descrA, SPARSE_LAYOUT_ROW_MAJOR, y_m, nrhs, nrhs, x_m_calculated, nrhs);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 1 mkl_sparse_d_trsm: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("                                     \n");
    printf("   OUTPUT DATA FOR mkl_sparse_d_trsm \n");
    printf("   WITH TRIANGULAR MATRIX            \n");
    for (i = 0; i < m; i++) {
        for (j = 0; j < nrhs; j++) {
            printf("%7.1f", x_m_calculated[i*nrhs+j]);
        };
        printf("\n");
    };

    printf( "   Validating output data\n" );
    printf( "   against input data\n" );
    for (i = 0; i < m*nrhs; i++) {
        if (!ALMOST_EQUAL(x_m_original[i], x_m_calculated[i])) {
            printf( " Error in Task 5: output data is not equal to" );
            printf( " input data \n" );
            exit_status = 1;
            goto exit;
        };
    };
    printf( "   Done \n" );
    printf("-----------------------------------------------\n");

//*******************************************************************************
//    Task 2.    Obtain Triangular matrix-vector multiply (U+I) *x_v_original --> y_v
//    and solve triangular system   (U+I) *x_v_calculated = y_v with single right hand sides
//    Array x_v_calculated must be equal to the array x_v_original
//*******************************************************************************
    printf("                                     \n");
    printf("   TASK 2:                           \n");
    printf("   INPUT DATA FOR mkl_sparse_d_mv    \n");
    printf("   WITH UNIT UPPER TRIANGULAR MATRIX \n");
    printf("     ALPHA = %4.1f  BETA = %4.1f     \n", alpha, beta);
    printf("     SPARSE_OPERATION_NON_TRANSPOSE  \n" );
    printf("   Input vector                      \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", x_v_original[i]);
    };

    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_UPPER;
    descrA.diag = SPARSE_DIAG_UNIT;

    status = mkl_sparse_d_mv( SPARSE_OPERATION_NON_TRANSPOSE, alpha, cscA, descrA, x_v_original, beta, y_v);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 2 mkl_sparse_d_mv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("                                   \n");
    printf("   OUTPUT DATA FOR mkl_sparse_d_mv \n");
    printf("   WITH TRIANGULAR MATRIX          \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", y_v[i]);
    };
    printf("-----------------------------------------------\n");
    printf("   Solve triangular system   \n");
    printf("   with obtained             \n");
    printf("   right hand side           \n");

    status =  mkl_sparse_d_trsv( SPARSE_OPERATION_NON_TRANSPOSE, alpha, cscA, descrA, y_v, x_v_calculated);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 2 mkl_sparse_d_trsv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("                                     \n");
    printf("   OUTPUT DATA FOR mkl_sparse_d_trsv \n");
    printf("   WITH TRIANGULAR MATRIX            \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", x_v_calculated[i]);
    };

    printf( "   Validating output data\n" );
    printf( "   against input data\n" );
    for (i = 0; i < m; i++) {
        if (!ALMOST_EQUAL(x_v_original[i], x_v_calculated[i])) {
            printf( " Error in Task 2: output data is not equal to" );
            printf( " input data \n" );
            exit_status = 1;
            goto exit;
        }
    };

    printf( "   Validating output data\n" );
    printf( "   against input data\n" );
    for (i = 0; i < m; i++) {
        if (!ALMOST_EQUAL(x_v_original[i], x_v_calculated[i])) {
            printf( " Error in Task 2: output data is not equal to" );
            printf( " input data \n" );
            exit_status = 1;
            goto exit;
        }
    };
    printf( "   Done \n" );
    printf("-----------------------------------------------\n");

//*******************************************************************************
//    Task 3.  Obtain Diagonal matrix-vector multiply D *x_v_original --> y_v
//    and solve triangular system   D *x_v_calculated = y_v with single right hand side
//    Array x_v_calculated must be equal to the array x_v_original
//*******************************************************************************
    printf("                                    \n");
    printf("   TASK 3:                          \n");
    printf("   INPUT DATA FOR mkl_sparse_d_mv   \n");
    printf("   WITH DIAGONAL MATRIX             \n");
    printf("     m = " INT_PRINT_FORMAT "       \n", m);
    printf("     SPARSE_OPERATION_NON_TRANSPOSE \n" );
    printf("   Input vector                     \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", x_v_original[i]);
    };

    descrA.type = SPARSE_MATRIX_TYPE_DIAGONAL;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_d_mv( SPARSE_OPERATION_NON_TRANSPOSE, alpha, cscA, descrA, x_v_original, beta, y_v);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 3 mkl_sparse_d_mv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("                                   \n");
    printf("   OUTPUT DATA FOR mkl_sparse_d_mv \n");
    printf("   WITH DIAGONAL MATRIX            \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", y_v[i]);
    };
    printf("-----------------------------------------------\n");
    printf("   Multiply by inverse      \n");
    printf("   matrix with the help     \n");
    printf("   of MKL_SPARSE_D_TRSV            \n");

    status =  mkl_sparse_d_trsv( SPARSE_OPERATION_NON_TRANSPOSE, alpha, cscA, descrA, y_v, x_v_calculated);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 3  mkl_sparse_d_trsv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("                                     \n");
    printf("   OUTPUT DATA FOR mkl_sparse_d_trsv \n");
    printf("   WITH DIAGONAL MATRIX              \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", x_v_calculated[i]);
    };

    printf( "   Validating output data\n" );
    printf( "   against input data\n" );
    for (i = 0; i < m; i++) {
        if (!ALMOST_EQUAL(x_v_original[i], x_v_calculated[i])) {
            printf( " Error in Task 3: output data is not equal to" );
            printf( " input data \n" );
            exit_status = 1;
            goto exit;
        }
    };

    printf( "   Validating output data\n" );
    printf( "   against input data\n" );
    for (i = 0; i < m; i++) {
        if (!ALMOST_EQUAL(x_v_original[i], x_v_calculated[i])) {
            printf( " Error in Task 3: output data is not equal to" );
            printf( " input data \n" );
            exit_status = 1;
            goto exit;
        }
    };
    printf( "   Done \n" );
    printf("-----------------------------------------------\n");

//*******************************************************************************
//    Task 4.  Obtain Symmetric matrix-vector multiply (L+D+L')*x_v_original --> y_v
//    with the help of MKL_SPARSE_D_MV
//
//*******************************************************************************
    printf("                                    \n");
    printf("   TASK 4:                          \n");
    printf("   INPUT DATA FOR mkl_sparse_d_mv   \n");
    printf("   WITH SYMMETRIC LOWER MATRIX      \n");
    printf("     ALPHA = %4.1f  BETA = %4.1f    \n", alpha, beta);
    printf("     SPARSE_OPERATION_NON_TRANSPOSE \n" );
    printf("   Input vector                     \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", x_v_original[i]);
    };

    descrA.type = SPARSE_MATRIX_TYPE_SYMMETRIC;
    descrA.mode = SPARSE_FILL_MODE_LOWER;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_d_mv( SPARSE_OPERATION_NON_TRANSPOSE, alpha, cscA, descrA, x_v_original, beta, y_v);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 4 mkl_sparse_d_mv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("                                   \n");
    printf("   OUTPUT DATA FOR mkl_sparse_d_mv \n");
    printf("   WITH SYMMETRIC LOWER MATRIX     \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", y_v[i]);
    };
    printf("-----------------------------------------------\n");

//*******************************************************************************
//    Task 5. Obtain General matrix-vector multiply A'*x_v_original --> y_v with
//    the help of MKL_SPARSE_D_MV
//
//*******************************************************************************
    printf("                                 \n");
    printf("   TASK 5:                       \n");
    printf("   INPUT DATA FOR mkl_sparse_d_mv\n");
    printf("   WITH GENERAL MATRIX           \n");
    printf("     ALPHA = %4.1f  BETA = %4.1f \n", alpha, beta);
    printf("     SPARSE_OPERATION_TRANSPOSE  \n" );
    printf("   Input vector                  \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", x_v_original[i]);
    };

    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_d_mv( SPARSE_OPERATION_TRANSPOSE, alpha, cscA, descrA, x_v_original, beta, y_v);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 5 mkl_sparse_d_mv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("                                   \n");
    printf("   OUTPUT DATA FOR mkl_sparse_d_mv \n");
    printf("   WITH GENERAL MATRIX             \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", y_v[i]);
    };
    printf("-----------------------------------------------\n");


exit:
    // Release matrix handle and deallocate matrix
    mkl_sparse_destroy ( cscA );

    return exit_status;
}

