#!/usr/bin/env python
#
# Copyright (C) 2016 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#

import sys
import os
import subprocess
import shutil
import time
import getpass

from tempfile import gettempdir

# Common messages
COPYRIGHTS = "Copyright (C) 2009-2021 Intel Corporation. All rights reserved."
NECESSARY_FILES_NOT_FOUND = "The necessary files could not be found."
TEMP_FOLDER_NOT_AVAILABLE = "Temporary directory is not available. Use the option '--log-dir' to specify a custom folder for logs."
FOLDER_NOT_EMPTY = "Custom folder for log is not empty."

class FileNotExistError(Exception):
    def __init__(self, path):
        self.path = path       

def check_file_exist(path):
    if not os.path.isfile(path):
        raise FileNotExistError(path)


class Log(object):
    def __init__(self, log_path):
        self.log_location = log_path
        self.log_file = open(log_path, 'w+')
        self.log_to_stdout = False
        # Forward log to stdout too in oneCI test
        # because too hard to get logs from machine
        env_value = os.environ.get('SELF_CHECK_LOG_TO_STDOUT')
        if env_value:
            self.log_to_stdout = True

    def __del__(self):
        self.to_stdout("\nLog location: " + str(self.log_location))
        self.log_file.close()       

    def to_log(self, message):
        self.log_file.write(message + '\n')
        self.log_file.flush()
        if self.log_to_stdout is True:
            sys.stdout.write(message + '\n')
            sys.stdout.flush()

    def to_stdout(self, message, temp=False):
        if temp:
            sys.stdout.write(message + '\r')
        else:
            sys.stdout.write(message + '\n')
        sys.stdout.flush()
        if self.log_to_stdout is False:
            self.to_log(message.lstrip(' '))

    def phase_error(self, error):
        title = error.title
        description = error.description
        self.to_stdout('\n' + str(title))
        if description:
            self.to_log(str(description))


class State(object):
    def __init__(self, bin_dir, work_dir, prod_abbr, int_pref, cl_tool, copyrights_tool):
        self.status = 'OK'
        self.bin_dir = bin_dir
        self.prod_abbr = prod_abbr
        self.int_pref = int_pref
        self.copyrights_tool = copyrights_tool

        if work_dir:
            self.work_dir = work_dir
        else:
            work_dir = self.get_temp_work_dir()
            if work_dir is None:
                sys.stdout.write("%s\n" % TEMP_FOLDER_NOT_AVAILABLE)
                self.status = 'FAIL'
            else:
                self.work_dir = work_dir

        if not self._check_dir_empty():
            sys.stdout.write("\n%s\n\n" % FOLDER_NOT_EMPTY)
            self.status = 'FAIL'

        self.product_dir = os.path.dirname(self.bin_dir)

        self.log_location = os.path.abspath(os.path.join(self.work_dir, 'log.txt'))
        self.log = Log(self.log_location)
        self.write_header()

        self.cl_path = self.get_runtool_path(cl_tool)
        self.runss_path = self.get_runtool_path(self.int_pref + 'runss')

        try:
            self._check_necessary_files_exist()
        except FileNotExistError as e:
            self.log.to_stdout(NECESSARY_FILES_NOT_FOUND)
            self.log.to_log("Cannot find file by path: %s" % e.path)
            self.status = 'FAIL'

        self.output_indent = '    '
        self.stdout_indent = '  '
        self.has_warnings = False

    def get_temp_work_dir(self):
        temp_dir = gettempdir()
        if temp_dir:
            timestamp = time.strftime("%Y.%m.%d_%H.%M.%S")
            user_name = getpass.getuser()
            parent_dir = os.path.join(temp_dir, self.prod_abbr+ '-tmp-' + user_name)
            if not os.path.exists(parent_dir):
                    os.makedirs(parent_dir)
            work_dir = os.path.abspath(os.path.join(parent_dir, 'self-checker-' + timestamp))
            if not os.path.exists(work_dir):
                    os.makedirs(work_dir)
            else:
                shutil.rmtree(work_dir, ignore_errors=True)
                os.makedirs(work_dir)
            return work_dir
        else:
            return None

    def get_runtool_path(self, runtool_type):
        runtool_name = runtool_type
        if 'win32' == sys.platform:
            runtool_name += '.exe'
        runtool_path = os.path.join(self.bin_dir, runtool_name)
        return runtool_path

    def get_support_path(self):
        support_name = 'support.txt'
        install_dir = os.path.dirname(self.bin_dir)
        support_path = os.path.join(install_dir, support_name)
        check_file_exist(support_path)
        return support_path

    def get_build_number(self):
        try:
            support_path = self.get_support_path()
        except FileNotExistError as e:
            self.log.to_log("Cannot find 'support.txt' by path: %s" % e.path)
        if support_path:
            support_file = open(support_path, 'r')
            try:
                data = support_file.readlines()
            finally:
                support_file.close()
            for line in data:
                if line.startswith('Build Number:'):
                    return line
        return None

    def write_header(self):
        copyrights_tool = "%s\n" % self.copyrights_tool
        copyrights = "%s\n" % COPYRIGHTS
        build_number = self.get_build_number()
        if build_number:
            self.log.to_stdout(copyrights_tool + copyrights + build_number.strip() + '\n')
        else:
            self.log.to_stdout(copyrights_tool + copyrights)

    def append_to_work_dir(self, name):
        appended_path = os.path.join(self.work_dir, name)
        return appended_path

    def remove_dir(self, dir):
        shutil.rmtree(dir, ignore_errors=True)

    def _check_necessary_files_exist(self):
        check_file_exist(self.cl_path)
        check_file_exist(self.runss_path)

    def _check_dir_empty(self):
        if not os.listdir(self.work_dir):
            return True
        return False


class TestCase(object):
    def __init__(self, state):
        self.state = state
        self.log = state.log
        self.status = 'OK'

    def run(self):
        raise NotImplementedError('run() method is not implemented in test case!')

    def subprocess_wrapper(self, cl_args, indent='', grep=False, grep_args=''):
        self.log.to_log("Command line:\n%s" % ' '.join(cl_args))
        if grep:
            # Using of shell=True is dangerous
            # See: https://docs.python.org/3/library/subprocess.html#replacing-shell-pipeline
            serv_process = subprocess.Popen(cl_args, stdout=subprocess.PIPE)
            process = subprocess.Popen(["grep", grep_args], stdin=serv_process.stdout, stdout=subprocess.PIPE, encoding="utf-8")
            serv_process.stdout.close()
            stdout, stderr = process.communicate()
        else:
            process = subprocess.Popen(cl_args, stdout=subprocess.PIPE, stderr=subprocess.PIPE, encoding="utf-8")
            stdout, stderr = process.communicate()
        if stdout:
            stdout = stdout.splitlines()
            self.log.to_log("Stdout:")
            for line in stdout:
                self.log.to_log(indent + line.strip())
        if stderr:
            stderr = stderr.splitlines()
            self.log.to_log("Stderr:")
            for line in stderr:
                self.log.to_log(indent + line.strip())

        return process.returncode, stdout, stderr

    def set_has_warnings(self):
        if not self.state.has_warnings:
            self.state.has_warnings = True


class ContextValuesTest(TestCase):
    def __init__(self, state):
        TestCase.__init__(self, state)
        self.context_values = dict()
        self.sep_permission_warn = ''

    def run(self):
        self.log.to_log("=" * 80)
        self.log.to_log("Context values:")
        title, status = self.get_context_values()
        self.log.to_log(title + ': ' + status)

    def get_context_values(self):
        title = 'Getting context values'
        args = [self.state.runss_path, "--context-value-list"]
        ret_code, stdout, stderr = self.subprocess_wrapper(args, indent=self.state.stdout_indent)
        if ret_code != 0:
            self.status = 'FAIL'
            status = 'FAIL'
        else:
            status = 'OK'
            self.parse_context_values(stdout)
        return title, status

    def parse_context_values(self, stdout):
        for line in stdout:
            key = line.split(':')[0].strip()
            value = ':'.join(line.split(':')[1:]).strip()
            self.context_values[key] = value

        return False
