#
# Copyright (C) 2017 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#

from __future__ import print_function

import os
import sys

# check table
IS_TABLE_EMPTY = 'select * from csCachesimModel'

#cache model attributes per site:
ATTRIBUTES_QUERY = """select a.attribute_name, a.attribute_value
from csCachesimModel m
inner join csCachesimModelAttributes a on a.model_id = m.id
where m.loop_id = ?
order by a.attribute_name
"""

#per site utilization:
UTILIZATION_QUERY = """select 64 * sum(T.total_ev) as 'optimal usage', sum(T.bytes) as 'actual usage',
cast(sum(T.bytes) as real) / (64 * sum(T.total_ev)) * 100 as 'utilization'
from
    (select u.size, sum(u.evictions) 'total_ev', sum(u.size * u.evictions) 'bytes'
    from csCachesimModel m
    inner join csCachesimUtilization u on u.model_id = m.id
    where m.loop_id = ? and u.object_id = -1
    group by u.size) as 'T'
"""

#evicted cache lines utilization per site:
EVICTED_UTILIZATION_QUERY = """select u.size, sum(u.evictions)
from csCachesimModel m
inner join csCachesimUtilization u on u.model_id = m.id
where m.loop_id = ? and u.object_id = -1
group by u.size
order by u.size
"""

class CacheSimulatorInfo(object):
    def __init__(self, dbfile, site_id):
        import sqlite3
        self.empty = True
        if not os.path.exists(dbfile):
            print('Can\'t find db file: ' + str(dbfile))
            return

        conn = sqlite3.connect(dbfile)
        c = conn.cursor()

        c.execute(IS_TABLE_EMPTY)
        if not c.fetchall():
            #print('No cachesim information')
            return

        c.execute(ATTRIBUTES_QUERY, (site_id,))
        output = c.fetchall()
        self.attrs = {k: v for k, v in output}

        c.execute(UTILIZATION_QUERY, (site_id,))
        output = c.fetchone()
        self.utilization = 0
        if output and output[-1]:
            self.utilization = output[-1]

        c.execute(EVICTED_UTILIZATION_QUERY, (site_id,))
        output = c.fetchall()
        self.evicted = [['Bytes used', 'Evicted lines']] + [[str(v) for v in row] for row in output]
        conn.close()
        self.empty = False

    def __nonzero__(self):
        return not self.empty

    __bool__ = __nonzero__
