#
# Copyright (C) 2018 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#


# ------------------------------------------------------------------------------
# This example shows how to obtain roofs calculated for a certain number of
# threads. For example, the expected output might look like:
#
# - L1 Bandwidth 348 GB/s
# - L2 Bandwidth 193 GB/s
# - L3 Bandwidth 66 GB/s
# - DRAM Bandwidth 16 GB/s
# - Scalar Add Peak 7 GFLOPS
# - SP Vector Add Peak 58 GFLOPS
# - DP Vector Add Peak 27 GFLOPS
# ------------------------------------------------------------------------------

import math
import sys

try:

    import advisor

except ImportError:

    print(
        """Import error: Python could not resolve path to Advisor's pythonapi directory.
        To fix, either manually add path to the pythonapi directory into PYTHONPATH environment
        variable, or use advixe-vars.* scripts to set up product environment variables automatically."""
    )
    sys.exit(1)

# Check command-line arguments.
if len(sys.argv) < 2:
    print('Usage: "python {} path_to_project_dir"'.format(__file__))
    sys.exit(2)

# Open the Advisor Project and load the data.
project = advisor.open_project(sys.argv[1])
data = project.load(advisor.SURVEY)

# The data.get_roofs(number_of_threads, advisor.RoofsStrategy.MULTI_THREAD | advisor.RoofsStrategy.SINGLE_THREAD)
# returns the roofs calculated for a certain number of threads using a specific roof calculation strategy.
#
# The data.get_roofs() returns single-threaded and multi-threaded roofs. It is equvalient to data.roofs.
roofs = data.get_roofs(4, advisor.RoofsStrategy.MULTI_THREAD)

for roof in roofs:

    # Process memory roofs.
    if roof.has_type(advisor.RoofType.MEMORY):

        # Convert to GByte/s.
        bandwidth = roof.bandwidth // math.pow(10, 9)
        print("{} {:.0f} GB/s".format(roof.name, bandwidth))

    # Process compute roofs.
    else:

        # Convert to GFLOPS.
        bandwidth = roof.bandwidth // math.pow(10, 9)
        print("{} {:.0f} GFLOPS".format(roof.name, bandwidth))
