!=======================================================================
!
!  SAMPLE SOURCE CODE - SUBJECT TO THE TERMS OF END-USER LICENSE AGREEMENT FOR
!  INTEL(R) ADVISOR XE 2013.
!
!  Copyright (C) 2008-2011 Intel Corporation. All rights reserved
!
!  THIS FILE IS PROVIDED "AS IS" WITH NO WARRANTIES, EXPRESS OR IMPLIED, INCLUDING BUT
!  NOT LIMITED TO ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
!  PURPOSE, NON-INFRINGEMENT OF INTELLECTUAL PROPERTY RIGHTS.
!
! ========================================================================

program NQueens

! Solve the nqueens problem - base serial version
!
! Original C++ code by Ralf Ratering & Mario Deilmann
! Fortran version by Steve Lionel
!
! Read http://en.wikipedia.org/wiki/Nqueens for background
!

use advisor_annotate

implicit none

integer :: nrOfSolutions = 0 
integer :: size = 0

character(2) :: cmdarg
integer :: stat
integer :: time_start, time_end, count_rate
integer, allocatable :: queens(:)

integer, pointer, dimension(:)  :: correct_solution

allocate(correct_solution(16))
correct_solution = (/ 1,0,0,1,2,10,4,40,92,352,724,2680,14200,73712,365596,2279194 /)

! Get command line argument

if (command_argument_count() < 1) then
  print *, "Usage: 2_nqueens_annotated[_debug] boardSize"
  size = 14
  print *, "Using default size of 14"
else
  call get_command_argument (1, cmdarg, status=stat)
  if (stat /= 0) then
    print *, "Invalid boardSize"
    stop
  end if
  read (cmdarg,*,iostat=stat) size
  if ((stat /= 0) .or. (size < 1)) then
    print *, "Error: boardSize must be between 1 and 15"
    stop
  end if
  if (size > 15) then
    print *, "Error: boardSize must be between 1 and 15"
    stop
  end if
endif

! Begin
print 101, "Starting serial recursive solver for size ", size
101 format (A,I0,A)
call system_clock (time_start)
allocate (queens(size))
call solve (queens)
call system_clock (time_end, count_rate)
print 101, "Number of solutions: ", nrOfSolutions
print 101, "Calculations took ", (time_end-time_start) / (count_rate/1000), "ms."
if (nrOfSolutions == correct_solution(size+1)) then
  print 101, "Correct Result!"
else
  print 101, "Incorrect Result!"
end if

deallocate (queens)
deallocate (correct_solution)

contains

! Routine to print the board

subroutine print (queens)
  implicit none
  integer, intent(in) :: queens(:)
  integer :: row, col

  do row=1,size
    do col=1,size
      if (queens(row) == col) then
        write (*,'(A)',advance='no') "Q"
      else
        write (*,'(A)',advance='no') "-"
      end if
    end do
  write (*,'(A)')
  end do
write (*,*)
end subroutine print

! Recursive routine to set a queen on the board
!DEC$ ATTRIBUTES NOINLINE :: setQueen

recursive subroutine setQueen (queens, row, col)
  implicit none
  integer, intent(inout) :: queens(:)
  integer, intent(in) :: row, col
  integer :: i
  integer, volatile :: j 
  !In order to avoid a data race on the "queens" array create a local copy.
  !Uncomment the statements using the "lcl_queens" array and comment the 
  !statememts using the "queens" array.
  !integer :: lcl_queens(ubound(queens,dim=1))

  ! Make copy of queens array
  !lcl_queens = queens
  
  do i=1,row-1
    ! vertical attacks
    !if (lcl_queens(i) == col) return
    if (queens(i) == col) return
    ! diagonal attacks
    !if (abs(lcl_queens(i)-col) == (row-i)) return
    if (abs(queens(i)-col) == (row-i)) return
    end do
    
  ! column is ok, set the queen
  !lcl_queens(row) = col
  queens(row) = col
  
  if (row == size) then
    !call annotate_lock_acquire(0)
    nrOfSolutions = nrOfSolutions + 1
    !call annotate_lock_release(0)
  else
    ! try to fill next row
    do j=1,size
      !call setQueen (lcl_queens, row+1, j)
      call setQueen (queens, row+1, j)
    end do
  end if
end subroutine SetQueen

! Main solver routine
!DEC$ ATTRIBUTES NOINLINE :: solve
subroutine solve (queens)
  implicit none
  integer, intent(inout) :: queens(:)
  integer i

  call annotate_site_begin("solve")
  do i=1,size
    call annotate_iteration_task("setQueen")
    ! try all positions in first row
    call SetQueen (queens, 1, i)
  end do
  call annotate_site_end()

end subroutine solve

end program nQueens
