/*******************************************************************************
* Copyright 2024 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*
*  Content:
*       This example demonstrates use of oneAPI Math Kernel Library (oneMKL)
*       DPCPP USM API oneapi::mkl::sparse::matmatd to perform general
*       sparse matrix-sparse matrix multiplication with a dense result matrix
*       on a SYCL device (GPU). This example uses sparse matrices in CSR format.
*
*           C = alpha * op(A) * op(B) + beta * C
*
*       where op() is defined by one of
*           oneapi::mkl::transpose::{nontrans,trans,conjtrans}
*
*       The supported floating point data types for matmatd matrix data are:
*           float
*           double
*           std::complex<float>
*           std::complex<double>
*
*       The supported matrix formats for matmatd are:
*           CSR
*
*******************************************************************************/

// stl includes
#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iterator>
#include <limits>
#include <list>
#include <vector>

#include "mkl.h"
#include "oneapi/mkl.hpp"
#include <sycl/sycl.hpp>

// local includes
#include "common_for_examples.hpp"
#include "./include/common_for_sparse_examples.hpp"

//
// Main example for Sparse Matrix-Sparse Matrix Multiply with a dense result,
// consisting of
// initialization of A and B matrices through process of creating C matrix as
// the product
//
// C = alpha op(A) * op(B) + beta C
//
// In this case, the square symmetric matrices A and B are generated separately
// occupying different memory, but with the same structure and values, so that
// in the end C is the square of A.
//
template <typename fpType, typename intType>
int run_sparse_matrix_sparse_matrix_dense_result_multiply_example(const sycl::device &dev)
{
    // Initialize data for Sparse Matrix - Sparse Matrix Multiply
    oneapi::mkl::transpose opA = oneapi::mkl::transpose::nontrans;
    oneapi::mkl::transpose opB = oneapi::mkl::transpose::nontrans;

    oneapi::mkl::index_base a_index = oneapi::mkl::index_base::zero;
    oneapi::mkl::index_base b_index = oneapi::mkl::index_base::one;

    // Catch asynchronous exceptions
    auto exception_handler = [](sycl::exception_list exceptions) {
        for (std::exception_ptr const &e : exceptions) {
            try {
                std::rethrow_exception(e);
            }
            catch (sycl::exception const &e) {
                std::cout << "Caught asynchronous SYCL "
                             "exception during sparse::matmatd example:\n"
                          << e.what() << std::endl;
            }
        }
    };

    // create execution main_queue
    sycl::queue main_queue(dev, exception_handler);

    //
    // set up dimensions of matrix products
    //
    intType size = 4;

    intType a_nrows = size * size * size;
    intType a_ncols = a_nrows;
    intType a_nnz   = 27 * a_nrows;
    intType b_nrows = size * size * size;
    intType b_ncols = b_nrows;
    intType b_nnz   = 27 * b_nrows;
    intType c_nrows = size * size * size;
    intType c_ncols = c_nrows;

    //
    // setup A data locally in CSR format
    //
    std::vector<intType, mkl_allocator<intType, 64>> ia;
    std::vector<intType, mkl_allocator<intType, 64>> ja;
    std::vector<fpType, mkl_allocator<fpType, 64>> a;

    ia.resize(a_nrows + 1);
    ja.resize(a_nnz);
    a.resize(a_nnz);

    intType a_ind = a_index == oneapi::mkl::index_base::zero ? 0 : 1;
    generate_sparse_matrix<fpType, intType>(size, ia, ja, a, a_ind);
    a_nnz = ia[a_nrows] - a_ind;

    //
    // setup B data locally in CSR format
    //
    std::vector<intType, mkl_allocator<intType, 64>> ib;
    std::vector<intType, mkl_allocator<intType, 64>> jb;
    std::vector<fpType, mkl_allocator<fpType, 64>> b;

    ib.resize(b_nrows + 1);
    jb.resize(b_nnz);
    b.resize(b_nnz);

    intType b_ind = b_index == oneapi::mkl::index_base::zero ? 0 : 1;
    generate_sparse_matrix<fpType, intType>(size, ib, jb, b, b_ind);
    b_nnz = ib[b_nrows] - b_ind;

    //
    // setup C data locally in dense matrix format
    //
    fpType alpha = fpType(1);
    fpType beta = fpType(0);
    auto c_layout = oneapi::mkl::layout::row_major;
    intType ldc = (c_layout == oneapi::mkl::layout::row_major) ? c_ncols : c_nrows;
    intType c_size = (c_layout == oneapi::mkl::layout::row_major) ? ldc * c_nrows : ldc * c_ncols;
    std::vector<fpType, mkl_allocator<fpType, 64>> c;
    c.resize(c_size);

    //
    // Execute Matrix Multiply
    //

    std::cout << "\n\t\tsparse::matmatd parameters:\n";
    std::cout << "\t\t\topA = " << opA << std::endl;
    std::cout << "\t\t\topB = " << opB << std::endl;

    std::cout << "\t\t\tA_nrows = A_ncols = " << a_nrows << std::endl;
    std::cout << "\t\t\tB_nrows = B_ncols = " << b_nrows << std::endl;
    std::cout << "\t\t\tC_nrows = C_ncols = " << c_nrows << std::endl;

    std::cout << "\t\t\tA_index = " << a_index << std::endl;
    std::cout << "\t\t\tB_index = " << b_index << std::endl;

    oneapi::mkl::sparse::matrix_handle_t A = nullptr;
    oneapi::mkl::sparse::matrix_handle_t B = nullptr;

    sycl::buffer<intType, 1> a_rowptr(ia.data(), a_nrows + 1);
    sycl::buffer<intType, 1> a_colind(ja.data(), a_nnz);
    sycl::buffer<fpType, 1> a_values(a.data(), a_nnz);
    sycl::buffer<intType, 1> b_rowptr(ib.data(), b_nrows + 1);
    sycl::buffer<intType, 1> b_colind(jb.data(), b_nnz);
    sycl::buffer<fpType, 1> b_values(b.data(), b_nnz);
    sycl::buffer<fpType, 1> c_values(c.data(), c_size);

    try {
        oneapi::mkl::sparse::init_matrix_handle(&A);
        oneapi::mkl::sparse::init_matrix_handle(&B);

        oneapi::mkl::sparse::set_csr_data(main_queue, A, a_nrows, a_ncols, a_index,
                a_rowptr, a_colind, a_values);
        oneapi::mkl::sparse::set_csr_data(main_queue, B, b_nrows, b_ncols, b_index,
                b_rowptr, b_colind, b_values);
        main_queue.submit([&](sycl::handler& cgh) {
            auto c_acc = c_values.template get_access<sycl::access_mode::read_write>(cgh);
            cgh.fill(c_acc, fpType(0));
        });

        //
        // compute sparse matrix-matrix product with dense result
        //
        oneapi::mkl::sparse::matmatd(
            main_queue, c_layout, opA, opB, alpha, A, B, beta, c_values, c_nrows, c_ncols, ldc);

        // print out a portion of C solution
        const intType c_nrows_copy = std::min<intType>(2, c_nrows); // only copy over this many rows of C to host
        {
            std::cout << "C matrix [first " << c_nrows_copy << " rows]:" << std::endl;
            auto c_acc = c_values.get_host_access(sycl::read_only);
            for (intType row = 0; row < c_nrows_copy; ++row) {
                for (intType col = 0; col < c_ncols; ++col) {
                    fpType val = (c_layout == oneapi::mkl::layout::row_major) ?
                        c_acc[ldc*row + col] : c_acc[ldc*col + row];
                    std::cout << "C(" << row << ", " << col << ") = " << val
                              << std::endl;
                }
            }
        }

        // clean up
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A);
        auto ev_relB = oneapi::mkl::sparse::release_matrix_handle(main_queue, &B);


    }
    catch (sycl::exception const &e) {
        std::cout << "\t\tCaught synchronous SYCL exception:\n" << e.what() << std::endl;

        main_queue.wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A).wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &B).wait();

        return 1;
    }
    catch (oneapi::mkl::unimplemented const &e) {
        std::cout << "\t\tCaught oneMKL unimplemented exception:\n" << e.what() << std::endl;

        main_queue.wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A).wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &B).wait();

        return 0;
    }
    catch (std::exception const &e) {
        std::cout << "\t\tCaught std exception:\n" << e.what() << std::endl;

        main_queue.wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A).wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &B).wait();

        return 1;
    }

    main_queue.wait();
    return 0;
}

//
// Description of example setup, apis used and supported floating point type
// precisions
//
void print_example_banner()
{

    std::cout << "" << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << "# Sparse Matrix-Sparse Matrix Multiply with dense result Example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "#    C = alpha * op(A) * op(B) + beta * C" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# where A and B are sparse matrices in CSR format, and C is the\n"
                 "# dense matrix product"
              << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using apis:" << std::endl;
    std::cout << "#   sparse::matmatd" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "#   sparse::init_matrix_handle" << std::endl;
    std::cout << "#   sparse::set_csr_data" << std::endl;
    std::cout << "#   sparse::release_matrix_handle" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported floating point type precisions:" << std::endl;
    std::cout << "#   float" << std::endl;
    std::cout << "#   double" << std::endl;
    std::cout << "#   std::complex<float>" << std::endl;
    std::cout << "#   std::complex<double>" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << std::endl;
}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU implementation
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU implementation
// -DSYCL_DEVICES_all (default) -- runs on all: cpu and gpu devices
//
//  For each device selected and each supported data type, MatrixMultiplyExample
//  runs is with all supported data types
//

int main(int argc, char **argv)
{

    print_example_banner();

    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    int status = 0;
    for (auto it = list_of_devices.begin(); it != list_of_devices.end(); ++it) {

        sycl::device my_dev;
        bool my_dev_is_found = false;
        get_sycl_device(my_dev, my_dev_is_found, *it);

        if (my_dev_is_found) {
            std::cout << "Running tests on " << sycl_device_names[*it] << ".\n";

            std::cout << "\tRunning with single precision real data type:" << std::endl;
            status = run_sparse_matrix_sparse_matrix_dense_result_multiply_example<float, std::int32_t>(my_dev);
            if (status != 0)
                return status;

            if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                std::cout << "\tRunning with double precision real data type:" << std::endl;
                status = run_sparse_matrix_sparse_matrix_dense_result_multiply_example<double, std::int32_t>(
                        my_dev);
                if (status != 0)
                    return status;
            }

            std::cout << "\tRunning with single precision complex data type:" << std::endl;
            status = run_sparse_matrix_sparse_matrix_dense_result_multiply_example<std::complex<float>, std::int32_t>(my_dev);
            if (status != 0)
                return status;

            if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                std::cout << "\tRunning with double precision complex data type:" << std::endl;
                status = run_sparse_matrix_sparse_matrix_dense_result_multiply_example<std::complex<double>, std::int32_t>(
                        my_dev);
                if (status != 0)
                    return status;
            }
        }
        else {
#ifdef FAIL_ON_MISSING_DEVICES
            std::cout << "No " << sycl_device_names[*it]
                      << " devices found; Fail on missing devices "
                         "is enabled.\n";
            return 1;
#else
            std::cout << "No " << sycl_device_names[*it] << " devices found; skipping "
                      << sycl_device_names[*it] << " tests.\n";
#endif
        }
    }

    mkl_free_buffers();
    return status;
}
