#
# Copyright (C) 2020 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#

from . import ResultType

def _get_result_path(data_model,result_type):
    return data_model._get_rt_result_path(result_type)

def _get_cfg_info(self, result_type: ResultType = ResultType.Survey,
        file_name='context_values.cfg'):
    import sys
    import os
    sys.path.append(os.path.dirname(os.path.abspath(__file__)))

    from defusedxml import ElementTree as ET

    folder_path = _get_result_path(self, result_type)

    def _get_contextvalue_cfg(file_path):
        '''
            get configs by the format <contextvalue id=elem_id value=elem_value>
            the value key may have namespace prefix, like `unsignedShort:value`
            so find value key with endswith()
        '''
        _config_root = ET.parse(file_path).getroot()
        configs = {}
        for node in _config_root:
            attrib = node.attrib
            if attrib:
                for k in attrib.keys():
                    if k.endswith('value'):
                        attrib['value'] = attrib[k]
                        break
                configs[attrib['id']] = attrib['value']
        return configs

    def _get_collect_cfg(file_path):
        '''
            collect_cfg is not a valid xml file, so it needs to be preprocess
            drop lines to fit _get_contextvalue_cfg()
        '''
        from io import StringIO
        import re
        drop_lines_format = (r'<.*pointer.*>', r'<.*workload.*>', r'<.*context>')

        def _is_drop(line):
            for d in drop_lines_format:
                if re.search(d, line) != None:
                    return True
            return False

        tmp_cfg = str()
        with open(file_path) as ori_cfg:
            for line in ori_cfg.readlines():
                if not _is_drop(line):
                    tmp_cfg += line
        with StringIO(tmp_cfg) as tmp_file:
            configs = _get_contextvalue_cfg(tmp_file)
        return configs

    def _get_tag_text_cfg(file_path):
        '''
            get configs by the format <elem_tag>elem_text</elem_tag>
            find all leaf node recursively
        '''
        _config_root = ET.parse(file_path).getroot()

        def get_child_info(root):
            configs = {}
            for node in root:
                childs = list(node)
                if not childs:
                    if node.text:
                        configs[node.tag] = node.text
                else:
                    configs.update(get_child_info(node))
            return configs
        return get_child_info(_config_root)

    file_path = os.path.join(folder_path, 'config', file_name)
    if file_name == 'context_values.cfg':
        return _get_contextvalue_cfg(file_path)
    elif file_name == 'collection.cfg':
        return _get_collect_cfg(file_path)
    else:
        return _get_tag_text_cfg(file_path)

def _get_runtool_cfg(self, result_type: ResultType):
    import os
    result_2_cfg_files = {ResultType.Survey: 'runss.options',
        ResultType.MAP: 'runtc.options',
        ResultType.TripCounts: 'runtrc.options',
        ResultType.Correctness: 'runtc.options',
        ResultType.PerfProjection: 'runpp.options',
        ResultType.Suitability: 'runss.options',
        ResultType.DCFG: 'rundcfg.options'}

    folder_path = _get_result_path(self, result_type)
    if not folder_path:
        return {}
    result = result_2_cfg_files.get(result_type)
    if result == None:
        return {}

    file_path = os.path.join(folder_path, 'config', result)

    runtool_options_dict = {}
    with open(file_path) as file:
        option = ''
        for line in file.readlines():
            line = line.rstrip()
            if line.startswith('-') and option != 'executable': # Checking that the line contains an option
                if '=' in line:
                    sublines = line.split('=')
                    runtool_options_dict[sublines[0].lstrip('-')] = sublines[1]
                else:
                    option = line.lstrip('-') if line != '--' else 'executable'
                    is_inverted_op = option.startswith('no-')
                    if is_inverted_op:
                        option = option.replace('no-', '')
                    runtool_options_dict[option] = not is_inverted_op
            else: # The line contains only a value
                if isinstance(runtool_options_dict[option], str):
                    runtool_options_dict[option] += ' ' + line
                else:
                    runtool_options_dict[option] = line

    return runtool_options_dict
