#
# Copyright (C) 2017 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#


# ------------------------------------------------------------------------------
# This example shows how to obtain the memory access count metric for loops.
#
# This example shows how to obtain a custom instruction mix breakdown,requested
# by mix targets in order to interest. The following are valid targets:
#
# - advisor.MixTarget.CATEGORY,
# - advisor.MixTarget.TYPE
# - advisor.MixTarget.ISA
# - advisor.MixTarget.OPERAND
# - advisor.MixTarget.DATATYPE
#
# In the example below we will have the instsruction mix categorized by CATEGORY
# first (compute/memory), then both divided into two buckets by TYPE
# (vectorized/scalar). In addition, we can make breakdown by ISA (SSE, AVX etc.)
# and OPERAND (memory/vector registers).
#
# It is possible to perform analysis in any order of mix targets. The output is
# serialized xml container with tree of metrics.
# ------------------------------------------------------------------------------

import itertools
import sys

try:

    import advisor

except ImportError:

    print(
        """Import error: Python could not resolve path to Advisor's pythonapi directory.
        To fix, either manually add path to the pythonapi directory into PYTHONPATH environment
        variable, or use advixe-vars.* scripts to set up product environment variables automatically."""
    )
    sys.exit(1)

# Check command-line arguments.
if len(sys.argv) < 2:
    print('Usage: "python {} path_to_project_dir"'.format(__file__))
    sys.exit(2)

# Open the Advisor Project and load the data.
project = advisor.open_project(sys.argv[1])
data = project.load(advisor.SURVEY)

for entry in data.bottomup:

    print(entry["function_call_sites_and_loops"])

    try:

        print(entry.get_custom_instruction_mix([advisor.MixTarget.CATEGORY, advisor.MixTarget.TYPE]))

    except ValueError:

        print("Cannot Cannot serialize XML tree to string.")

# In addition, it is possible to make breakdown using instruction types below:
# GETMANT
# SUB
# KTEST
# MIN
# SHIFT
# KAND
# COMPRESS
# CONVERT
# MASKED
# SCALE
# KNOT
# MUL
# DIV
# KOR
# RECCP512F
# SCATTER
# EXPAND
# INSERT
# SHUFFLE
# LDD
# UNPACK
# CONDITIONAL
# MASK
# KSHIFT
# SAD
# BROADCAST
# RECCP
# KMOV
# STORE
# KXNOR
# CALL
# BASICBLOCK
# EXTRACT
# PERMUTE
# GETEXP
# LOAD
# RECCP512ER
# VECTORIZED
# PERMT2
# MEMORY
# SIGN
# PERMI2
# ADD
# RECSQRT
# GATHERPREFETCH
# BLEND
# MOVNT
# KUNPCK
# FMA
# SCATTERPREFETCH
# RECIPROCAL
# MAX
# GATHER
# KADD
# MOV
# SQRT
# ABS
# PREFETCHW
# RECSQRT512F
# KXOR
# AVG
# RECSQRT512ER
# VPCONFLICT
# PACK
# INC
# DEC
# LEA
#
# It is needed to create dictionary with key as a name of category and list of
# types as a value.

dictionary = {
    "sqrt": [advisor.InstructionType.SQRT],
    "inc": [advisor.InstructionType.INC],
    "dec": [advisor.InstructionType.DEC],
    "lea": [advisor.InstructionType.LEA],
    "div": [advisor.InstructionType.DIV],
    "add": [advisor.InstructionType.ADD],
    "mul": [advisor.InstructionType.MUL],
    "movs": [advisor.InstructionType.MOVNT, advisor.InstructionType.MOV],
}

for entry in data.bottomup:

    try:
        print(
            "{}: {}".format(
                entry["function_call_sites_and_loops"],
                entry.get_custom_instruction_categories_mix(
                    dictionary, [advisor.MixTarget.CATEGORY, advisor.MixTarget.TYPE], [advisor.MixTarget.DATATYPE],
                ),
            )
        )

    except ValueError:
        print("Cannot Cannot serialize xml tree to string")
