# Copyright (C) 2023 Intel Corporation
# SPDX-License-Identifier: MIT

import re
from abc import ABC
from typing import Dict, List

from mpp.parsers.system_information_parser import SystemInformationParser, SystemInfoParserAttributes, UnitNames, \
    SectionNames, SymbolTable, SymbolNames

unit_names = UnitNames('OS_Processor', 'Package_Id', '', 'Core_Type', 'Core_Id', 'Thread_Id')

section_names = SectionNames('=====  Processor information  =====', '=====  Processor topology  =====')

symbol_names = SymbolNames('Num sockets', 'Threads (logical cores) per physical core', 'Physical cores per socket')


class PcmSystemInformationParser(SystemInformationParser):
    parser_attributes = SystemInfoParserAttributes("Update every", ' ', unit_names, section_names, symbol_names)

    class _ProcessorMappingState(SystemInformationParser._ProcessorMappingState):

        def __init__(self):
            super().__init__()
            self.MAP_TABLE_SEPARATOR = '=====  Placement on packages  ====='
            self._is_table_start = False

    class _FeatureState(ABC, SystemInformationParser._State):
        VAL_RE = [re.compile(r'^(?P<name>[\s\S]+): (?P<value>[\s\S]+)$')]

    def _set_ref_tsc(self):
        tsc_detector = 'Nominal core frequency'
        if self.current_core_type in self.processor_features.keys() and tsc_detector in self.processor_features[
            self.current_core_type].keys():
            ref_tsc = self.processor_features[self.current_core_type][tsc_detector].split(' ', 1)[0]
            self._ref_tsc = float(ref_tsc)
        else:
            self._ref_tsc = 0


class PcmSymbolTable(SymbolTable):

    def __init__(self,
                 processor_features: Dict,
                 system_features: Dict,
                 uncore_units: Dict,
                 ref_tsc: int,
                 unique_core_types: List[str],
                 symbols: SymbolNames = symbol_names):
        super().__init__(processor_features, system_features, uncore_units, ref_tsc, unique_core_types, symbols)

    def _get_processor_symbols(self, core_type: str):
        processor_symbols = super()._get_processor_symbols(core_type)
        threads_per_socket = processor_symbols['THREADS_PER_CORE'] * processor_symbols['CORES_PER_SOCKET']
        processor_symbols.update({
            'system.sockets[0].cpus.count': threads_per_socket,
            'THREADS_PER_SOCKET': threads_per_socket,
            'HYPERTHREADING_ON': processor_symbols['THREADS_PER_CORE'] > 1,
        })
        cha_count_key = 'Last level cache slices per socket'
        # TODO: https://jira.devtools.intel.com/browse/THORS-341
        # Hopefully this can be replaced by SymbolTable.__get_uncore_symbols once the above is done
        if cha_count_key in self.processor_features[core_type].keys():
            processor_symbols.update({'CHAS_PER_SOCKET': 'system.CHAS_PER_SOCKET'})
            processor_symbols.update({'system.CHAS_PER_SOCKET': self.processor_features[core_type][cha_count_key]})
        return processor_symbols
