/*
 * Copyright (C) 2005 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

/*
 *  CVS_Id="$Id$"
 */

#include "lwpmudrv_defines.h"

#include "lwpmudrv_types.h"
#include "rise_errors.h"
#include "lwpmudrv_ecb.h"
#if defined(DRV_IA32) || defined(DRV_EM64T)
#include "apic.h"
#endif
#include "lwpmudrv.h"
#include "control.h"
#include "utility.h"
#include "cpumon.h"

#ifdef DRV_USE_NMI
#include <sys/bus.h>
#include <sys/pmckern.h>

#include <machine/intr_machdep.h>
#if __FreeBSD_version >= 1100006
#include <x86/apicvar.h>
#else
#include <machine/apicvar.h>
#endif

#include "pmi.h"
#endif

#include <machine/md_var.h>

/*
 * CPU Monitoring Functionality
 */


/*
 * General per-processor initialization
 */

#if defined(DRV_IA32)

typedef union {
    unsigned long long    u64[1];
    unsigned short int    u16[4];
} local_handler_t;

/* ------------------------------------------------------------------------- */
/*!
 * @fn void cpumon_Save_Cpu(param)
 *
 * @param    param    unused parameter
 *
 * @return   None     No return needed
 *
 * @brief  Save the old handler for restoration when done
 *
 */
static void
cpumon_Save_Cpu (
    PVOID parm
)
{
    unsigned long        eflags;
    U64                 *idt_base;
    CPU_STATE            pcpu;

    SEP_DRV_LOG_TRACE_IN("Saving current interrupt handler...");

    preempt_disable();
    pcpu = &pcb[CONTROL_THIS_CPU()];
    preempt_enable();

    SYS_Local_Irq_Save(eflags);
    CPU_STATE_idt_base(pcpu) = idt_base = SYS_Get_IDT_Base();
    // save original perf. vector
    CPU_STATE_saved_ih(pcpu) = idt_base[CPU_PERF_VECTOR];
    SEP_DRV_LOG_TRACE("saved_ih is 0x%llx.", CPU_STATE_saved_ih(pcpu));
    SYS_Local_Irq_Restore(eflags);

    SEP_DRV_LOG_TRACE_OUT("Complete!");
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn void cpumon_Init_Cpu(param)
 *
 * @param    param    unused parameter
 *
 * @return   None     No return needed
 *
 * @brief  Set up the interrupt handler.
 *
 */
static VOID
cpumon_Init_Cpu (
    PVOID parm
)
{
    unsigned long        eflags;
    U64                 *idt_base;
    CPU_STATE            pcpu;
    local_handler_t      lhandler;

    SEP_DRV_LOG_TRACE_IN("Installing interrupt handler...");

    preempt_disable();
    pcpu = &pcb[CONTROL_THIS_CPU()];
    preempt_enable();
    SYS_Local_Irq_Save(eflags);

    idt_base = CPU_STATE_idt_base(pcpu);
    // install perf. handler
    // These are the necessary steps to have an ISR entry
    // Note the changes in the data written
    lhandler.u64[0] = (unsigned long)SYS_Perfvec_Handler;
    lhandler.u16[3] = lhandler.u16[1];
    lhandler.u16[1] = SYS_Get_cs();
    lhandler.u16[2] = 0xee00;

    idt_base[CPU_PERF_VECTOR] = lhandler.u64[0];
    SYS_Local_Irq_Restore(eflags);

    SEP_DRV_LOG_TRACE_OUT("Complete!");
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn void cpumon_Destroy_Cpu(param)
 *
 * @param    param    unused parameter
 *
 * @return   None     No return needed
 *
 * @brief  Restore the old handler
 * @brief  Finish clean up of the apic
 *
 */
static VOID
cpumon_Destroy_Cpu (
    PVOID ctx
)
{
    unsigned long        eflags;
    unsigned long long  *idt_base;
    CPU_STATE            pcpu;

    SEP_DRV_LOG_TRACE_IN("Uninstalling interrupt handler...");

    preempt_disable();
    pcpu = &pcb[CONTROL_THIS_CPU()];
    preempt_enable();

    SYS_Local_Irq_Save(eflags);
    // restore perf. vector (to a safe stub pointer)
    idt_base = SYS_Get_IDT_Base();
    APIC_Disable_PMI();
    idt_base[CPU_PERF_VECTOR] = CPU_STATE_saved_ih(pcpu);
    SYS_Local_Irq_Restore(eflags);

    SEP_DRV_LOG_TRACE_OUT("Complete!");
    return;
}
#endif

#if defined(DRV_EM64T) && !defined(DRV_USE_NMI)
/* ------------------------------------------------------------------------- */
/*!
 * @fn void cpumon_Set_IDT_Func(idt, func)
 *
 * @param  GATE_STRUCT*  - address of the idt vector
 * @param  PVOID         - function to set in IDT
 *
 * @return None     No return needed
 *
 * @brief  Set up the interrupt handler.
 * @brief  Save the old handler for restoration when done
 *
 */
static VOID
cpumon_Set_IDT_Func (
    GATE_STRUCT   *idt,
    PVOID          func
)
{
    SEP_DRV_LOG_TRACE_IN("Installing interrupt handler...");

    setidt(CPU_PERF_VECTOR, func, SDT_SYSIGT, SEL_KPL, 0);

    SEP_DRV_LOG_TRACE_OUT("Complete!");
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn void cpumon_Save_Cpu(param)
 *
 * @param  param - Unused, set up to enable parallel calls
 *
 * @return None     No return needed
 *
 * @brief  Set up the interrupt handler.
 * @brief  Save the old handler for restoration when done
 *
 */
static VOID
cpumon_Save_Cpu (
    PVOID parm
)
{
    unsigned long        eflags;
    IDTGDT_DESC          idt_base;
    CPU_STATE            pcpu = &pcb[CONTROL_THIS_CPU()];
    GATE_STRUCT          old_gate;
    GATE_STRUCT         *idt;

    SEP_DRV_LOG_TRACE_IN("Saving current interrupt handler...");

    SYS_Local_Irq_Save(eflags);
    SYS_Get_IDT_Base((PVOID*)&idt_base);
    idt  = idt_base.idtgdt_base;

    CPU_STATE_idt_base(pcpu) = idt;
    memcpy (&old_gate, &idt[CPU_PERF_VECTOR], 16);

    CPU_STATE_saved_ih(pcpu)  = (PVOID) ((((U64) old_gate.gd_hioffset) << 16)   |
                                          ((U64) old_gate.gd_looffset));

    SEP_DRV_LOG_TRACE("saved_ih is 0x%llx.", CPU_STATE_saved_ih(pcpu));
    SYS_Local_Irq_Restore(eflags);

    SEP_DRV_LOG_TRACE_OUT("Complete!");
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn void cpumon_Init_Cpu(param)
 *
 * @param    param    unused parameter
 *
 * @return   None     No return needed
 *
 * @brief  Set up the interrupt handler.
 *
 */
static VOID
cpumon_Init_Cpu (
    PVOID parm
)
{
    unsigned long        eflags;
    CPU_STATE            pcpu = &pcb[CONTROL_THIS_CPU()];
    GATE_STRUCT         *idt;

    SEP_DRV_LOG_TRACE_IN("Installing interrupt handler...");

    SYS_Local_Irq_Save(eflags);
    idt = CPU_STATE_idt_base(pcpu);
    cpumon_Set_IDT_Func(idt, SYS_Perfvec_Handler);
    SYS_Local_Irq_Restore(eflags);

    SEP_DRV_LOG_TRACE_OUT("Complete!");
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn void cpumon_Destroy_Cpu(param)
 *
 * @param    param    unused parameter
 *
 * @return   None     No return needed
 *
 * @brief  Restore the old handler
 * @brief  Finish clean up of the apic
 *
 */
static VOID
cpumon_Destroy_Cpu (
    PVOID ctx
)
{
    unsigned long        eflags;
    CPU_STATE            pcpu = &pcb[CONTROL_THIS_CPU()];
    GATE_STRUCT         *idt;

    SEP_DRV_LOG_TRACE_IN("Uninstalling interrupt handler...");

    SYS_Local_Irq_Save(eflags);
    APIC_Disable_PMI();
    idt = CPU_STATE_idt_base(pcpu);
    cpumon_Set_IDT_Func(idt, CPU_STATE_saved_ih(pcpu));
    SYS_Local_Irq_Restore(eflags);

    SEP_DRV_LOG_TRACE_OUT("Complete!");
    return;
}
#endif /* defined(DRV_EM64T) && !defined(DRV_USE_NMI) */

#if defined(DRV_IA32) || defined(DRV_EM64T)
/* ------------------------------------------------------------------------- */
/*!
 * @fn extern void CPUMON_Install_Cpuhools(void)
 *
 * @param    None
 *
 * @return   None     No return needed
 *
 * @brief  set up the interrupt handler (on a per-processor basis)
 * @brief  Initialize the APIC in two phases (current CPU, then others)
 *
 */
extern VOID
CPUMON_Install_Cpuhooks (
    void
)
{
#ifdef DRV_USE_NMI
    SEP_DRV_LOG_INIT_IN("Installing NMI interrupt handler...");
    FBSD_PMC_Register_PMI_Handler();
    SEP_DRV_LOG_INIT_OUT("Successfully installed NMI interrupt handler!");
#else
    PVOID linear;
    S32  me = 0;

    SEP_DRV_LOG_INIT_IN("Installing interrupt handler...");

    CONTROL_Invoke_Parallel(cpumon_Save_Cpu, (PVOID)(size_t)me);
    CONTROL_Invoke_Parallel(cpumon_Init_Cpu, (PVOID)(size_t)me);

    linear = NULL;
    APIC_Init(&linear);
    CONTROL_Invoke_Parallel(APIC_Init, &linear);
    CONTROL_Invoke_Parallel(APIC_Install_Interrupt_Handler, (PVOID)(size_t)me);

    SEP_DRV_LOG_INIT_OUT("Complete!");
#endif
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn extern void CPUMON_Remove_Cpuhools(void)
 *
 * @param    None
 *
 * @return   None     No return needed
 *
 * @brief  De-Initialize the APIC in phases
 * @brief  clean up the interrupt handler (on a per-processor basis)
 *
 */
extern VOID
CPUMON_Remove_Cpuhooks (
    void
)
{
#ifdef DRV_USE_NMI
    SEP_DRV_LOG_INIT_IN("Uninstalling NMI interrupt handler...");
    FBSD_PMC_Unregister_PMI_Handler();
    SEP_DRV_LOG_INIT_OUT("Successfully uninstalled NMI interrupt handler!");
#else
    int            i;
    unsigned long  eflags;

    SEP_DRV_LOG_INIT_IN("Uninstalling interrupt handler...");

    SYS_Local_Irq_Save(eflags);
    cpumon_Destroy_Cpu((PVOID)(size_t)0);
    SYS_Local_Irq_Restore(eflags);
    CONTROL_Invoke_Parallel_XS(cpumon_Destroy_Cpu,
                               (PVOID)(size_t)0);

    // de-initialize APIC
    APIC_Unmap(CPU_STATE_apic_linear_addr(&pcb[0]));
    for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
        APIC_Deinit_Phase1(i);
    }

    SEP_DRV_LOG_INIT_OUT("Successfully uninstalled interrupt handler!");
#endif
}

#endif /* defined(DRV_IA32) || defined(DRV_EM64T) */
