/*
 * Copyright (C) 2005 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

#include "lwpmudrv_defines.h"

#include "lwpmudrv_types.h"
#include "rise_errors.h"
#include "lwpmudrv_ecb.h"
#include "lwpmudrv_struct.h"
#include "lwpmudrv.h"
#include "control.h"
#include "core2.h"
#include "utility.h"
#include "pebs.h"

static PVOID                          pebs_global_memory      = NULL;
static size_t                         pebs_global_memory_size = 0;

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID pebs_Corei7_Initialize_Threshold (dts, pebs_record_size)
 *
 * @brief       The nehalem specific initialization
 *
 * @param       NONE
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 */
static VOID
pebs_Corei7_Initialize_Threshold (
    DTS_BUFFER_EXT   dts,
    U32              pebs_record_size
)
{
    DTS_BUFFER_EXT_pebs_threshold(dts)  = DTS_BUFFER_EXT_pebs_base(dts) + pebs_record_size;

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID pebs_Corei7_Overflow ()
 *
 * @brief       The Nehalem specific overflow check
 *
 * @param       NONE
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 *    Check the global overflow field of the buffer descriptor.
 *    Precise events can be allocated on any of the 4 general purpose
 *    registers.
 */
static U64
pebs_Corei7_Overflow (
    S32  this_cpu,
    U64  overflow_status
)
{
    DTS_BUFFER_EXT   dtes     = CPU_STATE_dts_buffer(&pcb[this_cpu]);
    U8               status   = FALSE;
    PEBS_REC_EXT     pebs_base;

    if (!dtes) {
        return overflow_status;
    }
    status = (U8)((dtes) && (DTS_BUFFER_EXT_pebs_index(dtes) != DTS_BUFFER_EXT_pebs_base(dtes)));
    if (status) {
        pebs_base = (PEBS_REC_EXT)(UIOP)DTS_BUFFER_EXT_pebs_base(dtes);
        overflow_status  |= PEBS_REC_EXT_glob_perf_overflow(pebs_base);
    }

    return overflow_status;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID pebs_Corei7_Overflow_APEBS ()
 *
 * @brief       The SunnyCove/Icelake specific overflow check
 *
 * @param       this_cpu        - cpu id
 *              overflow_status - overflow status
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 *    Check the global overflow field of the buffer descriptor.
 *    Precise events can be allocated on any of the 8 general purpose
 *    registers or 4 fixed registers.
 */
static U64
pebs_Corei7_Overflow_APEBS (
    S32  this_cpu,
    U64  overflow_status
)
{
    DTS_BUFFER_EXT1   dtes     = CPU_STATE_dts_buffer(&pcb[this_cpu]);
    S8               *pebs_base, *pebs_index;
    ADAPTIVE_PEBS_BASIC_INFO      pb;
    U8                            pebs_ptr_check = FALSE;
    U32                           dev_idx = core_to_dev_map[this_cpu];
    DEV_CONFIG                    pcfg = LWPMU_DEVICE_pcfg(&devices[dev_idx]);

    if (!dtes) {
        return overflow_status;
    }
    pebs_base  = (S8 *)(UIOP)DTS_BUFFER_EXT1_pebs_base(dtes);
    pebs_index = (S8 *)(UIOP)DTS_BUFFER_EXT1_pebs_index(dtes);
    pebs_ptr_check = (pebs_base && pebs_index && pebs_base != pebs_index);

    if (pebs_ptr_check && DEV_CONFIG_enable_adaptive_pebs(pcfg)) {
        pb = (ADAPTIVE_PEBS_BASIC_INFO)pebs_base;
        overflow_status |= ADAPTIVE_PEBS_BASIC_INFO_applicable_counters(pb);
    }
    return overflow_status;
}
/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID pebs_Core2_Initialize_Threshold (dts, pebs_record_size)
 *
 * @brief       The Core2 specific initialization
 *
 * @param       NONE
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 */
static VOID
pebs_Core2_Initialize_Threshold (
    DTS_BUFFER_EXT   dts,
    U32              pebs_record_size
)
{
    DTS_BUFFER_EXT_pebs_threshold(dts)  = DTS_BUFFER_EXT_pebs_base(dts);

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID pebs_Core2_Overflow (dts, pebs_record_size)
 *
 * @brief       The Core2 specific overflow check
 *
 * @param       NONE
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 *    Check the base and the index fields of the circular buffer, if they are
 *    not the same, then a precise event has overflowed.  Precise events are
 *    allocated only on register#0.
 */
static U64
pebs_Core2_Overflow (
    S32  this_cpu,
    U64  overflow_status
)
{
    DTS_BUFFER_EXT   dtes     = CPU_STATE_dts_buffer(&pcb[this_cpu]);
    U8               status   = FALSE;

    if (!dtes) {
        return overflow_status;
    }
    status = (U8)((dtes) && (DTS_BUFFER_EXT_pebs_index(dtes) != DTS_BUFFER_EXT_pebs_base(dtes)));
    if (status) {
        // Merom allows only for general purpose register 0 to be precise capable
        overflow_status  |= 0x1;
    }

    return overflow_status;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID pebs_Modify_IP (sample, is_64bit_addr)
 *
 * @brief       Change the IP field in the sample to that in the PEBS record
 *
 * @param       sample        - sample buffer
 * @param       is_64bit_addr - are we in a 64 bit module
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 *              <NONE>
 */
static VOID
pebs_Modify_IP (
    void        *sample,
    DRV_BOOL     is_64bit_addr
)
{
    SampleRecordPC  *psamp = sample;
    DTS_BUFFER_EXT   dtes  = CPU_STATE_dts_buffer(&pcb[CONTROL_THIS_CPU()]);
    S8              *pebs_base, *pebs_index;
    PEBS_REC_EXT     pb;
    U8               pebs_ptr_check = FALSE;

    if (!dtes || !psamp) {
        return;
    }
    pebs_base  = (S8 *)(UIOP)DTS_BUFFER_EXT_pebs_base(dtes);
    pebs_index = (S8 *)(UIOP)DTS_BUFFER_EXT_pebs_index(dtes);
    pebs_ptr_check = (pebs_base && pebs_index && pebs_base != pebs_index);

    SEP_DRV_LOG_TRACE("In PEBS Fill Buffer: cpu %d.", CONTROL_THIS_CPU());
    if (pebs_ptr_check) {
        pb = (PEBS_REC_EXT)pebs_base;
        if (is_64bit_addr) {
            SAMPLE_RECORD_iip(psamp)    = PEBS_REC_EXT_linear_ip(pb);
            SAMPLE_RECORD_ipsr(psamp)   = PEBS_REC_EXT_r_flags(pb);
        }
        else {
            SAMPLE_RECORD_eip(psamp)    = PEBS_REC_EXT_linear_ip(pb) & 0xFFFFFFFF;
            SAMPLE_RECORD_eflags(psamp) = PEBS_REC_EXT_r_flags(pb) & 0xFFFFFFFF;
        }
    }
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID pebs_Modify_IP_With_Eventing_IP (sample, is_64bit_addr)
 *
 * @brief       Change the IP field in the sample to that in the PEBS record
 *
 * @param       sample        - sample buffer
 * @param       is_64bit_addr - are we in a 64 bit module
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 *              <NONE>
 */
static VOID
pebs_Modify_IP_With_Eventing_IP (
    void        *sample,
    DRV_BOOL     is_64bit_addr
)
{
    SampleRecordPC  *psamp = sample;
    U32              this_cpu = CONTROL_THIS_CPU();
    DTS_BUFFER_EXT   dtes  = CPU_STATE_dts_buffer(&pcb[this_cpu]);
    S8              *pebs_base, *pebs_index;
    U64              ip = 0, flags = 0;
    U8               pebs_ptr_check = FALSE;
    U32              dev_idx = core_to_dev_map[this_cpu];
    DEV_CONFIG       pcfg = LWPMU_DEVICE_pcfg(&devices[dev_idx]);

    if (!dtes || !psamp) {
        return;
    }
    pebs_base  = (S8 *)(UIOP)DTS_BUFFER_EXT_pebs_base(dtes);
    pebs_index = (S8 *)(UIOP)DTS_BUFFER_EXT_pebs_index(dtes);
    SEP_DRV_LOG_TRACE("In PEBS Fill Buffer: cpu %d.", this_cpu);
    pebs_ptr_check = (pebs_base && pebs_index && pebs_base != pebs_index);
    if (!pebs_ptr_check) {
        return;
    }
    if (DEV_CONFIG_enable_adaptive_pebs(pcfg)) {
        ip = ADAPTIVE_PEBS_BASIC_INFO_eventing_ip((ADAPTIVE_PEBS_BASIC_INFO)pebs_base);
        if (DEV_CONFIG_adaptive_pebs_collect_gpr(pcfg)) {
            flags = ADAPTIVE_PEBS_GPR_INFO_rflags((ADAPTIVE_PEBS_GPR_INFO)(pebs_base + LWPMU_DEVICE_apebs_gpr_offset(&devices[dev_idx])));
        }
    }
    else {
        ip    = PEBS_REC_EXT1_eventing_ip((PEBS_REC_EXT1)pebs_base);
        flags = PEBS_REC_EXT1_r_flags((PEBS_REC_EXT1)pebs_base);
    }
    if (is_64bit_addr) {
        SAMPLE_RECORD_iip(psamp)    = ip;
        SAMPLE_RECORD_ipsr(psamp)   = flags;
    }
    else {
        SAMPLE_RECORD_eip(psamp)    = ip & 0xFFFFFFFF;
        SAMPLE_RECORD_eflags(psamp) = flags & 0xFFFFFFFF;
    }
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID pebs_Modify_TSC (sample)
 *
 * @brief       Change the TSC field in the sample to that in the PEBS record
 *
 * @param       sample        - sample buffer
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 *              <NONE>
 */
static VOID
pebs_Modify_TSC (
    void        *sample
)
{
    SampleRecordPC  *psamp = sample;
    U32              this_cpu = CONTROL_THIS_CPU();
    DTS_BUFFER_EXT   dtes  = CPU_STATE_dts_buffer(&pcb[this_cpu]);
    S8              *pebs_base, *pebs_index;
    U64              tsc;
    U8               pebs_ptr_check = FALSE;
    U32              dev_idx = core_to_dev_map[this_cpu];
    DEV_CONFIG       pcfg = LWPMU_DEVICE_pcfg(&devices[dev_idx]);

    if (!dtes || !psamp) {
        return;
    }

    pebs_base  = (S8 *)(UIOP)DTS_BUFFER_EXT_pebs_base(dtes);
    pebs_index = (S8 *)(UIOP)DTS_BUFFER_EXT_pebs_index(dtes);
    pebs_ptr_check = (pebs_base && pebs_index && pebs_base != pebs_index);
    SEP_DRV_LOG_TRACE("In PEBS Fill Buffer: cpu %d.", this_cpu);
    if (!pebs_ptr_check) {
        return;
    }
    if (DEV_CONFIG_enable_adaptive_pebs(pcfg)) {
        tsc = ADAPTIVE_PEBS_BASIC_INFO_tsc((ADAPTIVE_PEBS_BASIC_INFO)pebs_base);
    }
    else {
        tsc = PEBS_REC_EXT2_tsc((PEBS_REC_EXT2)pebs_base);
    }
    SAMPLE_RECORD_tsc(psamp) = tsc;
    return;
}

/*
 * Initialize the pebs micro dispatch tables
 */
PEBS_DISPATCH_NODE  core2_pebs =
{
     pebs_Core2_Initialize_Threshold,
     pebs_Core2_Overflow,
     pebs_Modify_IP,
     NULL
};

PEBS_DISPATCH_NODE  core2p_pebs =
{
     pebs_Corei7_Initialize_Threshold,
     pebs_Core2_Overflow,
     pebs_Modify_IP,
     NULL
};

PEBS_DISPATCH_NODE  corei7_pebs =
{
     pebs_Corei7_Initialize_Threshold,
     pebs_Corei7_Overflow,
     pebs_Modify_IP,
     NULL
};

PEBS_DISPATCH_NODE  haswell_pebs =
{
     pebs_Corei7_Initialize_Threshold,
     pebs_Corei7_Overflow,
     pebs_Modify_IP_With_Eventing_IP,
     NULL
};

PEBS_DISPATCH_NODE  perfver4_pebs =
{
     pebs_Corei7_Initialize_Threshold,
     pebs_Corei7_Overflow,
     pebs_Modify_IP_With_Eventing_IP,
     pebs_Modify_TSC
};

PEBS_DISPATCH_NODE perfver4_apebs = // adaptive PEBS
{
    pebs_Corei7_Initialize_Threshold,
    pebs_Corei7_Overflow_APEBS,
    pebs_Modify_IP_With_Eventing_IP,
    pebs_Modify_TSC
};

#define PER_CORE_BUFFER_SIZE(dts_size, record_size)  (dts_size +  2 * (record_size) + 64)

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID* pebs_Alloc_DTS_Buffer (VOID)
 *
 * @brief       Allocate buffers used for latency and pebs sampling
 *
 * @param       NONE
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 *              Allocate the memory needed to hold the DTS and PEBS records buffer.
 *              This routine is called by a thread that corresponds to a single core
 */
static VOID*
pebs_Alloc_DTS_Buffer (
    U32 pebs_record_size
)
{
    UIOP            pebs_base;
    U32             dts_size;
    VOID           *dts_buffer;
    DTS_BUFFER_EXT  dts;
    int             this_cpu;
    CPU_STATE       pcpu;
    U32             dev_idx;
    DEV_CONFIG      pcfg;
    PEBS_DISPATCH   pebs_dispatch;

    /*
     * one PEBS record... need 2 records so that
     * threshold can be less than absolute max
     */
    preempt_disable();
    this_cpu = CONTROL_THIS_CPU();
    preempt_enable();
    dts_size      = sizeof(DTS_BUFFER_EXT_NODE);
    pcpu          = &pcb[this_cpu];
    dev_idx       = core_to_dev_map[this_cpu];
    pcfg          = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    pebs_dispatch = LWPMU_DEVICE_pebs_dispatch(&devices[dev_idx]);

    if (DEV_CONFIG_enable_adaptive_pebs(pcfg)) {
        if (DEV_CONFIG_pebs_mode(pcfg) == 6) {
            dts_size = sizeof(DTS_BUFFER_EXT1_NODE);
        }
        else if (DEV_CONFIG_pebs_mode(pcfg) == 7 || DEV_CONFIG_pebs_mode(pcfg) == 8) {
            dts_size = sizeof(DTS_BUFFER_EXT2_NODE);
        }
        else {
            SEP_DRV_LOG_ERROR("Invalid PEBS mode for Adaptive pebs(%d).", DEV_CONFIG_pebs_mode(pcfg));
            return NULL;
        }
    }
    /*
     * account for extra bytes to align PEBS base to cache line boundary
     */
    dts_buffer = (char *)pebs_global_memory + CPU_STATE_dts_buffer_offset(pcpu);
    if (!dts_buffer) {
        SEP_DRV_LOG_ERROR("Failed to allocate space for DTS buffer.");
        return NULL;
    }

    pebs_base = (UIOP)(dts_buffer) + dts_size;

    //  Make 32 byte aligned
    if ((pebs_base & 0x000001F) != 0x0) {
        pebs_base = ALIGN_32(pebs_base);
    }

    /*
     * Program the DTES Buffer for Precise EBS.
     * Set PEBS buffer for one PEBS record
     */
    dts = (DTS_BUFFER_EXT)dts_buffer;

    DTS_BUFFER_EXT_base(dts)            = 0;
    DTS_BUFFER_EXT_index(dts)           = 0;
    DTS_BUFFER_EXT_max(dts)             = 0;
    DTS_BUFFER_EXT_threshold(dts)       = 0;
    DTS_BUFFER_EXT_pebs_base(dts)       = pebs_base;
    DTS_BUFFER_EXT_pebs_index(dts)      = pebs_base;
    DTS_BUFFER_EXT_pebs_max(dts)        = pebs_base + 2 * pebs_record_size;

    pebs_dispatch->initialize_threshold(dts, pebs_record_size);

    SEP_DRV_LOG_TRACE("base --- %p.", DTS_BUFFER_EXT_pebs_base(dts));
    SEP_DRV_LOG_TRACE("index --- %p.", DTS_BUFFER_EXT_pebs_index(dts));
    SEP_DRV_LOG_TRACE("max --- %p.", DTS_BUFFER_EXT_pebs_max(dts));
    SEP_DRV_LOG_TRACE("threahold --- %p.", DTS_BUFFER_EXT_pebs_threshold(dts));
    SEP_DRV_LOG_TRACE("DTES buffer allocated for PEBS: %p.", dts_buffer);

    return dts_buffer;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID* pebs_Allocate_Buffers (VOID *params)
 *
 * @brief       Allocate memory and set up MSRs in preparation for PEBS
 *
 * @param       NONE
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 *              Set up the DS area and program the DS_AREA msrs in preparation
 *              for a PEBS run.  Save away the old value in the DS_AREA.
 *              This routine is called via the parallel thread call.
 */
static VOID
pebs_Allocate_Buffers (
    VOID  *params
)
{
    U64         value;
    U32         this_cpu         = CONTROL_THIS_CPU();
    CPU_STATE   pcpu             = &pcb[this_cpu];
    U32         dev_idx          = core_to_dev_map[this_cpu];
    DEV_CONFIG  pcfg             = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    U32         pebs_record_size = LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]);

    if (!DEV_CONFIG_num_events(pcfg) || !DEV_CONFIG_pebs_mode(pcfg)) {
        return;
    }

    SYS_Write_MSR(IA32_PEBS_ENABLE, 0LL);
    value = SYS_Read_MSR(IA32_MISC_ENABLE);
    if ((value & 0x80) && !(value & 0x1000)) {
        CPU_STATE_old_dts_buffer(pcpu) = (PVOID)(UIOP)SYS_Read_MSR(IA32_DS_AREA);
        CPU_STATE_dts_buffer(pcpu)     = pebs_Alloc_DTS_Buffer(pebs_record_size);
        SEP_DRV_LOG_TRACE("Old dts buffer - %p.", CPU_STATE_old_dts_buffer(pcpu));
        SEP_DRV_LOG_TRACE("New dts buffer - %p.", CPU_STATE_dts_buffer(pcpu));
        SYS_Write_MSR(IA32_DS_AREA, (U64)(UIOP)CPU_STATE_dts_buffer(pcpu));
    }

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID pebs_Dellocate_Buffers (VOID *params)
 *
 * @brief       Clean up PEBS buffers and restore older values into the DS_AREA
 *
 * @param       NONE
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 *              Clean up the DS area and all restore state prior to the sampling run
 *              This routine is called via the parallel thread call.
 */
static VOID
pebs_Deallocate_Buffers (
    VOID  *params
)
{
    U32         this_cpu = CONTROL_THIS_CPU();
    CPU_STATE   pcpu     = &pcb[this_cpu];
    U32         dev_idx  = core_to_dev_map[this_cpu];
    DEV_CONFIG  pcfg     = LWPMU_DEVICE_pcfg(&devices[dev_idx]);

    if (!DEV_CONFIG_num_events(pcfg) || !DEV_CONFIG_pebs_mode(pcfg)) {
        return;
    }

    SEP_DRV_LOG_TRACE("Entered deallocate buffers.");
    SYS_Write_MSR(IA32_DS_AREA, (U64)(UIOP)CPU_STATE_old_dts_buffer(pcpu));
    CPU_STATE_dts_buffer(pcpu) = NULL;

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          U64 PEBS_Overflowed (this_cpu, overflow_status)
 *
 * @brief       Figure out if the PEBS event caused an overflow
 *
 * @param       this_cpu        -- the current cpu
 *              overflow_status -- current value of the global overflow status
 *
 * @return      updated overflow_status
 *
 * <I>Special Notes:</I>
 *              Figure out if the PEBS area has data that need to be transferred
 *              to the output sample.
 *              Update the overflow_status that is passed and return this value.
 *              The overflow_status defines the events/status to be read
 */
extern U64
PEBS_Overflowed (
    S32  this_cpu,
    U64  overflow_status
)
{
    U32           dev_idx       = core_to_dev_map[this_cpu];
    PEBS_DISPATCH pebs_dispatch = LWPMU_DEVICE_pebs_dispatch(&devices[dev_idx]);

    return pebs_dispatch->overflow(this_cpu, overflow_status);
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID PEBS_Reset_Index (this_cpu)
 *
 * @brief       Reset the PEBS index pointer
 *
 * @param       this_cpu        -- the current cpu
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 *              reset index to next PEBS record to base of buffer
 */
extern VOID
PEBS_Reset_Index (
    S32    this_cpu
)
{
    DTS_BUFFER_EXT   dtes = CPU_STATE_dts_buffer(&pcb[this_cpu]);

    if (dtes) {
        SEP_DRV_LOG_TRACE("PEBS Reset Index: %d.", this_cpu);
        DTS_BUFFER_EXT_pebs_index(dtes) = DTS_BUFFER_EXT_pebs_base(dtes);
    }

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID PEBS_Reset_Counter (this_cpu, index, op_type)
 *
 * @brief       set reset value for PMC after overflow
 *
 * @param       this_cpu        -- the current cpu
 *              index           -- PMC register index
 *              op_type         -- Type of Operation
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 */
extern VOID
PEBS_Reset_Counter (
    S32    this_cpu,
    U32    index,
    U32    op_type
)
{
    DTS_BUFFER_EXT  dts             = CPU_STATE_dts_buffer(&pcb[this_cpu]);
    DTS_BUFFER_EXT1 dts_ext         = NULL;
    DTS_BUFFER_EXT2 dts_ext2        = NULL;
    U32             dev_idx         = core_to_dev_map[this_cpu];
    DEV_CONFIG      pcfg            = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    U32             counter_index   = 0;
    U32             gp_index        = 0;
    U32             fixed_index     = 0;
    U32             fixed_offset    = 0;
    CPU_STATE       pcpu            = &pcb[this_cpu];
    U32             cur_grp         = CPU_STATE_current_group(pcpu);
    ECB             pecb            = LWPMU_DEVICE_PMU_register_data(&devices[dev_idx])[cur_grp];
    U64             value;

    if (!DEV_CONFIG_num_events(pcfg)) {
        SEP_DRV_LOG_ERROR("No valid number of events");
        return;
    }

    if (!pecb) {
        SEP_DRV_LOG_ERROR("Invalid ECB data");
        return;
    }

    if (!dts) {
        SEP_DRV_LOG_ERROR("Invalid PEBS Buffer");
        return;
    }

    value = ECB_entries_reg_value(pecb, index);

    if (ECB_entries_fixed_reg_get(pecb, index)) {
        if (DEV_CONFIG_pebs_mode(pcfg) == 7 || DEV_CONFIG_pebs_mode(pcfg) == 8) {
            fixed_offset = EXT2_FIXED_OFFSET;
        } else if (DEV_CONFIG_pebs_mode(pcfg) <= 6) {
            fixed_offset = EXT1_FIXED_OFFSET;
        } else {
            SEP_DRV_LOG_ERROR("Not supported PEBS Mode");
            return;
        }

        if (op_type == PMU_OPERATION_ALL_REG) {
            counter_index = (ECB_entries_reg_id(pecb, index) - IA32_FIXED_CTR0 + fixed_offset);
        }
        else {
            counter_index = index - ECB_operations_register_start(pecb, PMU_OPERATION_DATA_FIXED) + fixed_offset;
        }
    }
    else {
        if (op_type == PMU_OPERATION_ALL_REG) {
            counter_index = (ECB_entries_reg_id(pecb, index) - IA32_PMC0);
        }
        else {
            counter_index = index - ECB_operations_register_start(pecb, PMU_OPERATION_DATA_GP);
        }
    }

    SEP_DRV_LOG_TRACE("PEBS Reset Counter: cpu %d, index=%u, value=%llx.",
            this_cpu, counter_index, value);

    switch(counter_index) {
        case 0:
            DTS_BUFFER_EXT_counter_reset0(dts) = value;
            break;
        case 1:
            DTS_BUFFER_EXT_counter_reset1(dts) = value;
            break;
        case 2:
            DTS_BUFFER_EXT_counter_reset2(dts) = value;
            break;
        case 3:
            DTS_BUFFER_EXT_counter_reset3(dts) = value;
            break;
    }

    if (counter_index < NUM_EXT_COUNTER_RESET) {
	    return;
    }

    if (!DEV_CONFIG_enable_adaptive_pebs(pcfg)) {
        return;
    }

    if (DEV_CONFIG_pebs_mode(pcfg) == 6) {
        dts_ext = CPU_STATE_dts_buffer(&pcb[this_cpu]);

        switch(counter_index) {
            case 4:
                DTS_BUFFER_EXT1_counter_reset4(dts_ext) = value;
                break;
            case 5:
                DTS_BUFFER_EXT1_counter_reset5(dts_ext) = value;
                break;
            case 6:
                DTS_BUFFER_EXT1_counter_reset6(dts_ext) = value;
                break;
            case 7:
                DTS_BUFFER_EXT1_counter_reset7(dts_ext) = value;
                break;
            case 8:
                DTS_BUFFER_EXT1_fixed_counter_reset0(dts_ext) = value;
                break;
            case 9:
                DTS_BUFFER_EXT1_fixed_counter_reset1(dts_ext) = value;
                break;
            case 10:
                DTS_BUFFER_EXT1_fixed_counter_reset2(dts_ext) = value;
                break;
            case 11:
                DTS_BUFFER_EXT1_fixed_counter_reset3(dts_ext) = value;
                break;
        }
    }
    else if (DEV_CONFIG_pebs_mode(pcfg) == 7 || DEV_CONFIG_pebs_mode(pcfg) == 8) {
        dts_ext2 = CPU_STATE_dts_buffer(&pcb[this_cpu]);

        if (counter_index < (NUM_EXT_COUNTER_RESET + NUM_EXT2_COUNTER_RESET)) {
            gp_index = counter_index - NUM_EXT_COUNTER_RESET;
            DTS_BUFFER_EXT2_counter_reset(dts_ext2)[gp_index] = value;
        }
        else {
            fixed_index = counter_index - (NUM_EXT_COUNTER_RESET + NUM_EXT2_COUNTER_RESET);
            if (fixed_index <= NUM_EXT2_FIXED_COUNTER_RESET) {
                DTS_BUFFER_EXT2_fixed_counter_reset(dts_ext2)[fixed_index] = value;
            }
        }
    }
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID PEBS_Modify_IP (sample, is_64bit_addr)
 *
 * @brief       Change the IP field in the sample to that in the PEBS record
 *
 * @param       sample        - sample buffer
 * @param       is_64bit_addr - are we in a 64 bit module
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 *              <NONE>
 */
extern VOID
PEBS_Modify_IP (
    void        *sample,
    DRV_BOOL     is_64bit_addr
)
{
    U32           this_cpu      = CONTROL_THIS_CPU();
    U32           dev_idx       = core_to_dev_map[this_cpu];
    PEBS_DISPATCH pebs_dispatch = LWPMU_DEVICE_pebs_dispatch(&devices[dev_idx]);

    pebs_dispatch->modify_ip(sample, is_64bit_addr);
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID PEBS_Modify_TSC (sample)
 *
 * @brief       Change the TSC field in the sample to that in the PEBS record
 *
 * @param       sample        - sample buffer
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 *              <NONE>
 */
extern VOID
PEBS_Modify_TSC (
    void        *sample
)
{
    U32           this_cpu      = CONTROL_THIS_CPU();
    U32           dev_idx       = core_to_dev_map[this_cpu];
    PEBS_DISPATCH pebs_dispatch = LWPMU_DEVICE_pebs_dispatch(&devices[dev_idx]);

    if (pebs_dispatch->modify_tsc != NULL) {
        pebs_dispatch->modify_tsc(sample);
    }
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID PEBS_Fill_Buffer (S8 *buffer, EVENT_CONFIG ec)
 *
 * @brief       Fill the buffer with the pebs data
 *
 * @param       buffer  -  area to write the data into
 *              ec      -  current event config
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 *              <NONE>
 */
extern U64
PEBS_Fill_Buffer (
    S8           *buffer,
    EVENT_DESC    evt_desc
)
{
    LATENCY_INFO_NODE   latency_info  = {0};
    PEBS_REC_EXT1       pebs_base_ext1;
    S8                 *pebs_base, *pebs_index;
    U8                  pebs_ptr_check = FALSE;
    U64                 lbr_tos_from_ip = 0ULL;
    U32                 this_cpu = CONTROL_THIS_CPU();
    DTS_BUFFER_EXT      dtes     = CPU_STATE_dts_buffer(&pcb[this_cpu]);
    U32                 dev_idx  = core_to_dev_map[this_cpu];
    DEV_CONFIG          pcfg     = LWPMU_DEVICE_pcfg(&devices[dev_idx]);

    if (!DEV_CONFIG_num_events(pcfg)) {
        return lbr_tos_from_ip;
    }

    if (DEV_CONFIG_enable_adaptive_pebs(pcfg)) {
        lbr_tos_from_ip = APEBS_Fill_Buffer(buffer,
                                            evt_desc);
        return lbr_tos_from_ip;
    }
    if (!dtes) {
        return lbr_tos_from_ip;
    }
    pebs_base  = (S8 *)(UIOP)DTS_BUFFER_EXT_pebs_base(dtes);
    pebs_index = (S8 *)(UIOP)DTS_BUFFER_EXT_pebs_index(dtes);
    SEP_DRV_LOG_TRACE("In PEBS Fill Buffer: cpu %d.", CONTROL_THIS_CPU());
    pebs_ptr_check = (pebs_base && pebs_index && pebs_base != pebs_index);
    if (!pebs_ptr_check) {
        return lbr_tos_from_ip;
    }
    if (EVENT_DESC_pebs_offset(evt_desc)) {
        SEP_DRV_LOG_TRACE("PEBS buffer has data available.");
        memcpy(buffer + EVENT_DESC_pebs_offset(evt_desc),
               pebs_base,
               EVENT_DESC_pebs_size(evt_desc));
    }
    if (EVENT_DESC_eventing_ip_offset(evt_desc)) {
        pebs_base_ext1 = (PEBS_REC_EXT1)pebs_base;
        *(U64*)(buffer + EVENT_DESC_eventing_ip_offset(evt_desc)) = PEBS_REC_EXT1_eventing_ip(pebs_base_ext1);
    }
    if (EVENT_DESC_hle_offset(evt_desc)) {
        pebs_base_ext1 = (PEBS_REC_EXT1)pebs_base;
        *(U64*)(buffer + EVENT_DESC_hle_offset(evt_desc)) = PEBS_REC_EXT1_hle_info(pebs_base_ext1);
    }
    if (EVENT_DESC_latency_offset_in_sample(evt_desc)) {
        pebs_base_ext1 = (PEBS_REC_EXT1)pebs_base;
        memcpy(&latency_info,
                pebs_base + EVENT_DESC_latency_offset_in_pebs_record(evt_desc),
                EVENT_DESC_latency_size_from_pebs_record(evt_desc));
        memcpy(&LATENCY_INFO_stack_pointer(&latency_info),
               &PEBS_REC_EXT1_rsp(pebs_base_ext1),
               sizeof(U64));
        memcpy(buffer + EVENT_DESC_latency_offset_in_sample(evt_desc),
               &latency_info,
               sizeof(LATENCY_INFO_NODE) );
    }
    return lbr_tos_from_ip;
}
/* ------------------------------------------------------------------------- */
/*!
 * @fn          U64 APEBS_Fill_Buffer (S8 *buffer, EVENT_DESC evt_desc)
 *
 * @brief       Fill the buffer with the pebs data
 *
 * @param       buffer                   -  area to write the data into
 *              event_desc               -  event descriptor of the pebs event
 *
 * @return      LBR_TOS_FROM_IP
 *
 * <I>Special Notes:</I>
 *              <NONE>
 */
extern U64
APEBS_Fill_Buffer (
    S8           *buffer,
    EVENT_DESC    evt_desc
)
{
    DTS_BUFFER_EXT1             dtes;
    LATENCY_INFO_NODE           latency_info           = {0};
    U64                         dtes_record_size       = 0;
    U64                         dtes_record_format     = 0;
    U64                         dtes_num_lbr_entries   = 0;
    ADAPTIVE_PEBS_MEM_INFO      apebs_mem              = NULL;
    ADAPTIVE_PEBS_GPR_INFO      apebs_gpr              = NULL;
    ADAPTIVE_PEBS_BASIC_INFO    apebs_basic            = NULL;
    S8                         *pebs_base, *pebs_index;
    U64                         lbr_tos_from_ip        = 0ULL;
    U8                          pebs_ptr_check         = FALSE;
    U32                         this_cpu               = CONTROL_THIS_CPU();
    U32                         dev_idx                = core_to_dev_map[this_cpu];
    DEV_CONFIG                  pcfg                   = LWPMU_DEVICE_pcfg(&devices[dev_idx]);

    if (!DEV_CONFIG_num_events(pcfg)) {
        return lbr_tos_from_ip;
    }

    dtes = CPU_STATE_dts_buffer(&pcb[this_cpu]);

    if (!dtes || !DEV_CONFIG_enable_adaptive_pebs(pcfg)) {
        return lbr_tos_from_ip;
    }
    pebs_base  = (S8 *)(UIOP)DTS_BUFFER_EXT1_pebs_base(dtes);
    pebs_index = (S8 *)(UIOP)DTS_BUFFER_EXT1_pebs_index(dtes);
    pebs_ptr_check = (pebs_base && pebs_index && pebs_base != pebs_index);
    if (!pebs_ptr_check) {
        return lbr_tos_from_ip;
    }
    apebs_basic        = (ADAPTIVE_PEBS_BASIC_INFO)(pebs_base + LWPMU_DEVICE_apebs_basic_offset(&devices[dev_idx]));

    dtes_record_size     = (ADAPTIVE_PEBS_BASIC_INFO_record_info(apebs_basic) & APEBS_RECORD_SIZE_MASK) >> 48;     // [63:48]
    dtes_record_format   = (ADAPTIVE_PEBS_BASIC_INFO_record_info(apebs_basic) & APEBS_RECORD_FORMAT_MASK);         // [23:0]
    dtes_num_lbr_entries = (ADAPTIVE_PEBS_BASIC_INFO_record_info(apebs_basic) & APEBS_NUM_LBR_ENTRIES_MASK) >> 24; // [31:24]

    if (EVENT_DESC_pebs_offset(evt_desc)) {
        *(U64*)(buffer + EVENT_DESC_pebs_offset(evt_desc)) = ADAPTIVE_PEBS_BASIC_INFO_record_info(apebs_basic);
    }
    if (EVENT_DESC_eventing_ip_offset(evt_desc)) {
        *(U64*)(buffer + EVENT_DESC_eventing_ip_offset(evt_desc)) = ADAPTIVE_PEBS_BASIC_INFO_eventing_ip(apebs_basic);
    }
    if (EVENT_DESC_pebs_tsc_offset(evt_desc)) {
        *(U64*)(buffer + EVENT_DESC_pebs_tsc_offset(evt_desc)) = ADAPTIVE_PEBS_BASIC_INFO_tsc(apebs_basic);
    }
    if (EVENT_DESC_applicable_counters_offset(evt_desc)) {
        *(U64*)(buffer + EVENT_DESC_applicable_counters_offset(evt_desc)) = ADAPTIVE_PEBS_BASIC_INFO_applicable_counters(apebs_basic);
    }
    if (DEV_CONFIG_adaptive_pebs_collect_gpr(pcfg) && EVENT_DESC_gpr_info_offset(evt_desc)) {
        if (!(dtes_record_format & APEBS_GPR_RECORD_FORMAT_MASK)) {
            SEP_DRV_LOG_TRACE("GPR info not found in DS PEBS record.");
        }
        memcpy(buffer    + EVENT_DESC_gpr_info_offset(evt_desc),
               pebs_base + LWPMU_DEVICE_apebs_gpr_offset(&devices[dev_idx]),
               EVENT_DESC_gpr_info_size(evt_desc));
    }
    if (DEV_CONFIG_adaptive_pebs_collect_mem_info(pcfg) && EVENT_DESC_latency_offset_in_sample(evt_desc)) {
        if (!(dtes_record_format & APEBS_MEM_RECORD_FORMAT_MASK)) {
            SEP_DRV_LOG_TRACE("MEM info not found in DS PEBS record.");
        }
        apebs_mem = (ADAPTIVE_PEBS_MEM_INFO)(pebs_base + LWPMU_DEVICE_apebs_mem_offset(&devices[dev_idx]));
        memcpy(&LATENCY_INFO_linear_address(&latency_info),
               &ADAPTIVE_PEBS_MEM_INFO_data_linear_address(apebs_mem),
               sizeof(U64));
        memcpy(&LATENCY_INFO_data_source(&latency_info),
               &ADAPTIVE_PEBS_MEM_INFO_data_source(apebs_mem),
               sizeof(U64));
        memcpy(&LATENCY_INFO_latency(&latency_info),
               &ADAPTIVE_PEBS_MEM_INFO_latency(apebs_mem),
               sizeof(U64));
        LATENCY_INFO_stack_pointer(&latency_info) = 0;
        if (DEV_CONFIG_adaptive_pebs_collect_gpr(pcfg)) {
            apebs_gpr = (ADAPTIVE_PEBS_GPR_INFO)(pebs_base + LWPMU_DEVICE_apebs_gpr_offset(&devices[dev_idx]));
            memcpy(&LATENCY_INFO_stack_pointer(&latency_info),
                   &ADAPTIVE_PEBS_GPR_INFO_rsp(apebs_gpr),
                   sizeof(U64));
        }
        memcpy(buffer + EVENT_DESC_latency_offset_in_sample(evt_desc),
               &latency_info,
               sizeof(LATENCY_INFO_NODE));
    }
    if (DEV_CONFIG_adaptive_pebs_collect_mem_info(pcfg) && EVENT_DESC_hle_offset(evt_desc)) {
        *(U64*)(buffer + EVENT_DESC_hle_offset(evt_desc)) =  ADAPTIVE_PEBS_MEM_INFO_hle_info((ADAPTIVE_PEBS_MEM_INFO)(pebs_base + LWPMU_DEVICE_apebs_mem_offset(&devices[dev_idx])));
    }
    if (DEV_CONFIG_adaptive_pebs_collect_xmm(pcfg) && EVENT_DESC_xmm_info_offset(evt_desc)) {
        if (!(dtes_record_format & APEBS_XMM_RECORD_FORMAT_MASK)) {
            SEP_DRV_LOG_TRACE("XMM info not found in DS PEBS record.");
        }
        memcpy(buffer    + EVENT_DESC_xmm_info_offset(evt_desc),
               pebs_base + LWPMU_DEVICE_apebs_xmm_offset(&devices[dev_idx]),
               EVENT_DESC_xmm_info_size(evt_desc));
    }
    if (DEV_CONFIG_adaptive_pebs_collect_lbrs(pcfg) && EVENT_DESC_lbr_offset(evt_desc)) {
        if (!(dtes_record_format & APEBS_LBR_RECORD_FORMAT_MASK)) {
            SEP_DRV_LOG_TRACE("LBR info not found in DS PEBS record.");
        }
        if (dtes_num_lbr_entries != (DEV_CONFIG_num_lbr_entries(pcfg)-1)) {
            SEP_DRV_LOG_TRACE("DRV_CONFIG_num_lbr_entries does not match with PEBS record.");
        }
        *(U64*)(buffer + EVENT_DESC_lbr_offset(evt_desc)) =
                           DEV_CONFIG_num_lbr_entries(pcfg)-1; //Top-of-Stack(TOS) pointing to last entry
        //Populating lbr callstack as SST_ENTRY_N to SST_ENTRY_0 in tb util, hence setting TOS to SST_ENTRY_N
        memcpy(buffer    + EVENT_DESC_lbr_offset(evt_desc) + sizeof(U64),
               pebs_base + LWPMU_DEVICE_apebs_lbr_offset(&devices[dev_idx]),
               EVENT_DESC_lbr_info_size(evt_desc)-sizeof(U64));
        lbr_tos_from_ip = ADAPTIVE_PEBS_LBR_INFO_lbr_from((ADAPTIVE_PEBS_LBR_INFO)(pebs_base + LWPMU_DEVICE_apebs_lbr_offset(&devices[dev_idx])));
    }
	if (DEV_CONFIG_adaptive_pebs_collect_css(pcfg) && EVENT_DESC_css_info_offset(evt_desc)) {
		memcpy(buffer + EVENT_DESC_css_info_offset(evt_desc),
			   pebs_base + LWPMU_DEVICE_apebs_css_offset(&devices[dev_idx]),
			   EVENT_DESC_css_info_size(evt_desc));
	}
    return lbr_tos_from_ip;
}
/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID PEBS_Initialize (U32 dev_idx)
 *
 * @brief       Initialize the pebs buffers
 *
 * @param       dev_idx - Device index
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 *              If the user is asking for PEBS information.  Allocate the DS area
 */
extern VOID
PEBS_Initialize (
    U32  dev_idx
)
{
    DEV_CONFIG pcfg = LWPMU_DEVICE_pcfg(&devices[dev_idx]);

    SEP_DRV_LOG_INIT_IN("dev_idx: %p.", dev_idx);

    if (!DEV_CONFIG_pebs_mode(pcfg)) {
        SEP_DRV_LOG_INIT_OUT("Early return: not in PEBS mode.");
        return;
    }

    switch (DEV_CONFIG_pebs_mode(pcfg)) {
        case 1:
            SEP_DRV_LOG_INIT("Set up the Core2 dispatch table.");
            LWPMU_DEVICE_pebs_dispatch(&devices[dev_idx]) = &core2_pebs;
            LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]) = sizeof(PEBS_REC_NODE);
            break;
        case 2:
            SEP_DRV_LOG_INIT("Set up the Nehalem dispatch.");
            LWPMU_DEVICE_pebs_dispatch(&devices[dev_idx]) = &corei7_pebs;
            LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]) = sizeof(PEBS_REC_EXT_NODE);
            break;
        case 3:
            SEP_DRV_LOG_INIT("Set up the Core2 (PNR) dispatch table.");
            LWPMU_DEVICE_pebs_dispatch(&devices[dev_idx]) = &core2p_pebs;
            LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]) = sizeof(PEBS_REC_NODE);
            break;
        case 4:
            SEP_DRV_LOG_INIT("Set up the Haswell dispatch table.");
            LWPMU_DEVICE_pebs_dispatch(&devices[dev_idx]) = &haswell_pebs;
            LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]) = sizeof(PEBS_REC_EXT1_NODE);
            break;
        case 5:
            SEP_DRV_LOG_INIT("Set up the Perf version4 dispatch table.");
            LWPMU_DEVICE_pebs_dispatch(&devices[dev_idx]) = &perfver4_pebs;
            LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]) = sizeof(PEBS_REC_EXT2_NODE);
            break;
        case 6:
        case 7:
        case 8:
            if (!DEV_CONFIG_enable_adaptive_pebs(pcfg)) {
                SEP_DRV_LOG_INIT("APEBS needs to be enabled in perf version4 SNC dispatch mode.");
            }
            LWPMU_DEVICE_pebs_dispatch(&devices[dev_idx]) = &perfver4_apebs;
            LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]) = sizeof(ADAPTIVE_PEBS_BASIC_INFO_NODE);
            if (DEV_CONFIG_adaptive_pebs_collect_mem_info(pcfg)) {
                LWPMU_DEVICE_apebs_mem_offset(&devices[dev_idx])  = LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]);
                LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]) += sizeof(ADAPTIVE_PEBS_MEM_INFO_NODE);
            }
            if (DEV_CONFIG_adaptive_pebs_collect_gpr(pcfg)) {
                LWPMU_DEVICE_apebs_gpr_offset(&devices[dev_idx])  = LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]);
                LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]) += sizeof(ADAPTIVE_PEBS_GPR_INFO_NODE);
            }
            if (DEV_CONFIG_adaptive_pebs_collect_xmm(pcfg)) {
                LWPMU_DEVICE_apebs_xmm_offset(&devices[dev_idx])  = LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]);
                LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]) += sizeof(ADAPTIVE_PEBS_XMM_INFO_NODE);
            }
            if (DEV_CONFIG_adaptive_pebs_collect_lbrs(pcfg)) {
                LWPMU_DEVICE_apebs_lbr_offset(&devices[dev_idx])  = LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]);
                LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]) += (sizeof(ADAPTIVE_PEBS_LBR_INFO_NODE) * DEV_CONFIG_num_lbr_entries(pcfg));
            }
            if (DEV_CONFIG_adaptive_pebs_collect_css(pcfg)) {
                U32 num_perf_metrics_entries = DEV_CONFIG_num_perf_metrics(pcfg) ? (sizeof(U64) * 2) : 0;
                LWPMU_DEVICE_apebs_css_offset(&devices[dev_idx]) = LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]);
                LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]) += (sizeof(ADAPTIVE_PEBS_CSS_INFO_NODE) + (sizeof(U64) * LWPMU_DEVICE_num_events(&devices[dev_idx])) + num_perf_metrics_entries);
            }
            SEP_DRV_LOG_INIT("Size of adaptive pebs record - %d.", LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]));
            break;
        default:
            SEP_DRV_LOG_WARNING("Unknown PEBS type. Will not collect PEBS information.");
            break;
    }

    SEP_DRV_LOG_INIT_OUT("");
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID PEBS_Allocate (void)
 *
 * @brief       Allocate the pebs related buffers
 *
 * @param       NONE
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 *             Allocated the DS area used for PEBS capture
 */
extern VOID
PEBS_Allocate (
    VOID
)
{
    S32        cpu_num;
    CPU_STATE  pcpu;
    U32        dev_idx;
    U32        dts_size;
    DEV_CONFIG pcfg;

    SEP_DRV_LOG_INIT_IN("");

    for (cpu_num = 0; cpu_num < GLOBAL_STATE_num_cpus(driver_state); cpu_num++) {
        pcpu    = &pcb[cpu_num];
        dev_idx = core_to_dev_map[cpu_num];
        pcfg    = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
        if (LWPMU_DEVICE_pebs_dispatch(&devices[dev_idx])) {
            if (DEV_CONFIG_enable_adaptive_pebs(pcfg)) {
                if (DEV_CONFIG_pebs_mode(pcfg) == 6) {
                    dts_size = sizeof(DTS_BUFFER_EXT1_NODE);
                }
                else if (DEV_CONFIG_pebs_mode(pcfg) == 7 || DEV_CONFIG_pebs_mode(pcfg) == 8) {
                    dts_size = sizeof(DTS_BUFFER_EXT2_NODE);
                }
                else {
                    SEP_DRV_LOG_ERROR("Invalid PEBS mode for Adaptive pebs(%d).", DEV_CONFIG_pebs_mode(pcfg));
                    return;
                }
            }
            else {
                dts_size = sizeof(DTS_BUFFER_EXT_NODE);
            }
            CPU_STATE_dts_buffer_offset(pcpu) = pebs_global_memory_size;
            pebs_global_memory_size += PER_CORE_BUFFER_SIZE(dts_size, LWPMU_DEVICE_pebs_record_size(&devices[dev_idx]));
        }
    }
    if (pebs_global_memory_size) {
        pebs_global_memory = (PVOID)CONTROL_Allocate_KMemory(pebs_global_memory_size);
#if !defined(DISABLE_PEBS)
#if __FreeBSD_version >= 1101511
        pmap_pti_add_kva((vm_offset_t)pebs_global_memory, (vm_offset_t)pebs_global_memory + pebs_global_memory_size, false);
#endif
#endif
    }

    CONTROL_Invoke_Parallel(pebs_Allocate_Buffers, NULL);

    SEP_DRV_LOG_INIT_OUT("");
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID PEBS_Destroy (VOID)
 *
 * @brief       Clean up the pebs related buffers
 *
 * @param       NONE
 *
 * @return      NONE
 *
 * <I>Special Notes:</I>
 *             Deallocated the DS area used for PEBS capture
 */
extern VOID
PEBS_Destroy (
    VOID
)
{
    SEP_DRV_LOG_INIT_IN("");

    CONTROL_Invoke_Parallel(pebs_Deallocate_Buffers, (VOID *)(size_t)0);
    if (pebs_global_memory) {
#if __FreeBSD_version >= 1101511
        pmap_pti_remove_kva((vm_offset_t)pebs_global_memory, (vm_offset_t)pebs_global_memory + pebs_global_memory_size);
#endif
        pebs_global_memory = CONTROL_Free_KMemory(pebs_global_memory,pebs_global_memory_size);
        pebs_global_memory_size = 0;
        SEP_DRV_LOG_INIT("PEBS buffer successfully freed.");
    }

    SEP_DRV_LOG_INIT_OUT("");
    return;
}

