// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

/*****************************************************************************
 *  LoongArch64 instructions for JIT compiler
 *
 *   id          -- the enum name for the instruction
 *   nm          -- textual name (for assembly dipslay)
 *   ld/st/cmp   -- load/store/compare instruction
 *   encode      -- encoding
 *   mask        -- instruction's mask
 *   fmt         -- disasmbly format
 *
 ******************************************************************************/

#if !defined(TARGET_LOONGARCH64)
#error Unexpected target type
#endif

#ifndef INST
#error INST must be defined before including this file.
#endif

/*****************************************************************************/
/*               The following is LOONGARCH64-specific                               */
/*****************************************************************************/

// If you're adding a new instruction:
// You need not only to fill in one of these macros describing the instruction, but also:
//   * If the instruction writes to more than one destination register, update the function
//     emitInsMayWriteMultipleRegs in emitLoongarch64.cpp.

// clang-format off
INST(invalid,       "INVALID",        0,    BAD_CODE,0,DF_G_INVALID)
INST(nop ,          "nop",            0,    0x03400000,0xffffffff,DF_G_ALIAS)

                    // INS_bceqz/INS_beq/INS_blt/INS_bltu must be even number.
INST(bceqz,         "bceqz",          0,    0x48000000,0xfc000300,DF_F_B1) // cj,offs20<<2
INST(bcnez,         "bcnez",          0,    0x48000100,0xfc000300,DF_F_B1)

INST(beq,           "beq",            0,    0x58000000,0xfc000000,DF_G_B2) // rj,rd,offs16<<2
INST(bne,           "bne",            0,    0x5c000000,0xfc000000,DF_G_B2)
INST(blt,           "blt",            0,    0x60000000,0xfc000000,DF_G_B2)
INST(bge,           "bge",            0,    0x64000000,0xfc000000,DF_G_B2)
INST(bltu,          "bltu",           0,    0x68000000,0xfc000000,DF_G_B2)
INST(bgeu,          "bgeu",           0,    0x6c000000,0xfc000000,DF_G_B2)
////R_I.
INST(beqz,          "beqz",           0,    0x40000000,0xfc000000,DF_G_B1) // rj,offs20<<2
INST(bnez,          "bnez",           0,    0x44000000,0xfc000000,DF_G_B1)

////I.
INST(b,             "b",              0,    0x50000000,0xfc000000,DF_G_B0) // offs26<<2
INST(bl,            "bl",             0,    0x54000000,0xfc000000,DF_G_B0)

///////////////////////////////////////////////////////////////////////////////////////////
////NOTE:  Begin
////     the following instructions will be used by emitter::emitInsMayWriteToGCReg().
////////////////////////////////////////////////
//    enum     name     LD/ST   ENCODE    MASK    FMT   
//
////NOTE: mov must be the first one !!! more info to see emitter::emitInsMayWriteToGCReg().
///////////////////////////////////////////////////////////////////////////////////////////
//  mov     rd,rj
//  In fact, mov is an alias instruction, "ori rd,rj,0"
INST(mov,           "mov",            0,    0x03800000,0xfffffc00,DF_G_ALIAS)
                    // dneg is an alias instruction.
                    // sub_d rd, zero, rk
INST(dneg,          "dneg",           0,    0x00118000,0xffff83e0,DF_G_ALIAS)
                    // neg is an alias instruction.
                    // sub_w rd, zero, rk
INST(neg,           "neg",            0,    0x00110000,0xffff83e0,DF_G_ALIAS)
                    // not is an alias instruction.
                    // nor rd, rj, zero
INST(not,           "not",            0,    0x00140000,0xffff8000,DF_G_ALIAS)

//   enum:id        name             FP   LD/ST   Format   ENCODE
////R_R_R.
INST(add_w,         "add.w",          0,    0x00100000,0xffff8000,DF_G_3R) // rd,rj,rk
INST(add_d,         "add.d",          0,    0x00108000,0xffff8000,DF_G_3R)
INST(sub_w,         "sub.w",          0,    0x00110000,0xffff8000,DF_G_3R)
INST(sub_d,         "sub.d",          0,    0x00118000,0xffff8000,DF_G_3R)
INST(and,           "and",            0,    0x00148000,0xffff8000,DF_G_3R)
INST(or,            "or",             0,    0x00150000,0xffff8000,DF_G_3R)
INST(nor,           "nor",            0,    0x00140000,0xffff8000,DF_G_3R)
INST(xor,           "xor",            0,    0x00158000,0xffff8000,DF_G_3R)
INST(andn,          "andn",           0,    0x00168000,0xffff8000,DF_G_3R)
INST(orn,           "orn",            0,    0x00160000,0xffff8000,DF_G_3R)
INST(mul_w,         "mul.w",          0,    0x001c0000,0xffff8000,DF_G_3R)
INST(mul_d,         "mul.d",          0,    0x001d8000,0xffff8000,DF_G_3R)
INST(mulh_w,        "mulh.w",         0,    0x001c8000,0xffff8000,DF_G_3R)
INST(mulh_wu,       "mulh.wu",        0,    0x001d0000,0xffff8000,DF_G_3R)
INST(mulh_d,        "mulh.d",         0,    0x001e0000,0xffff8000,DF_G_3R)
INST(mulh_du,       "mulh.du",        0,    0x001e8000,0xffff8000,DF_G_3R)
INST(mulw_d_w,      "mulw.d.w",       0,    0x001f0000,0xffff8000,DF_G_3R)
INST(mulw_d_wu,     "mulw.d.wu",      0,    0x001f8000,0xffff8000,DF_G_3R)
INST(div_w,         "div.w",          0,    0x00200000,0xffff8000,DF_G_3R)
INST(div_wu,        "div.wu",         0,    0x00210000,0xffff8000,DF_G_3R)
INST(div_d,         "div.d",          0,    0x00220000,0xffff8000,DF_G_3R)
INST(div_du,        "div.du",         0,    0x00230000,0xffff8000,DF_G_3R)
INST(mod_w,         "mod.w",          0,    0x00208000,0xffff8000,DF_G_3R)
INST(mod_wu,        "mod.wu",         0,    0x00218000,0xffff8000,DF_G_3R)
INST(mod_d,         "mod.d",          0,    0x00228000,0xffff8000,DF_G_3R)
INST(mod_du,        "mod.du",         0,    0x00238000,0xffff8000,DF_G_3R)
INST(sll_w,         "sll.w",          0,    0x00170000,0xffff8000,DF_G_3R)
INST(srl_w,         "srl.w",          0,    0x00178000,0xffff8000,DF_G_3R)
INST(sra_w,         "sra.w",          0,    0x00180000,0xffff8000,DF_G_3R)
INST(rotr_w,        "rotr_w",         0,    0x001b0000,0xffff8000,DF_G_3R)
INST(sll_d,         "sll.d",          0,    0x00188000,0xffff8000,DF_G_3R)
INST(srl_d,         "srl.d",          0,    0x00190000,0xffff8000,DF_G_3R)
INST(sra_d,         "sra.d",          0,    0x00198000,0xffff8000,DF_G_3R)
INST(rotr_d,        "rotr.d",         0,    0x001b8000,0xffff8000,DF_G_3R)
INST(maskeqz,       "maskeqz",        0,    0x00130000,0xffff8000,DF_G_3R)
INST(masknez,       "masknez",        0,    0x00138000,0xffff8000,DF_G_3R)
INST(slt,           "slt",            0,    0x00120000,0xffff8000,DF_G_3R)
INST(sltu,          "sltu",           0,    0x00128000,0xffff8000,DF_G_3R)
INST(crc_w_b_w,     "crc.w.b.w",      0,    0x00240000,0xffff8000,DF_G_3R)
INST(crc_w_h_w,     "crc.w.h.w",      0,    0x00248000,0xffff8000,DF_G_3R)
INST(crc_w_w_w,     "crc.w.w.w",      0,    0x00250000,0xffff8000,DF_G_3R)
INST(crc_w_d_w,     "crc.w.d.w",      0,    0x00258000,0xffff8000,DF_G_3R)
INST(crcc_w_b_w,    "crcc.w.b.w",     0,    0x00260000,0xffff8000,DF_G_3R)
INST(crcc_w_h_w,    "crcc.w.h.w",     0,    0x00268000,0xffff8000,DF_G_3R)
INST(crcc_w_w_w,    "crcc.w.w.w",     0,    0x00270000,0xffff8000,DF_G_3R)
INST(crcc_w_d_w,    "crcc.w.d.w",     0,    0x00278000,0xffff8000,DF_G_3R)
INST(amswap_w,      "amswap.w",       0,    0x38600000,0xffff8000,DF_G_3R) // rd,rk,rj
INST(amswap_d,      "amswap.d",       0,    0x38608000,0xffff8000,DF_G_3R)
INST(amswap_db_w,   "amswap_db.w",    0,    0x38690000,0xffff8000,DF_G_3R)
INST(amswap_db_d,   "amswap_db.d",    0,    0x38698000,0xffff8000,DF_G_3R)
INST(amadd_w,       "amadd.w",        0,    0x38610000,0xffff8000,DF_G_3R)
INST(amadd_d,       "amadd.d",        0,    0x38618000,0xffff8000,DF_G_3R)
INST(amadd_db_w,    "amadd_db.w",     0,    0x386a0000,0xffff8000,DF_G_3R)
INST(amadd_db_d,    "amadd_db.d",     0,    0x386a8000,0xffff8000,DF_G_3R)
INST(amand_w,       "amand.w",        0,    0x38620000,0xffff8000,DF_G_3R)
INST(amand_d,       "amand.d",        0,    0x38628000,0xffff8000,DF_G_3R)
INST(amand_db_w,    "amand_db.w",     0,    0x386b0000,0xffff8000,DF_G_3R)
INST(amand_db_d,    "amand_db.d",     0,    0x386b8000,0xffff8000,DF_G_3R)
INST(amor_w,        "amor.w",         0,    0x38630000,0xffff8000,DF_G_3R)
INST(amor_d,        "amor.d",         0,    0x38638000,0xffff8000,DF_G_3R)
INST(amor_db_w,     "amor_db.w",      0,    0x386c0000,0xffff8000,DF_G_3R)
INST(amor_db_d,     "amor_db.d",      0,    0x386c8000,0xffff8000,DF_G_3R)
INST(amxor_w,       "amxor.w",        0,    0x38640000,0xffff8000,DF_G_3R)
INST(amxor_d,       "amxor.d",        0,    0x38648000,0xffff8000,DF_G_3R)
INST(amxor_db_w,    "amxor_db.w",     0,    0x386d0000,0xffff8000,DF_G_3R)
INST(amxor_db_d,    "amxor_db.d",     0,    0x386d8000,0xffff8000,DF_G_3R)
INST(ammax_w,       "ammax.w",        0,    0x38650000,0xffff8000,DF_G_3R)
INST(ammax_d,       "ammax.d",        0,    0x38658000,0xffff8000,DF_G_3R)
INST(ammax_db_w,    "ammax_db.w",     0,    0x386e0000,0xffff8000,DF_G_3R)
INST(ammax_db_d,    "ammax_db.d",     0,    0x386e8000,0xffff8000,DF_G_3R)
INST(ammin_w,       "ammin.w",        0,    0x38660000,0xffff8000,DF_G_3R)
INST(ammin_d,       "ammin.d",        0,    0x38668000,0xffff8000,DF_G_3R)
INST(ammin_db_w,    "ammin_db.w",     0,    0x386f0000,0xffff8000,DF_G_3R)
INST(ammin_db_d,    "ammin_db.d",     0,    0x386f8000,0xffff8000,DF_G_3R)
INST(ammax_wu,      "ammax.wu",       0,    0x38670000,0xffff8000,DF_G_3R)
INST(ammax_du,      "ammax.du",       0,    0x38678000,0xffff8000,DF_G_3R)
INST(ammax_db_wu,   "ammax_db.wu",    0,    0x38700000,0xffff8000,DF_G_3R)
INST(ammax_db_du,   "ammax_db.du",    0,    0x38708000,0xffff8000,DF_G_3R)
INST(ammin_wu,      "ammin.wu",       0,    0x38680000,0xffff8000,DF_G_3R)
INST(ammin_du,      "ammin.du",       0,    0x38688000,0xffff8000,DF_G_3R)
INST(ammin_db_wu,   "ammin_db.wu",    0,    0x38710000,0xffff8000,DF_G_3R)
INST(ammin_db_du,   "ammin_db.du",    0,    0x38718000,0xffff8000,DF_G_3R)

////R_R_R_I.
INST(alsl_w,        "alsl.w",         0,    0x00040000,0xfffe0000,DF_G_3R2IU) // 2i+1
INST(alsl_wu,       "alsl.wu",        0,    0x00060000,0xfffe0000,DF_G_3R2IU)
INST(alsl_d,        "alsl.d",         0,    0x002c0000,0xfffe0000,DF_G_3R2IU)

INST(bytepick_w,    "bytepick.w",     0,    0x00080000,0xfffe0000,DF_G_3RX) // rd,rj,rk,sa2
INST(bytepick_d,    "bytepick.d",     0,    0x000c0000,0xfffc0000,DF_G_3RX) // rd,rj,rk,sa3

INST(fsel,          "fsel",           0,    0x0d000000,0xfffc0000,DF_F_3RX3)

////R_I.
// si20
INST(lu12i_w,       "lu12i.w",        0,    0x14000000,0xfe000000,DF_G_R20I)
INST(lu32i_d,       "lu32i.d",        0,    0x16000000,0xfe000000,DF_G_R20I)
INST(pcaddi,        "pcaddi",         0,    0x18000000,0xfe000000,DF_G_R20I) // si20<<2
INST(pcaddu12i,     "pcaddu12i",      0,    0x1c000000,0xfe000000,DF_G_R20I) // si20<<12
INST(pcalau12i,     "pcalau12i",      0,    0x1a000000,0xfe000000,DF_G_R20I) // si20<<12
INST(pcaddu18i,     "pcaddu18i",      0,    0x1e000000,0xfe000000,DF_G_R20I) // si20<<18

////R_R.
INST(ext_w_b,       "ext.w.b",        0,    0x00005c00,0xfffffc00,DF_G_2R)
INST(ext_w_h,       "ext.w.h",        0,    0x00005800,0xfffffc00,DF_G_2R)
INST(clo_w,         "clo.w",          0,    0x00001000,0xfffffc00,DF_G_2R)
INST(clz_w,         "clz.w",          0,    0x00001400,0xfffffc00,DF_G_2R)
INST(cto_w,         "cto.w",          0,    0x00001800,0xfffffc00,DF_G_2R)
INST(ctz_w,         "ctz.w",          0,    0x00001c00,0xfffffc00,DF_G_2R)
INST(clo_d,         "clo.d",          0,    0x00002000,0xfffffc00,DF_G_2R)
INST(clz_d,         "clz.d",          0,    0x00002400,0xfffffc00,DF_G_2R)
INST(cto_d,         "cto.d",          0,    0x00002800,0xfffffc00,DF_G_2R)
INST(ctz_d,         "ctz.d",          0,    0x00002c00,0xfffffc00,DF_G_2R)
INST(revb_2h,       "revb.2h",        0,    0x00003000,0xfffffc00,DF_G_2R)
INST(revb_4h,       "revb.4h",        0,    0x00003400,0xfffffc00,DF_G_2R)
INST(revb_2w,       "revb.2w",        0,    0x00003800,0xfffffc00,DF_G_2R)
INST(revb_d,        "revb.d",         0,    0x00003c00,0xfffffc00,DF_G_2R)
INST(revh_2w,       "revh.2w",        0,    0x00004000,0xfffffc00,DF_G_2R)
INST(revh_d,        "revh.d",         0,    0x00004400,0xfffffc00,DF_G_2R)
INST(bitrev_4b,     "bitrev.4b",      0,    0x00004800,0xfffffc00,DF_G_2R)
INST(bitrev_8b,     "bitrev.8b",      0,    0x00004c00,0xfffffc00,DF_G_2R)
INST(bitrev_w,      "bitrev.w",       0,    0x00005000,0xfffffc00,DF_G_2R)
INST(bitrev_d,      "bitrev.d",       0,    0x00005400,0xfffffc00,DF_G_2R)
INST(rdtimel_w,     "rdtimel.w",      0,    0x00006000,0xfffffc00,DF_G_2R)
INST(rdtimeh_w,     "rdtimeh.w",      0,    0x00006400,0xfffffc00,DF_G_2R)
INST(rdtime_d,      "rdtime.d",       0,    0x00006800,0xfffffc00,DF_G_2R)
INST(cpucfg,        "cpucfg",         0,    0x00006c00,0xfffffc00,DF_G_2R)

////R_R, fr---->gr
INST(movfr2gr_s,    "movfr2gr.s",     0,    0x0114b400,0xfffffc00,DF_F_GR) // rd,fj
INST(movfr2gr_d,    "movfr2gr.d",     0,    0x0114b800,0xfffffc00,DF_F_GR)
INST(movfrh2gr_s,   "movfrh2gr.s",    0,    0x0114bc00,0xfffffc00,DF_F_GR)

////R_R_I_I.
INST(bstrins_w,     "bstrins.w",      0,    0x00600000,0xffe08000,DF_G_2R5IW) // rd,rj,msbw,lsbw
INST(bstrins_d,     "bstrins.d",      0,    0x00800000,0xffc00000,DF_G_2R6ID) // rd,rj,msbd,lsbd
INST(bstrpick_w,    "bstrpick.w",     0,    0x00608000,0xffe08000,DF_G_2R5IW) // rd,rj,msbw,lsbw
INST(bstrpick_d,    "bstrpick.d",     0,    0x00c00000,0xffc00000,DF_G_2R6ID) // rd,rj,msbd,lsbd

////Load.
INST(ld_b,          "ld.b",           LD,   0x28000000,0xffc00000,DF_G_2R12I) // si12
INST(ld_h,          "ld.h",           LD,   0x28400000,0xffc00000,DF_G_2R12I)
INST(ld_w,          "ld.w",           LD,   0x28800000,0xffc00000,DF_G_2R12I)
INST(ld_d,          "ld.d",           LD,   0x28c00000,0xffc00000,DF_G_2R12I)
INST(ld_bu,         "ld.bu",          LD,   0x2a000000,0xffc00000,DF_G_2R12I)
INST(ld_hu,         "ld.hu",          LD,   0x2a400000,0xffc00000,DF_G_2R12I)
INST(ld_wu,         "ld.wu",          LD,   0x2a800000,0xffc00000,DF_G_2R12I)
// si14
INST(ldptr_w,       "ldptr.w",        LD,   0x24000000,0xff000000,DF_G_2R14I) // si14
INST(ldptr_d,       "ldptr.d",        LD,   0x26000000,0xff000000,DF_G_2R14I)
INST(ll_w,          "ll.w",           0,    0x20000000,0xff000000,DF_G_2R14I)
INST(ll_d,          "ll.d",           0,    0x22000000,0xff000000,DF_G_2R14I)

INST(ldx_b,         "ldx.b",          LD,   0x38000000,0xffff8000,DF_G_3R)
INST(ldx_h,         "ldx.h",          LD,   0x38040000,0xffff8000,DF_G_3R)
INST(ldx_w,         "ldx.w",          LD,   0x38080000,0xffff8000,DF_G_3R)
INST(ldx_d,         "ldx.d",          LD,   0x380c0000,0xffff8000,DF_G_3R)
INST(ldx_bu,        "ldx.bu",         LD,   0x38200000,0xffff8000,DF_G_3R)
INST(ldx_hu,        "ldx.hu",         LD,   0x38240000,0xffff8000,DF_G_3R)
INST(ldx_wu,        "ldx.wu",         LD,   0x38280000,0xffff8000,DF_G_3R)
INST(ldgt_b,        "ldgt.b",         0,    0x38780000,0xffff8000,DF_G_3R)
INST(ldgt_h,        "ldgt.h",         0,    0x38788000,0xffff8000,DF_G_3R)
INST(ldgt_w,        "ldgt.w",         0,    0x38790000,0xffff8000,DF_G_3R)
INST(ldgt_d,        "ldgt.d",         0,    0x38798000,0xffff8000,DF_G_3R)
INST(ldle_b,        "ldle.b",         0,    0x387a0000,0xffff8000,DF_G_3R)
INST(ldle_h,        "ldle.h",         0,    0x387a8000,0xffff8000,DF_G_3R)
INST(ldle_w,        "ldle.w",         0,    0x387b0000,0xffff8000,DF_G_3R)
INST(ldle_d,        "ldle.d",         0,    0x387b8000,0xffff8000,DF_G_3R)

////R_R_I.
// si12
INST(addi_w,        "addi.w",         0,    0x02800000,0xffc00000,DF_G_2R12I)
INST(addi_d,        "addi.d",         0,    0x02c00000,0xffc00000,DF_G_2R12I)
INST(lu52i_d,       "lu52i.d",        0,    0x03000000,0xffc00000,DF_G_2R12I)
INST(slti,          "slti",           0,    0x02000000,0xffc00000,DF_G_2R12I)
INST(sltui,         "sltui",          0,    0x02400000,0xffc00000,DF_G_2R12I)
// ui12
INST(andi,          "andi",           0,    0x03400000,0xffc00000,DF_G_2R12IU)
INST(ori,           "ori",            0,    0x03800000,0xffc00000,DF_G_2R12IU)
INST(xori,          "xori",           0,    0x03c00000,0xffc00000,DF_G_2R12IU)
// ui5
INST(slli_w,        "slli.w",         0,    0x00408000,0xffff8000,DF_G_2R5IU)
INST(srli_w,        "srli.w",         0,    0x00448000,0xffff8000,DF_G_2R5IU)
INST(srai_w,        "srai.w",         0,    0x00488000,0xffff8000,DF_G_2R5IU)
INST(rotri_w,       "rotri.w",        0,    0x004c8000,0xffff8000,DF_G_2R5IU)
// ui6
INST(slli_d,        "slli.d",         0,    0x00410000,0xffff0000,DF_G_2R6IU)
INST(srli_d,        "srli.d",         0,    0x00450000,0xffff0000,DF_G_2R6IU)
INST(srai_d,        "srai.d",         0,    0x00490000,0xffff0000,DF_G_2R6IU)
INST(rotri_d,       "rotri.d",        0,    0x004d0000,0xffff0000,DF_G_2R6IU)
// si16
INST(addu16i_d,     "addu16i.d",      0,    0x10000000,0xfc000000,DF_G_2R16I) // si16<<16
INST(jirl,          "jirl",           0,    0x4c000000,0xfc000000,DF_G_B2)    // rj,rd,offs16<<2
////////////////////////////////////////////////////////////////////////////////////////////
////NOTE: jirl must be the last one !!! more info to see emitter::emitInsMayWriteToGCReg().
//
////NOTE:  End
////     the above instructions will be used by emitter::emitInsMayWriteToGCReg().
////////////////////////////////////////////////////////////////////////////////////////////

////Store.
INST(st_b,          "st.b",           ST,   0x29000000,0xffc00000,DF_G_2R12I) // si12
INST(st_h,          "st.h",           ST,   0x29400000,0xffc00000,DF_G_2R12I)
INST(st_w,          "st.w",           ST,   0x29800000,0xffc00000,DF_G_2R12I)
INST(st_d,          "st.d",           ST,   0x29c00000,0xffc00000,DF_G_2R12I)

INST(stptr_w,       "stptr.w",        ST,   0x25000000,0xff000000,DF_G_2R14I) // si14
INST(stptr_d,       "stptr.d",        ST,   0x27000000,0xff000000,DF_G_2R14I)
INST(sc_w,          "sc.w",           0,    0x21000000,0xff000000,DF_G_2R14I)
INST(sc_d,          "sc.d",           0,    0x23000000,0xff000000,DF_G_2R14I)

INST(stx_b,         "stx.b",          ST,   0x38100000,0xffff8000,DF_G_3R)
INST(stx_h,         "stx.h",          ST,   0x38140000,0xffff8000,DF_G_3R)
INST(stx_w,         "stx.w",          ST,   0x38180000,0xffff8000,DF_G_3R)
INST(stx_d,         "stx.d",          ST,   0x381c0000,0xffff8000,DF_G_3R)
INST(stgt_b,        "stgt.b",         0,    0x387c0000,0xffff8000,DF_G_3R)
INST(stgt_h,        "stgt.h",         0,    0x387c8000,0xffff8000,DF_G_3R)
INST(stgt_w,        "stgt.w",         0,    0x387d0000,0xffff8000,DF_G_3R)
INST(stgt_d,        "stgt.d",         0,    0x387d8000,0xffff8000,DF_G_3R)
INST(stle_b,        "stle.b",         0,    0x387e0000,0xffff8000,DF_G_3R)
INST(stle_h,        "stle.h",         0,    0x387e8000,0xffff8000,DF_G_3R)
INST(stle_w,        "stle.w",         0,    0x387f0000,0xffff8000,DF_G_3R)
INST(stle_d,        "stle.d",         0,    0x387f8000,0xffff8000,DF_G_3R)

INST(dbar,          "dbar",           0,    0x38720000,0xffff8000,DF_G_15I) // hint15
INST(ibar,          "ibar",           0,    0x38728000,0xffff8000,DF_G_15I)
INST(syscall,       "syscall",        0,    0x002b0000,0xffff8000,DF_G_15I)
INST(break,         "break",          0,    0x002a0000,0xffff8000,DF_G_15I)

// INST(asrtle_d,      "asrtle.d",       0,    0x00010000,0xffff801f,DF_G_2RK)
// INST(asrtgt_d,      "asrtgt.d",       0,    0x00018000,0xffff801f,DF_G_2RK)

INST(preld,         "preld",          LD,   0x2ac00000,0xffc00000,DF_G_2R12I) // hint5,rj,si12
INST(preldx,        "preldx",         LD,   0x382c0000,0xffff8000,DF_G_3R)   // hint5,rj,rk

////Float instructions.
////R_R_R.
INST(fadd_s,        "fadd.s",         0,    0x01008000,0xffff8000,DF_F_3R)
INST(fadd_d,        "fadd.d",         0,    0x01010000,0xffff8000,DF_F_3R)
INST(fsub_s,        "fsub.s",         0,    0x01028000,0xffff8000,DF_F_3R)
INST(fsub_d,        "fsub.d",         0,    0x01030000,0xffff8000,DF_F_3R)
INST(fmul_s,        "fmul.s",         0,    0x01048000,0xffff8000,DF_F_3R)
INST(fmul_d,        "fmul.d",         0,    0x01050000,0xffff8000,DF_F_3R)
INST(fdiv_s,        "fdiv.s",         0,    0x01068000,0xffff8000,DF_F_3R)
INST(fdiv_d,        "fdiv.d",         0,    0x01070000,0xffff8000,DF_F_3R)
INST(fmax_s,        "fmax.s",         0,    0x01088000,0xffff8000,DF_F_3R)
INST(fmax_d,        "fmax.d",         0,    0x01090000,0xffff8000,DF_F_3R)
INST(fmin_s,        "fmin.s",         0,    0x010a8000,0xffff8000,DF_F_3R)
INST(fmin_d,        "fmin.d",         0,    0x010b0000,0xffff8000,DF_F_3R)
INST(fmaxa_s,       "fmaxa.s",        0,    0x010c8000,0xffff8000,DF_F_3R)
INST(fmaxa_d,       "fmaxa.d",        0,    0x010d0000,0xffff8000,DF_F_3R)
INST(fmina_s,       "fmina.s",        0,    0x010e8000,0xffff8000,DF_F_3R)
INST(fmina_d,       "fmina.d",        0,    0x010f0000,0xffff8000,DF_F_3R)
INST(fscaleb_s,     "fscaleb.s",      0,    0x01108000,0xffff8000,DF_F_3R)
INST(fscaleb_d,     "fscaleb.d",      0,    0x01110000,0xffff8000,DF_F_3R)
INST(fcopysign_s,   "fcopysign.s",    0,    0x01128000,0xffff8000,DF_F_3R)
INST(fcopysign_d,   "fcopysign.d",    0,    0x01130000,0xffff8000,DF_F_3R)

INST(fldx_s,        "fldx.s",         LD,   0x38300000,0xffff8000,DF_F_R2G) // fd,rj,rk
INST(fldx_d,        "fldx.d",         LD,   0x38340000,0xffff8000,DF_F_R2G)
INST(fstx_s,        "fstx.s",         ST,   0x38380000,0xffff8000,DF_F_R2G)
INST(fstx_d,        "fstx.d",         ST,   0x383c0000,0xffff8000,DF_F_R2G)
INST(fldgt_s,       "fldgt.s",        0,    0x38740000,0xffff8000,DF_F_R2G)
INST(fldgt_d,       "fldgt.d",        0,    0x38748000,0xffff8000,DF_F_R2G)
INST(fldle_s,       "fldle.s",        0,    0x38750000,0xffff8000,DF_F_R2G)
INST(fldle_d,       "fldle.d",        0,    0x38758000,0xffff8000,DF_F_R2G)
INST(fstgt_s,       "fstgt.s",        0,    0x38760000,0xffff8000,DF_F_R2G)
INST(fstgt_d,       "fstgt.d",        0,    0x38768000,0xffff8000,DF_F_R2G)
INST(fstle_s,       "fstle.s",        0,    0x38770000,0xffff8000,DF_F_R2G)
INST(fstle_d,       "fstle.d",        0,    0x38778000,0xffff8000,DF_F_R2G)

////R_R_R_R.
INST(fmadd_s,       "fmadd.s",        0,    0x08100000,0xfff00000,DF_F_4R)
INST(fmadd_d,       "fmadd.d",        0,    0x08200000,0xfff00000,DF_F_4R)
INST(fmsub_s,       "fmsub.s",        0,    0x08500000,0xfff00000,DF_F_4R)
INST(fmsub_d,       "fmsub.d",        0,    0x08600000,0xfff00000,DF_F_4R)
INST(fnmadd_s,      "fnmadd.s",       0,    0x08900000,0xfff00000,DF_F_4R)
INST(fnmadd_d,      "fnmadd.d",       0,    0x08a00000,0xfff00000,DF_F_4R)
INST(fnmsub_s,      "fnmsub.s",       0,    0x08d00000,0xfff00000,DF_F_4R)
INST(fnmsub_d,      "fnmsub.d",       0,    0x08e00000,0xfff00000,DF_F_4R)

////R_R.
INST(fabs_s,        "fabs.s",         0,    0x01140400,0xfffffc00,DF_F_2R)
INST(fabs_d,        "fabs.d",         0,    0x01140800,0xfffffc00,DF_F_2R)
INST(fneg_s,        "fneg.s",         0,    0x01141400,0xfffffc00,DF_F_2R)
INST(fneg_d,        "fneg.d",         0,    0x01141800,0xfffffc00,DF_F_2R)
INST(fsqrt_s,       "fsqrt.s",        0,    0x01144400,0xfffffc00,DF_F_2R)
INST(fsqrt_d,       "fsqrt.d",        0,    0x01144800,0xfffffc00,DF_F_2R)
INST(frsqrt_s,      "frsqrt.s",       0,    0x01146400,0xfffffc00,DF_F_2R)
INST(frsqrt_d,      "frsqrt.d",       0,    0x01146800,0xfffffc00,DF_F_2R)
INST(frsqrte_s,     "frsqrte.s",      0,    0x01148400,0xfffffc00,DF_F_2R)
INST(frsqrte_d,     "frsqrte.d",      0,    0x01148800,0xfffffc00,DF_F_2R)
INST(frecip_s,      "frecip.s",       0,    0x01145400,0xfffffc00,DF_F_2R)
INST(frecip_d,      "frecip.d",       0,    0x01145800,0xfffffc00,DF_F_2R)
INST(frecipe_s,     "frecipe.s",      0,    0x01147400,0xfffffc00,DF_F_2R)
INST(frecipe_d,     "frecipe.d",      0,    0x01147800,0xfffffc00,DF_F_2R)
INST(flogb_s,       "flogb.s",        0,    0x01142400,0xfffffc00,DF_F_2R)
INST(flogb_d,       "flogb.d",        0,    0x01142800,0xfffffc00,DF_F_2R)
INST(fclass_s,      "fclass.s",       0,    0x01143400,0xfffffc00,DF_F_2R)
INST(fclass_d,      "fclass.d",       0,    0x01143800,0xfffffc00,DF_F_2R)
INST(fcvt_s_d,      "fcvt.s.d",       0,    0x01191800,0xfffffc00,DF_F_2R)
INST(fcvt_d_s,      "fcvt.d.s",       0,    0x01192400,0xfffffc00,DF_F_2R)
INST(ffint_s_w,     "ffint.s.w",      0,    0x011d1000,0xfffffc00,DF_F_2R)
INST(ffint_s_l,     "ffint.s.l",      0,    0x011d1800,0xfffffc00,DF_F_2R)
INST(ffint_d_w,     "ffint.d.w",      0,    0x011d2000,0xfffffc00,DF_F_2R)
INST(ffint_d_l,     "ffint.d.l",      0,    0x011d2800,0xfffffc00,DF_F_2R)
INST(ftint_w_s,     "ftint.w.s",      0,    0x011b0400,0xfffffc00,DF_F_2R)
INST(ftint_w_d,     "ftint.w.d",      0,    0x011b0800,0xfffffc00,DF_F_2R)
INST(ftint_l_s,     "ftint.l.s",      0,    0x011b2400,0xfffffc00,DF_F_2R)
INST(ftint_l_d,     "ftint.l.d",      0,    0x011b2800,0xfffffc00,DF_F_2R)
INST(ftintrm_w_s,   "ftintrm.w.s",    0,    0x011a0400,0xfffffc00,DF_F_2R)
INST(ftintrm_w_d,   "ftintrm.w.d",    0,    0x011a0800,0xfffffc00,DF_F_2R)
INST(ftintrm_l_s,   "ftintrm.l.s",    0,    0x011a2400,0xfffffc00,DF_F_2R)
INST(ftintrm_l_d,   "ftintrm.l.d",    0,    0x011a2800,0xfffffc00,DF_F_2R)
INST(ftintrp_w_s,   "ftintrp.w.s",    0,    0x011a4400,0xfffffc00,DF_F_2R)
INST(ftintrp_w_d,   "ftintrp.w.d",    0,    0x011a4800,0xfffffc00,DF_F_2R)
INST(ftintrp_l_s,   "ftintrp.l.s",    0,    0x011a6400,0xfffffc00,DF_F_2R)
INST(ftintrp_l_d,   "ftintrp.l.d",    0,    0x011a6800,0xfffffc00,DF_F_2R)
INST(ftintrz_w_s,   "ftintrz.w.s",    0,    0x011a8400,0xfffffc00,DF_F_2R)
INST(ftintrz_w_d,   "ftintrz.w.d",    0,    0x011a8800,0xfffffc00,DF_F_2R)
INST(ftintrz_l_s,   "ftintrz.l.s",    0,    0x011aa400,0xfffffc00,DF_F_2R)
INST(ftintrz_l_d,   "ftintrz.l.d",    0,    0x011aa800,0xfffffc00,DF_F_2R)
INST(ftintrne_w_s,  "ftintrne.w.s",   0,    0x011ac400,0xfffffc00,DF_F_2R)
INST(ftintrne_w_d,  "ftintrne.w.d",   0,    0x011ac800,0xfffffc00,DF_F_2R)
INST(ftintrne_l_s,  "ftintrne.l.s",   0,    0x011ae400,0xfffffc00,DF_F_2R)
INST(ftintrne_l_d,  "ftintrne.l.d",   0,    0x011ae800,0xfffffc00,DF_F_2R)
INST(frint_s,       "frint.s",        0,    0x011e4400,0xfffffc00,DF_F_2R)
INST(frint_d,       "frint.d",        0,    0x011e4800,0xfffffc00,DF_F_2R)
INST(fmov_s,        "fmov.s",         0,    0x01149400,0xfffffc00,DF_F_2R)
INST(fmov_d,        "fmov.d",         0,    0x01149800,0xfffffc00,DF_F_2R)

INST(movgr2fr_w,    "movgr2fr.w",     0,    0x0114a400,0xfffffc00,DF_F_RG) // fd,rj
INST(movgr2fr_d,    "movgr2fr.d",     0,    0x0114a800,0xfffffc00,DF_F_RG)
INST(movgr2frh_w,   "movgr2frh.w",    0,    0x0114ac00,0xfffffc00,DF_F_RG)
INST(movgr2fcsr,    "movgr2fcsr",     0,    0x0114c000,0xfffffc00,DF_F_FG) // fcsr,rj
INST(movfcsr2gr,    "movfcsr2gr",     0,    0x0114c800,0xfffffc00,DF_F_GF) // rd,fcsr
INST(movfr2cf,      "movfr2cf",       0,    0x0114d000,0xfffffc18,DF_F_CR) // cd,fj
INST(movcf2fr,      "movcf2fr",       0,    0x0114d400,0xffffff00,DF_F_RC) // fd,cj
INST(movgr2cf,      "movgr2cf",       0,    0x0114d800,0xfffffc18,DF_F_CG) // cd,rj
INST(movcf2gr,      "movcf2gr",       0,    0x0114dc00,0xffffff00,DF_F_GC) // rd,cj

////R_R_I.
INST(fcmp_caf_s,    "fcmp.caf.s",     0,    0x0c100000,0xffff8018,DF_F_C2R) // cd,fj,fk
INST(fcmp_cun_s,    "fcmp.cun.s",     0,    0x0c140000,0xffff8018,DF_F_C2R)
INST(fcmp_ceq_s,    "fcmp.ceq.s",     0,    0x0c120000,0xffff8018,DF_F_C2R)
INST(fcmp_cueq_s,   "fcmp.cueq.s",    0,    0x0c160000,0xffff8018,DF_F_C2R)
INST(fcmp_clt_s,    "fcmp.clt.s",     0,    0x0c110000,0xffff8018,DF_F_C2R)
INST(fcmp_cult_s,   "fcmp.cult.s",    0,    0x0c150000,0xffff8018,DF_F_C2R)
INST(fcmp_cle_s,    "fcmp.cle.s",     0,    0x0c130000,0xffff8018,DF_F_C2R)
INST(fcmp_cule_s,   "fcmp.cule.s",    0,    0x0c170000,0xffff8018,DF_F_C2R)
INST(fcmp_cne_s,    "fcmp.cne.s",     0,    0x0c180000,0xffff8018,DF_F_C2R)
INST(fcmp_cor_s,    "fcmp.cor.s",     0,    0x0c1a0000,0xffff8018,DF_F_C2R)
INST(fcmp_cune_s,   "fcmp.cune.s",    0,    0x0c1c0000,0xffff8018,DF_F_C2R)
INST(fcmp_saf_d,    "fcmp.saf.d",     0,    0x0c208000,0xffff8018,DF_F_C2R)
INST(fcmp_sun_d,    "fcmp.sun.d",     0,    0x0c248000,0xffff8018,DF_F_C2R)
INST(fcmp_seq_d,    "fcmp.seq.d",     0,    0x0c228000,0xffff8018,DF_F_C2R)
INST(fcmp_sueq_d,   "fcmp.sueq.d",    0,    0x0c268000,0xffff8018,DF_F_C2R)
INST(fcmp_slt_d,    "fcmp.slt.d",     0,    0x0c218000,0xffff8018,DF_F_C2R)
INST(fcmp_sult_d,   "fcmp.sult.d",    0,    0x0c258000,0xffff8018,DF_F_C2R)
INST(fcmp_sle_d,    "fcmp.sle.d",     0,    0x0c238000,0xffff8018,DF_F_C2R)
INST(fcmp_sule_d,   "fcmp.sule.d",    0,    0x0c278000,0xffff8018,DF_F_C2R)
INST(fcmp_sne_d,    "fcmp.sne.d",     0,    0x0c288000,0xffff8018,DF_F_C2R)
INST(fcmp_sor_d,    "fcmp.sor.d",     0,    0x0c2a8000,0xffff8018,DF_F_C2R)
INST(fcmp_sune_d,   "fcmp.sune.d",    0,    0x0c2c8000,0xffff8018,DF_F_C2R)
INST(fcmp_caf_d,    "fcmp.caf.d",     0,    0x0c200000,0xffff8018,DF_F_C2R)
INST(fcmp_cun_d,    "fcmp.cun.d",     0,    0x0c240000,0xffff8018,DF_F_C2R)
INST(fcmp_ceq_d,    "fcmp.ceq.d",     0,    0x0c220000,0xffff8018,DF_F_C2R)
INST(fcmp_cueq_d,   "fcmp.cueq.d",    0,    0x0c260000,0xffff8018,DF_F_C2R)
INST(fcmp_clt_d,    "fcmp.clt.d",     0,    0x0c210000,0xffff8018,DF_F_C2R)
INST(fcmp_cult_d,   "fcmp.cult.d",    0,    0x0c250000,0xffff8018,DF_F_C2R)
INST(fcmp_cle_d,    "fcmp.cle.d",     0,    0x0c230000,0xffff8018,DF_F_C2R)
INST(fcmp_cule_d,   "fcmp.cule.d",    0,    0x0c270000,0xffff8018,DF_F_C2R)
INST(fcmp_cne_d,    "fcmp.cne.d",     0,    0x0c280000,0xffff8018,DF_F_C2R)
INST(fcmp_cor_d,    "fcmp.cor.d",     0,    0x0c2a0000,0xffff8018,DF_F_C2R)
INST(fcmp_cune_d,   "fcmp.cune.d",    0,    0x0c2c0000,0xffff8018,DF_F_C2R)
INST(fcmp_saf_s,    "fcmp.saf.s",     0,    0x0c108000,0xffff8018,DF_F_C2R)
INST(fcmp_sun_s,    "fcmp.sun.s",     0,    0x0c148000,0xffff8018,DF_F_C2R)
INST(fcmp_seq_s,    "fcmp.seq.s",     0,    0x0c128000,0xffff8018,DF_F_C2R)
INST(fcmp_sueq_s,   "fcmp.sueq.s",    0,    0x0c168000,0xffff8018,DF_F_C2R)
INST(fcmp_slt_s,    "fcmp.slt.s",     0,    0x0c118000,0xffff8018,DF_F_C2R)
INST(fcmp_sult_s,   "fcmp.sult.s",    0,    0x0c158000,0xffff8018,DF_F_C2R)
INST(fcmp_sle_s,    "fcmp.sle.s",     0,    0x0c138000,0xffff8018,DF_F_C2R)
INST(fcmp_sule_s,   "fcmp.sule.s",    0,    0x0c178000,0xffff8018,DF_F_C2R)
INST(fcmp_sne_s,    "fcmp.sne.s",     0,    0x0c188000,0xffff8018,DF_F_C2R)
INST(fcmp_sor_s,    "fcmp.sor.s",     0,    0x0c1a8000,0xffff8018,DF_F_C2R)
INST(fcmp_sune_s,   "fcmp.sune.s",    0,    0x0c1c8000,0xffff8018,DF_F_C2R)

////R_R_I.
INST(fld_s,         "fld.s",          LD,   0x2b000000,0xffc00000,DF_F_RG12I) // fd,rj,si12
INST(fld_d,         "fld.d",          LD,   0x2b800000,0xffc00000,DF_F_RG12I)
INST(fst_s,         "fst.s",          ST,   0x2b400000,0xffc00000,DF_F_RG12I)
INST(fst_d,         "fst.d",          ST,   0x2bc00000,0xffc00000,DF_F_RG12I)

#ifdef FEATURE_SIMD
// ---- PartI-LSX:128bits-SIMD
// R_R_R_R.
// vd, vj, vk, va
INST(vfmadd_s,          "vfmadd.s",         0,  0x09100000,0xfff00000,DF_S_4R)
INST(vfmadd_d,          "vfmadd.d",         0,  0x09200000,0xfff00000,DF_S_4R)
INST(vfmsub_s,          "vfmsub.s",         0,  0x09500000,0xfff00000,DF_S_4R)
INST(vfmsub_d,          "vfmsub.d",         0,  0x09600000,0xfff00000,DF_S_4R)
INST(vfnmadd_s,         "vfnmadd.s",        0,  0x09900000,0xfff00000,DF_S_4R)
INST(vfnmadd_d,         "vfnmadd.d",        0,  0x09a00000,0xfff00000,DF_S_4R)
INST(vfnmsub_s,         "vfnmsub.s",        0,  0x09d00000,0xfff00000,DF_S_4R)
INST(vfnmsub_d,         "vfnmsub.d",        0,  0x09e00000,0xfff00000,DF_S_4R)
INST(vbitsel_v,         "vbitsel.v",        0,  0x0d100000,0xfff00000,DF_S_4R)
INST(vshuf_b,           "vshuf.b",          0,  0x0d500000,0xfff00000,DF_S_4R)

// R_R_R.
// vd, rj, rk
INST(vldx,              "vldx",             LD, 0x38400000,0xffff8000,DF_S_3R)
INST(vstx,              "vstx",             ST, 0x38440000,0xffff8000,DF_S_3R)

// vd, vj, rk
INST(vreplve_b,         "vreplve.b",        0,  0x71220000,0xffff8000,DF_S_2RG)
INST(vreplve_h,         "vreplve.h",        0,  0x71228000,0xffff8000,DF_S_2RG)
INST(vreplve_w,         "vreplve.w",        0,  0x71230000,0xffff8000,DF_S_2RG)
INST(vreplve_d,         "vreplve.d",        0,  0x71238000,0xffff8000,DF_S_2RG)

// vd, vj, vk
INST(vfcmp_caf_s,       "vfcmp.caf.s",      0,  0x0c500000,0xffff8000,DF_S_3R)
INST(vfcmp_cun_s,       "vfcmp.cun.s",      0,  0x0c540000,0xffff8000,DF_S_3R)
INST(vfcmp_ceq_s,       "vfcmp.ceq.s",      0,  0x0c520000,0xffff8000,DF_S_3R)
INST(vfcmp_cueq_s,      "vfcmp.cueq.s",     0,  0x0c560000,0xffff8000,DF_S_3R)
INST(vfcmp_clt_s,       "vfcmp.clt.s",      0,  0x0c510000,0xffff8000,DF_S_3R)
INST(vfcmp_cult_s,      "vfcmp.cult.s",     0,  0x0c550000,0xffff8000,DF_S_3R)
INST(vfcmp_cle_s,       "vfcmp.cle.s",      0,  0x0c530000,0xffff8000,DF_S_3R)
INST(vfcmp_cule_s,      "vfcmp.cule.s",     0,  0x0c570000,0xffff8000,DF_S_3R)
INST(vfcmp_cne_s,       "vfcmp.cne.s",      0,  0x0c580000,0xffff8000,DF_S_3R)
INST(vfcmp_cor_s,       "vfcmp.cor.s",      0,  0x0c5a0000,0xffff8000,DF_S_3R)
INST(vfcmp_cune_s,      "vfcmp.cune.s",     0,  0x0c5c0000,0xffff8000,DF_S_3R)
INST(vfcmp_saf_s,       "vfcmp.saf.s",      0,  0x0c508000,0xffff8000,DF_S_3R)
INST(vfcmp_sun_s,       "vfcmp.sun.s",      0,  0x0c548000,0xffff8000,DF_S_3R)
INST(vfcmp_seq_s,       "vfcmp.seq.s",      0,  0x0c528000,0xffff8000,DF_S_3R)
INST(vfcmp_sueq_s,      "vfcmp.sueq.s",     0,  0x0c568000,0xffff8000,DF_S_3R)
INST(vfcmp_slt_s,       "vfcmp.slt.s",      0,  0x0c518000,0xffff8000,DF_S_3R)
INST(vfcmp_sult_s,      "vfcmp.sult.s",     0,  0x0c558000,0xffff8000,DF_S_3R)
INST(vfcmp_sle_s,       "vfcmp.sle.s",      0,  0x0c538000,0xffff8000,DF_S_3R)
INST(vfcmp_sule_s,      "vfcmp.sule.s",     0,  0x0c578000,0xffff8000,DF_S_3R)
INST(vfcmp_sne_s,       "vfcmp.sne.s",      0,  0x0c588000,0xffff8000,DF_S_3R)
INST(vfcmp_sor_s,       "vfcmp.sor.s",      0,  0x0c5a8000,0xffff8000,DF_S_3R)
INST(vfcmp_sune_s,      "vfcmp.sune.s",     0,  0x0c5c8000,0xffff8000,DF_S_3R)
INST(vfcmp_caf_d,       "vfcmp.caf.d",      0,  0x0c600000,0xffff8000,DF_S_3R)
INST(vfcmp_cun_d,       "vfcmp.cun.d",      0,  0x0c640000,0xffff8000,DF_S_3R)
INST(vfcmp_ceq_d,       "vfcmp.ceq.d",      0,  0x0c620000,0xffff8000,DF_S_3R)
INST(vfcmp_cueq_d,      "vfcmp.cueq.d",     0,  0x0c660000,0xffff8000,DF_S_3R)
INST(vfcmp_clt_d,       "vfcmp.clt.d",      0,  0x0c610000,0xffff8000,DF_S_3R)
INST(vfcmp_cult_d,      "vfcmp.cult.d",     0,  0x0c650000,0xffff8000,DF_S_3R)
INST(vfcmp_cle_d,       "vfcmp.cle.d",      0,  0x0c630000,0xffff8000,DF_S_3R)
INST(vfcmp_cule_d,      "vfcmp.cule.d",     0,  0x0c670000,0xffff8000,DF_S_3R)
INST(vfcmp_cne_d,       "vfcmp.cne.d",      0,  0x0c680000,0xffff8000,DF_S_3R)
INST(vfcmp_cor_d,       "vfcmp.cor.d",      0,  0x0c6a0000,0xffff8000,DF_S_3R)
INST(vfcmp_cune_d,      "vfcmp.cune.d",     0,  0x0c6c0000,0xffff8000,DF_S_3R)
INST(vfcmp_saf_d,       "vfcmp.saf.d",      0,  0x0c608000,0xffff8000,DF_S_3R)
INST(vfcmp_sun_d,       "vfcmp.sun.d",      0,  0x0c648000,0xffff8000,DF_S_3R)
INST(vfcmp_seq_d,       "vfcmp.seq.d",      0,  0x0c628000,0xffff8000,DF_S_3R)
INST(vfcmp_sueq_d,      "vfcmp.sueq.d",     0,  0x0c668000,0xffff8000,DF_S_3R)
INST(vfcmp_slt_d,       "vfcmp.slt.d",      0,  0x0c618000,0xffff8000,DF_S_3R)
INST(vfcmp_sult_d,      "vfcmp.sult.d",     0,  0x0c658000,0xffff8000,DF_S_3R)
INST(vfcmp_sle_d,       "vfcmp.sle.d",      0,  0x0c638000,0xffff8000,DF_S_3R)
INST(vfcmp_sule_d,      "vfcmp.sule.d",     0,  0x0c678000,0xffff8000,DF_S_3R)
INST(vfcmp_sne_d,       "vfcmp.sne.d",      0,  0x0c688000,0xffff8000,DF_S_3R)
INST(vfcmp_sor_d,       "vfcmp.sor.d",      0,  0x0c6a8000,0xffff8000,DF_S_3R)
INST(vfcmp_sune_d,      "vfcmp.sune.d",     0,  0x0c6c8000,0xffff8000,DF_S_3R)
INST(vseq_b,            "vseq.b",           0,  0x70000000,0xffff8000,DF_S_3R)
INST(vseq_h,            "vseq.h",           0,  0x70008000,0xffff8000,DF_S_3R)
INST(vseq_w,            "vseq.w",           0,  0x70010000,0xffff8000,DF_S_3R)
INST(vseq_d,            "vseq.d",           0,  0x70018000,0xffff8000,DF_S_3R)
INST(vsle_b,            "vsle.b",           0,  0x70020000,0xffff8000,DF_S_3R)
INST(vsle_h,            "vsle.h",           0,  0x70028000,0xffff8000,DF_S_3R)
INST(vsle_w,            "vsle.w",           0,  0x70030000,0xffff8000,DF_S_3R)
INST(vsle_d,            "vsle.d",           0,  0x70038000,0xffff8000,DF_S_3R)
INST(vsle_bu,           "vsle.bu",          0,  0x70040000,0xffff8000,DF_S_3R)
INST(vsle_hu,           "vsle.hu",          0,  0x70048000,0xffff8000,DF_S_3R)
INST(vsle_wu,           "vsle.wu",          0,  0x70050000,0xffff8000,DF_S_3R)
INST(vsle_du,           "vsle.du",          0,  0x70058000,0xffff8000,DF_S_3R)
INST(vslt_b,            "vslt.b",           0,  0x70060000,0xffff8000,DF_S_3R)
INST(vslt_h,            "vslt.h",           0,  0x70068000,0xffff8000,DF_S_3R)
INST(vslt_w,            "vslt.w",           0,  0x70070000,0xffff8000,DF_S_3R)
INST(vslt_d,            "vslt.d",           0,  0x70078000,0xffff8000,DF_S_3R)
INST(vslt_bu,           "vslt.bu",          0,  0x70080000,0xffff8000,DF_S_3R)
INST(vslt_hu,           "vslt.hu",          0,  0x70088000,0xffff8000,DF_S_3R)
INST(vslt_wu,           "vslt.wu",          0,  0x70090000,0xffff8000,DF_S_3R)
INST(vslt_du,           "vslt.du",          0,  0x70098000,0xffff8000,DF_S_3R)
INST(vadd_b,            "vadd.b",           0,  0x700a0000,0xffff8000,DF_S_3R)
INST(vadd_h,            "vadd.h",           0,  0x700a8000,0xffff8000,DF_S_3R)
INST(vadd_w,            "vadd.w",           0,  0x700b0000,0xffff8000,DF_S_3R)
INST(vadd_d,            "vadd.d",           0,  0x700b8000,0xffff8000,DF_S_3R)
INST(vsub_b,            "vsub.b",           0,  0x700c0000,0xffff8000,DF_S_3R)
INST(vsub_h,            "vsub.h",           0,  0x700c8000,0xffff8000,DF_S_3R)
INST(vsub_w,            "vsub.w",           0,  0x700d0000,0xffff8000,DF_S_3R)
INST(vsub_d,            "vsub.d",           0,  0x700d8000,0xffff8000,DF_S_3R)
INST(vaddwev_h_b,       "vaddwev.h.b",      0,  0x701e0000,0xffff8000,DF_S_3R)
INST(vaddwev_w_h,       "vaddwev.w.h",      0,  0x701e8000,0xffff8000,DF_S_3R)
INST(vaddwev_d_w,       "vaddwev.d.w",      0,  0x701f0000,0xffff8000,DF_S_3R)
INST(vaddwev_q_d,       "vaddwev.q.d",      0,  0x701f8000,0xffff8000,DF_S_3R)
INST(vsubwev_h_b,       "vsubwev.h.b",      0,  0x70200000,0xffff8000,DF_S_3R)
INST(vsubwev_w_h,       "vsubwev.w.h",      0,  0x70208000,0xffff8000,DF_S_3R)
INST(vsubwev_d_w,       "vsubwev.d.w",      0,  0x70210000,0xffff8000,DF_S_3R)
INST(vsubwev_q_d,       "vsubwev.q.d",      0,  0x70218000,0xffff8000,DF_S_3R)
INST(vaddwod_h_b,       "vaddwod.h.b",      0,  0x70220000,0xffff8000,DF_S_3R)
INST(vaddwod_w_h,       "vaddwod.w.h",      0,  0x70228000,0xffff8000,DF_S_3R)
INST(vaddwod_d_w,       "vaddwod.d.w",      0,  0x70230000,0xffff8000,DF_S_3R)
INST(vaddwod_q_d,       "vaddwod.q.d",      0,  0x70238000,0xffff8000,DF_S_3R)
INST(vsubwod_h_b,       "vsubwod.h.b",      0,  0x70240000,0xffff8000,DF_S_3R)
INST(vsubwod_w_h,       "vsubwod.w.h",      0,  0x70248000,0xffff8000,DF_S_3R)
INST(vsubwod_d_w,       "vsubwod.d.w",      0,  0x70250000,0xffff8000,DF_S_3R)
INST(vsubwod_q_d,       "vsubwod.q.d",      0,  0x70258000,0xffff8000,DF_S_3R)
INST(vaddwev_h_bu,      "vaddwev.h.bu",     0,  0x702e0000,0xffff8000,DF_S_3R)
INST(vaddwev_w_hu,      "vaddwev.w.hu",     0,  0x702e8000,0xffff8000,DF_S_3R)
INST(vaddwev_d_wu,      "vaddwev.d.wu",     0,  0x702f0000,0xffff8000,DF_S_3R)
INST(vaddwev_q_du,      "vaddwev.q.du",     0,  0x702f8000,0xffff8000,DF_S_3R)
INST(vsubwev_h_bu,      "vsubwev.h.bu",     0,  0x70300000,0xffff8000,DF_S_3R)
INST(vsubwev_w_hu,      "vsubwev.w.hu",     0,  0x70308000,0xffff8000,DF_S_3R)
INST(vsubwev_d_wu,      "vsubwev.d.wu",     0,  0x70310000,0xffff8000,DF_S_3R)
INST(vsubwev_q_du,      "vsubwev.q.du",     0,  0x70318000,0xffff8000,DF_S_3R)
INST(vaddwod_h_bu,      "vaddwod.h.bu",     0,  0x70320000,0xffff8000,DF_S_3R)
INST(vaddwod_w_hu,      "vaddwod.w.hu",     0,  0x70328000,0xffff8000,DF_S_3R)
INST(vaddwod_d_wu,      "vaddwod.d.wu",     0,  0x70330000,0xffff8000,DF_S_3R)
INST(vaddwod_q_du,      "vaddwod.q.du",     0,  0x70338000,0xffff8000,DF_S_3R)
INST(vsubwod_h_bu,      "vsubwod.h.bu",     0,  0x70340000,0xffff8000,DF_S_3R)
INST(vsubwod_w_hu,      "vsubwod.w.hu",     0,  0x70348000,0xffff8000,DF_S_3R)
INST(vsubwod_d_wu,      "vsubwod.d.wu",     0,  0x70350000,0xffff8000,DF_S_3R)
INST(vsubwod_q_du,      "vsubwod.q.du",     0,  0x70358000,0xffff8000,DF_S_3R)
INST(vaddwev_h_bu_b,    "vaddwev.h.bu.b",   0,  0x703e0000,0xffff8000,DF_S_3R)
INST(vaddwev_w_hu_h,    "vaddwev.w.hu.h",   0,  0x703e8000,0xffff8000,DF_S_3R)
INST(vaddwev_d_wu_w,    "vaddwev.d.wu.w",   0,  0x703f0000,0xffff8000,DF_S_3R)
INST(vaddwev_q_du_d,    "vaddwev.q.du.d",   0,  0x703f8000,0xffff8000,DF_S_3R)
INST(vaddwod_h_bu_b,    "vaddwod.h.bu.b",   0,  0x70400000,0xffff8000,DF_S_3R)
INST(vaddwod_w_hu_h,    "vaddwod.w.hu.h",   0,  0x70408000,0xffff8000,DF_S_3R)
INST(vaddwod_d_wu_w,    "vaddwod.d.wu.w",   0,  0x70410000,0xffff8000,DF_S_3R)
INST(vaddwod_q_du_d,    "vaddwod.q.du.d",   0,  0x70418000,0xffff8000,DF_S_3R)
INST(vsadd_b,           "vsadd.b",          0,  0x70460000,0xffff8000,DF_S_3R)
INST(vsadd_h,           "vsadd.h",          0,  0x70468000,0xffff8000,DF_S_3R)
INST(vsadd_w,           "vsadd.w",          0,  0x70470000,0xffff8000,DF_S_3R)
INST(vsadd_d,           "vsadd.d",          0,  0x70478000,0xffff8000,DF_S_3R)
INST(vssub_b,           "vssub.b",          0,  0x70480000,0xffff8000,DF_S_3R)
INST(vssub_h,           "vssub.h",          0,  0x70488000,0xffff8000,DF_S_3R)
INST(vssub_w,           "vssub.w",          0,  0x70490000,0xffff8000,DF_S_3R)
INST(vssub_d,           "vssub.d",          0,  0x70498000,0xffff8000,DF_S_3R)
INST(vsadd_bu,          "vsadd.bu",         0,  0x704a0000,0xffff8000,DF_S_3R)
INST(vsadd_hu,          "vsadd.hu",         0,  0x704a8000,0xffff8000,DF_S_3R)
INST(vsadd_wu,          "vsadd.wu",         0,  0x704b0000,0xffff8000,DF_S_3R)
INST(vsadd_du,          "vsadd.du",         0,  0x704b8000,0xffff8000,DF_S_3R)
INST(vssub_bu,          "vssub.bu",         0,  0x704c0000,0xffff8000,DF_S_3R)
INST(vssub_hu,          "vssub.hu",         0,  0x704c8000,0xffff8000,DF_S_3R)
INST(vssub_wu,          "vssub.wu",         0,  0x704d0000,0xffff8000,DF_S_3R)
INST(vssub_du,          "vssub.du",         0,  0x704d8000,0xffff8000,DF_S_3R)
INST(vhaddw_h_b,        "vhaddw.h.b",       0,  0x70540000,0xffff8000,DF_S_3R)
INST(vhaddw_w_h,        "vhaddw.w.h",       0,  0x70548000,0xffff8000,DF_S_3R)
INST(vhaddw_d_w,        "vhaddw.d.w",       0,  0x70550000,0xffff8000,DF_S_3R)
INST(vhaddw_q_d,        "vhaddw.q.d",       0,  0x70558000,0xffff8000,DF_S_3R)
INST(vhsubw_h_b,        "vhsubw.h.b",       0,  0x70560000,0xffff8000,DF_S_3R)
INST(vhsubw_w_h,        "vhsubw.w.h",       0,  0x70568000,0xffff8000,DF_S_3R)
INST(vhsubw_d_w,        "vhsubw.d.w",       0,  0x70570000,0xffff8000,DF_S_3R)
INST(vhsubw_q_d,        "vhsubw.q.d",       0,  0x70578000,0xffff8000,DF_S_3R)
INST(vhaddw_hu_bu,      "vhaddw.hu.bu",     0,  0x70580000,0xffff8000,DF_S_3R)
INST(vhaddw_wu_hu,      "vhaddw.wu.hu",     0,  0x70588000,0xffff8000,DF_S_3R)
INST(vhaddw_du_wu,      "vhaddw.du.wu",     0,  0x70590000,0xffff8000,DF_S_3R)
INST(vhaddw_qu_du,      "vhaddw.qu.du",     0,  0x70598000,0xffff8000,DF_S_3R)
INST(vhsubw_hu_bu,      "vhsubw.hu.bu",     0,  0x705a0000,0xffff8000,DF_S_3R)
INST(vhsubw_wu_hu,      "vhsubw.wu.hu",     0,  0x705a8000,0xffff8000,DF_S_3R)
INST(vhsubw_du_wu,      "vhsubw.du.wu",     0,  0x705b0000,0xffff8000,DF_S_3R)
INST(vhsubw_qu_du,      "vhsubw.qu.du",     0,  0x705b8000,0xffff8000,DF_S_3R)
INST(vadda_b,           "vadda.b",          0,  0x705c0000,0xffff8000,DF_S_3R)
INST(vadda_h,           "vadda.h",          0,  0x705c8000,0xffff8000,DF_S_3R)
INST(vadda_w,           "vadda.w",          0,  0x705d0000,0xffff8000,DF_S_3R)
INST(vadda_d,           "vadda.d",          0,  0x705d8000,0xffff8000,DF_S_3R)
INST(vabsd_b,           "vabsd.b",          0,  0x70600000,0xffff8000,DF_S_3R)
INST(vabsd_h,           "vabsd.h",          0,  0x70608000,0xffff8000,DF_S_3R)
INST(vabsd_w,           "vabsd.w",          0,  0x70610000,0xffff8000,DF_S_3R)
INST(vabsd_d,           "vabsd.d",          0,  0x70618000,0xffff8000,DF_S_3R)
INST(vabsd_bu,          "vabsd.bu",         0,  0x70620000,0xffff8000,DF_S_3R)
INST(vabsd_hu,          "vabsd.hu",         0,  0x70628000,0xffff8000,DF_S_3R)
INST(vabsd_wu,          "vabsd.wu",         0,  0x70630000,0xffff8000,DF_S_3R)
INST(vabsd_du,          "vabsd.du",         0,  0x70638000,0xffff8000,DF_S_3R)
INST(vavg_b,            "vavg.b",           0,  0x70640000,0xffff8000,DF_S_3R)
INST(vavg_h,            "vavg.h",           0,  0x70648000,0xffff8000,DF_S_3R)
INST(vavg_w,            "vavg.w",           0,  0x70650000,0xffff8000,DF_S_3R)
INST(vavg_d,            "vavg.d",           0,  0x70658000,0xffff8000,DF_S_3R)
INST(vavg_bu,           "vavg.bu",          0,  0x70660000,0xffff8000,DF_S_3R)
INST(vavg_hu,           "vavg.hu",          0,  0x70668000,0xffff8000,DF_S_3R)
INST(vavg_wu,           "vavg.wu",          0,  0x70670000,0xffff8000,DF_S_3R)
INST(vavg_du,           "vavg.du",          0,  0x70678000,0xffff8000,DF_S_3R)
INST(vavgr_b,           "vavgr.b",          0,  0x70680000,0xffff8000,DF_S_3R)
INST(vavgr_h,           "vavgr.h",          0,  0x70688000,0xffff8000,DF_S_3R)
INST(vavgr_w,           "vavgr.w",          0,  0x70690000,0xffff8000,DF_S_3R)
INST(vavgr_d,           "vavgr.d",          0,  0x70698000,0xffff8000,DF_S_3R)
INST(vavgr_bu,          "vavgr.bu",         0,  0x706a0000,0xffff8000,DF_S_3R)
INST(vavgr_hu,          "vavgr.hu",         0,  0x706a8000,0xffff8000,DF_S_3R)
INST(vavgr_wu,          "vavgr.wu",         0,  0x706b0000,0xffff8000,DF_S_3R)
INST(vavgr_du,          "vavgr.du",         0,  0x706b8000,0xffff8000,DF_S_3R)
INST(vmax_b,            "vmax.b",           0,  0x70700000,0xffff8000,DF_S_3R)
INST(vmax_h,            "vmax.h",           0,  0x70708000,0xffff8000,DF_S_3R)
INST(vmax_w,            "vmax.w",           0,  0x70710000,0xffff8000,DF_S_3R)
INST(vmax_d,            "vmax.d",           0,  0x70718000,0xffff8000,DF_S_3R)
INST(vmin_b,            "vmin.b",           0,  0x70720000,0xffff8000,DF_S_3R)
INST(vmin_h,            "vmin.h",           0,  0x70728000,0xffff8000,DF_S_3R)
INST(vmin_w,            "vmin.w",           0,  0x70730000,0xffff8000,DF_S_3R)
INST(vmin_d,            "vmin.d",           0,  0x70738000,0xffff8000,DF_S_3R)
INST(vmax_bu,           "vmax.bu",          0,  0x70740000,0xffff8000,DF_S_3R)
INST(vmax_hu,           "vmax.hu",          0,  0x70748000,0xffff8000,DF_S_3R)
INST(vmax_wu,           "vmax.wu",          0,  0x70750000,0xffff8000,DF_S_3R)
INST(vmax_du,           "vmax.du",          0,  0x70758000,0xffff8000,DF_S_3R)
INST(vmin_bu,           "vmin.bu",          0,  0x70760000,0xffff8000,DF_S_3R)
INST(vmin_hu,           "vmin.hu",          0,  0x70768000,0xffff8000,DF_S_3R)
INST(vmin_wu,           "vmin.wu",          0,  0x70770000,0xffff8000,DF_S_3R)
INST(vmin_du,           "vmin.du",          0,  0x70778000,0xffff8000,DF_S_3R)
INST(vmul_b,            "vmul.b",           0,  0x70840000,0xffff8000,DF_S_3R)
INST(vmul_h,            "vmul.h",           0,  0x70848000,0xffff8000,DF_S_3R)
INST(vmul_w,            "vmul.w",           0,  0x70850000,0xffff8000,DF_S_3R)
INST(vmul_d,            "vmul.d",           0,  0x70858000,0xffff8000,DF_S_3R)
INST(vmuh_b,            "vmuh.b",           0,  0x70860000,0xffff8000,DF_S_3R)
INST(vmuh_h,            "vmuh.h",           0,  0x70868000,0xffff8000,DF_S_3R)
INST(vmuh_w,            "vmuh.w",           0,  0x70870000,0xffff8000,DF_S_3R)
INST(vmuh_d,            "vmuh.d",           0,  0x70878000,0xffff8000,DF_S_3R)
INST(vmuh_bu,           "vmuh.bu",          0,  0x70880000,0xffff8000,DF_S_3R)
INST(vmuh_hu,           "vmuh.hu",          0,  0x70888000,0xffff8000,DF_S_3R)
INST(vmuh_wu,           "vmuh.wu",          0,  0x70890000,0xffff8000,DF_S_3R)
INST(vmuh_du,           "vmuh.du",          0,  0x70898000,0xffff8000,DF_S_3R)
INST(vmulwev_h_b,       "vmulwev.h.b",      0,  0x70900000,0xffff8000,DF_S_3R)
INST(vmulwev_w_h,       "vmulwev.w.h",      0,  0x70908000,0xffff8000,DF_S_3R)
INST(vmulwev_d_w,       "vmulwev.d.w",      0,  0x70910000,0xffff8000,DF_S_3R)
INST(vmulwev_q_d,       "vmulwev.q.d",      0,  0x70918000,0xffff8000,DF_S_3R)
INST(vmulwod_h_b,       "vmulwod.h.b",      0,  0x70920000,0xffff8000,DF_S_3R)
INST(vmulwod_w_h,       "vmulwod.w.h",      0,  0x70928000,0xffff8000,DF_S_3R)
INST(vmulwod_d_w,       "vmulwod.d.w",      0,  0x70930000,0xffff8000,DF_S_3R)
INST(vmulwod_q_d,       "vmulwod.q.d",      0,  0x70938000,0xffff8000,DF_S_3R)
INST(vmulwev_h_bu,      "vmulwev.h.bu",     0,  0x70980000,0xffff8000,DF_S_3R)
INST(vmulwev_w_hu,      "vmulwev.w.hu",     0,  0x70988000,0xffff8000,DF_S_3R)
INST(vmulwev_d_wu,      "vmulwev.d.wu",     0,  0x70990000,0xffff8000,DF_S_3R)
INST(vmulwev_q_du,      "vmulwev.q.du",     0,  0x70998000,0xffff8000,DF_S_3R)
INST(vmulwod_h_bu,      "vmulwod.h.bu",     0,  0x709a0000,0xffff8000,DF_S_3R)
INST(vmulwod_w_hu,      "vmulwod.w.hu",     0,  0x709a8000,0xffff8000,DF_S_3R)
INST(vmulwod_d_wu,      "vmulwod.d.wu",     0,  0x709b0000,0xffff8000,DF_S_3R)
INST(vmulwod_q_du,      "vmulwod.q.du",     0,  0x709b8000,0xffff8000,DF_S_3R)
INST(vmulwev_h_bu_b,    "vmulwev.h.bu.b",   0,  0x70a00000,0xffff8000,DF_S_3R)
INST(vmulwev_w_hu_h,    "vmulwev.w.hu.h",   0,  0x70a08000,0xffff8000,DF_S_3R)
INST(vmulwev_d_wu_w,    "vmulwev.d.wu.w",   0,  0x70a10000,0xffff8000,DF_S_3R)
INST(vmulwev_q_du_d,    "vmulwev.q.du.d",   0,  0x70a18000,0xffff8000,DF_S_3R)
INST(vmulwod_h_bu_b,    "vmulwod.h.bu.b",   0,  0x70a20000,0xffff8000,DF_S_3R)
INST(vmulwod_w_hu_h,    "vmulwod.w.hu.h",   0,  0x70a28000,0xffff8000,DF_S_3R)
INST(vmulwod_d_wu_w,    "vmulwod.d.wu.w",   0,  0x70a30000,0xffff8000,DF_S_3R)
INST(vmulwod_q_du_d,    "vmulwod.q.du.d",   0,  0x70a38000,0xffff8000,DF_S_3R)
INST(vmadd_b,           "vmadd.b",          0,  0x70a80000,0xffff8000,DF_S_3R)
INST(vmadd_h,           "vmadd.h",          0,  0x70a88000,0xffff8000,DF_S_3R)
INST(vmadd_w,           "vmadd.w",          0,  0x70a90000,0xffff8000,DF_S_3R)
INST(vmadd_d,           "vmadd.d",          0,  0x70a98000,0xffff8000,DF_S_3R)
INST(vmsub_b,           "vmsub.b",          0,  0x70aa0000,0xffff8000,DF_S_3R)
INST(vmsub_h,           "vmsub.h",          0,  0x70aa8000,0xffff8000,DF_S_3R)
INST(vmsub_w,           "vmsub.w",          0,  0x70ab0000,0xffff8000,DF_S_3R)
INST(vmsub_d,           "vmsub.d",          0,  0x70ab8000,0xffff8000,DF_S_3R)
INST(vmaddwev_h_b,      "vmaddwev.h.b",     0,  0x70ac0000,0xffff8000,DF_S_3R)
INST(vmaddwev_w_h,      "vmaddwev.w.h",     0,  0x70ac8000,0xffff8000,DF_S_3R)
INST(vmaddwev_d_w,      "vmaddwev.d.w",     0,  0x70ad0000,0xffff8000,DF_S_3R)
INST(vmaddwev_q_d,      "vmaddwev.q.d",     0,  0x70ad8000,0xffff8000,DF_S_3R)
INST(vmaddwod_h_b,      "vmaddwod.h.b",     0,  0x70ae0000,0xffff8000,DF_S_3R)
INST(vmaddwod_w_h,      "vmaddwod.w.h",     0,  0x70ae8000,0xffff8000,DF_S_3R)
INST(vmaddwod_d_w,      "vmaddwod.d.w",     0,  0x70af0000,0xffff8000,DF_S_3R)
INST(vmaddwod_q_d,      "vmaddwod.q.d",     0,  0x70af8000,0xffff8000,DF_S_3R)
INST(vmaddwev_h_bu,     "vmaddwev.h.bu",    0,  0x70b40000,0xffff8000,DF_S_3R)
INST(vmaddwev_w_hu,     "vmaddwev.w.hu",    0,  0x70b48000,0xffff8000,DF_S_3R)
INST(vmaddwev_d_wu,     "vmaddwev.d.wu",    0,  0x70b50000,0xffff8000,DF_S_3R)
INST(vmaddwev_q_du,     "vmaddwev.q.du",    0,  0x70b58000,0xffff8000,DF_S_3R)
INST(vmaddwod_h_bu,     "vmaddwod.h.bu",    0,  0x70b60000,0xffff8000,DF_S_3R)
INST(vmaddwod_w_hu,     "vmaddwod.w.hu",    0,  0x70b68000,0xffff8000,DF_S_3R)
INST(vmaddwod_d_wu,     "vmaddwod.d.wu",    0,  0x70b70000,0xffff8000,DF_S_3R)
INST(vmaddwod_q_du,     "vmaddwod.q.du",    0,  0x70b78000,0xffff8000,DF_S_3R)
INST(vmaddwev_h_bu_b,   "vmaddwev.h.bu.b",  0,  0x70bc0000,0xffff8000,DF_S_3R)
INST(vmaddwev_w_hu_h,   "vmaddwev.w.hu.h",  0,  0x70bc8000,0xffff8000,DF_S_3R)
INST(vmaddwev_d_wu_w,   "vmaddwev.d.wu.w",  0,  0x70bd0000,0xffff8000,DF_S_3R)
INST(vmaddwev_q_du_d,   "vmaddwev.q.du.d",  0,  0x70bd8000,0xffff8000,DF_S_3R)
INST(vmaddwod_h_bu_b,   "vmaddwod.h.bu.b",  0,  0x70be0000,0xffff8000,DF_S_3R)
INST(vmaddwod_w_hu_h,   "vmaddwod.w.hu.h",  0,  0x70be8000,0xffff8000,DF_S_3R)
INST(vmaddwod_d_wu_w,   "vmaddwod.d.wu.w",  0,  0x70bf0000,0xffff8000,DF_S_3R)
INST(vmaddwod_q_du_d,   "vmaddwod.q.du.d",  0,  0x70bf8000,0xffff8000,DF_S_3R)
INST(vdiv_b,            "vdiv.b",           0,  0x70e00000,0xffff8000,DF_S_3R)
INST(vdiv_h,            "vdiv.h",           0,  0x70e08000,0xffff8000,DF_S_3R)
INST(vdiv_w,            "vdiv.w",           0,  0x70e10000,0xffff8000,DF_S_3R)
INST(vdiv_d,            "vdiv.d",           0,  0x70e18000,0xffff8000,DF_S_3R)
INST(vmod_b,            "vmod.b",           0,  0x70e20000,0xffff8000,DF_S_3R)
INST(vmod_h,            "vmod.h",           0,  0x70e28000,0xffff8000,DF_S_3R)
INST(vmod_w,            "vmod.w",           0,  0x70e30000,0xffff8000,DF_S_3R)
INST(vmod_d,            "vmod.d",           0,  0x70e38000,0xffff8000,DF_S_3R)
INST(vdiv_bu,           "vdiv.bu",          0,  0x70e40000,0xffff8000,DF_S_3R)
INST(vdiv_hu,           "vdiv.hu",          0,  0x70e48000,0xffff8000,DF_S_3R)
INST(vdiv_wu,           "vdiv.wu",          0,  0x70e50000,0xffff8000,DF_S_3R)
INST(vdiv_du,           "vdiv.du",          0,  0x70e58000,0xffff8000,DF_S_3R)
INST(vmod_bu,           "vmod.bu",          0,  0x70e60000,0xffff8000,DF_S_3R)
INST(vmod_hu,           "vmod.hu",          0,  0x70e68000,0xffff8000,DF_S_3R)
INST(vmod_wu,           "vmod.wu",          0,  0x70e70000,0xffff8000,DF_S_3R)
INST(vmod_du,           "vmod.du",          0,  0x70e78000,0xffff8000,DF_S_3R)
INST(vsll_b,            "vsll.b",           0,  0x70e80000,0xffff8000,DF_S_3R)
INST(vsll_h,            "vsll.h",           0,  0x70e88000,0xffff8000,DF_S_3R)
INST(vsll_w,            "vsll.w",           0,  0x70e90000,0xffff8000,DF_S_3R)
INST(vsll_d,            "vsll.d",           0,  0x70e98000,0xffff8000,DF_S_3R)
INST(vsrl_b,            "vsrl.b",           0,  0x70ea0000,0xffff8000,DF_S_3R)
INST(vsrl_h,            "vsrl.h",           0,  0x70ea8000,0xffff8000,DF_S_3R)
INST(vsrl_w,            "vsrl.w",           0,  0x70eb0000,0xffff8000,DF_S_3R)
INST(vsrl_d,            "vsrl.d",           0,  0x70eb8000,0xffff8000,DF_S_3R)
INST(vsra_b,            "vsra.b",           0,  0x70ec0000,0xffff8000,DF_S_3R)
INST(vsra_h,            "vsra.h",           0,  0x70ec8000,0xffff8000,DF_S_3R)
INST(vsra_w,            "vsra.w",           0,  0x70ed0000,0xffff8000,DF_S_3R)
INST(vsra_d,            "vsra.d",           0,  0x70ed8000,0xffff8000,DF_S_3R)
INST(vrotr_b,           "vrotr.b",          0,  0x70ee0000,0xffff8000,DF_S_3R)
INST(vrotr_h,           "vrotr.h",          0,  0x70ee8000,0xffff8000,DF_S_3R)
INST(vrotr_w,           "vrotr.w",          0,  0x70ef0000,0xffff8000,DF_S_3R)
INST(vrotr_d,           "vrotr.d",          0,  0x70ef8000,0xffff8000,DF_S_3R)
INST(vsrlr_b,           "vsrlr.b",          0,  0x70f00000,0xffff8000,DF_S_3R)
INST(vsrlr_h,           "vsrlr.h",          0,  0x70f08000,0xffff8000,DF_S_3R)
INST(vsrlr_w,           "vsrlr.w",          0,  0x70f10000,0xffff8000,DF_S_3R)
INST(vsrlr_d,           "vsrlr.d",          0,  0x70f18000,0xffff8000,DF_S_3R)
INST(vsrar_b,           "vsrar.b",          0,  0x70f20000,0xffff8000,DF_S_3R)
INST(vsrar_h,           "vsrar.h",          0,  0x70f28000,0xffff8000,DF_S_3R)
INST(vsrar_w,           "vsrar.w",          0,  0x70f30000,0xffff8000,DF_S_3R)
INST(vsrar_d,           "vsrar.d",          0,  0x70f38000,0xffff8000,DF_S_3R)
INST(vsrln_b_h,         "vsrln.b.h",        0,  0x70f48000,0xffff8000,DF_S_3R)
INST(vsrln_h_w,         "vsrln.h.w",        0,  0x70f50000,0xffff8000,DF_S_3R)
INST(vsrln_w_d,         "vsrln.w.d",        0,  0x70f58000,0xffff8000,DF_S_3R)
INST(vsran_b_h,         "vsran.b.h",        0,  0x70f68000,0xffff8000,DF_S_3R)
INST(vsran_h_w,         "vsran.h.w",        0,  0x70f70000,0xffff8000,DF_S_3R)
INST(vsran_w_d,         "vsran.w.d",        0,  0x70f78000,0xffff8000,DF_S_3R)
INST(vsrlrn_b_h,        "vsrlrn.b.h",       0,  0x70f88000,0xffff8000,DF_S_3R)
INST(vsrlrn_h_w,        "vsrlrn.h.w",       0,  0x70f90000,0xffff8000,DF_S_3R)
INST(vsrlrn_w_d,        "vsrlrn.w.d",       0,  0x70f98000,0xffff8000,DF_S_3R)
INST(vsrarn_b_h,        "vsrarn.b.h",       0,  0x70fa8000,0xffff8000,DF_S_3R)
INST(vsrarn_h_w,        "vsrarn.h.w",       0,  0x70fb0000,0xffff8000,DF_S_3R)
INST(vsrarn_w_d,        "vsrarn.w.d",       0,  0x70fb8000,0xffff8000,DF_S_3R)
INST(vssrln_b_h,        "vssrln.b.h",       0,  0x70fc8000,0xffff8000,DF_S_3R)
INST(vssrln_h_w,        "vssrln.h.w",       0,  0x70fd0000,0xffff8000,DF_S_3R)
INST(vssrln_w_d,        "vssrln.w.d",       0,  0x70fd8000,0xffff8000,DF_S_3R)
INST(vssran_b_h,        "vssran.b.h",       0,  0x70fe8000,0xffff8000,DF_S_3R)
INST(vssran_h_w,        "vssran.h.w",       0,  0x70ff0000,0xffff8000,DF_S_3R)
INST(vssran_w_d,        "vssran.w.d",       0,  0x70ff8000,0xffff8000,DF_S_3R)
INST(vssrlrn_b_h,       "vssrlrn.b.h",      0,  0x71008000,0xffff8000,DF_S_3R)
INST(vssrlrn_h_w,       "vssrlrn.h.w",      0,  0x71010000,0xffff8000,DF_S_3R)
INST(vssrlrn_w_d,       "vssrlrn.w.d",      0,  0x71018000,0xffff8000,DF_S_3R)
INST(vssrarn_b_h,       "vssrarn.b.h",      0,  0x71028000,0xffff8000,DF_S_3R)
INST(vssrarn_h_w,       "vssrarn.h.w",      0,  0x71030000,0xffff8000,DF_S_3R)
INST(vssrarn_w_d,       "vssrarn.w.d",      0,  0x71038000,0xffff8000,DF_S_3R)
INST(vssrln_bu_h,       "vssrln.bu.h",      0,  0x71048000,0xffff8000,DF_S_3R)
INST(vssrln_hu_w,       "vssrln.hu.w",      0,  0x71050000,0xffff8000,DF_S_3R)
INST(vssrln_wu_d,       "vssrln.wu.d",      0,  0x71058000,0xffff8000,DF_S_3R)
INST(vssran_bu_h,       "vssran.bu.h",      0,  0x71068000,0xffff8000,DF_S_3R)
INST(vssran_hu_w,       "vssran.hu.w",      0,  0x71070000,0xffff8000,DF_S_3R)
INST(vssran_wu_d,       "vssran.wu.d",      0,  0x71078000,0xffff8000,DF_S_3R)
INST(vssrlrn_bu_h,      "vssrlrn.bu.h",     0,  0x71088000,0xffff8000,DF_S_3R)
INST(vssrlrn_hu_w,      "vssrlrn.hu.w",     0,  0x71090000,0xffff8000,DF_S_3R)
INST(vssrlrn_wu_d,      "vssrlrn.wu.d",     0,  0x71098000,0xffff8000,DF_S_3R)
INST(vssrarn_bu_h,      "vssrarn.bu.h",     0,  0x710a8000,0xffff8000,DF_S_3R)
INST(vssrarn_hu_w,      "vssrarn.hu.w",     0,  0x710b0000,0xffff8000,DF_S_3R)
INST(vssrarn_wu_d,      "vssrarn.wu.d",     0,  0x710b8000,0xffff8000,DF_S_3R)
INST(vbitclr_b,         "vbitclr.b",        0,  0x710c0000,0xffff8000,DF_S_3R)
INST(vbitclr_h,         "vbitclr.h",        0,  0x710c8000,0xffff8000,DF_S_3R)
INST(vbitclr_w,         "vbitclr.w",        0,  0x710d0000,0xffff8000,DF_S_3R)
INST(vbitclr_d,         "vbitclr.d",        0,  0x710d8000,0xffff8000,DF_S_3R)
INST(vbitset_b,         "vbitset.b",        0,  0x710e0000,0xffff8000,DF_S_3R)
INST(vbitset_h,         "vbitset.h",        0,  0x710e8000,0xffff8000,DF_S_3R)
INST(vbitset_w,         "vbitset.w",        0,  0x710f0000,0xffff8000,DF_S_3R)
INST(vbitset_d,         "vbitset.d",        0,  0x710f8000,0xffff8000,DF_S_3R)
INST(vbitrev_b,         "vbitrev.b",        0,  0x71100000,0xffff8000,DF_S_3R)
INST(vbitrev_h,         "vbitrev.h",        0,  0x71108000,0xffff8000,DF_S_3R)
INST(vbitrev_w,         "vbitrev.w",        0,  0x71110000,0xffff8000,DF_S_3R)
INST(vbitrev_d,         "vbitrev.d",        0,  0x71118000,0xffff8000,DF_S_3R)
INST(vpackev_b,         "vpackev.b",        0,  0x71160000,0xffff8000,DF_S_3R)
INST(vpackev_h,         "vpackev.h",        0,  0x71168000,0xffff8000,DF_S_3R)
INST(vpackev_w,         "vpackev.w",        0,  0x71170000,0xffff8000,DF_S_3R)
INST(vpackev_d,         "vpackev.d",        0,  0x71178000,0xffff8000,DF_S_3R)
INST(vpackod_b,         "vpackod.b",        0,  0x71180000,0xffff8000,DF_S_3R)
INST(vpackod_h,         "vpackod.h",        0,  0x71188000,0xffff8000,DF_S_3R)
INST(vpackod_w,         "vpackod.w",        0,  0x71190000,0xffff8000,DF_S_3R)
INST(vpackod_d,         "vpackod.d",        0,  0x71198000,0xffff8000,DF_S_3R)
INST(vilvl_b,           "vilvl.b",          0,  0x711a0000,0xffff8000,DF_S_3R)
INST(vilvl_h,           "vilvl.h",          0,  0x711a8000,0xffff8000,DF_S_3R)
INST(vilvl_w,           "vilvl.w",          0,  0x711b0000,0xffff8000,DF_S_3R)
INST(vilvl_d,           "vilvl.d",          0,  0x711b8000,0xffff8000,DF_S_3R)
INST(vilvh_b,           "vilvh.b",          0,  0x711c0000,0xffff8000,DF_S_3R)
INST(vilvh_h,           "vilvh.h",          0,  0x711c8000,0xffff8000,DF_S_3R)
INST(vilvh_w,           "vilvh.w",          0,  0x711d0000,0xffff8000,DF_S_3R)
INST(vilvh_d,           "vilvh.d",          0,  0x711d8000,0xffff8000,DF_S_3R)
INST(vpickev_b,         "vpickev.b",        0,  0x711e0000,0xffff8000,DF_S_3R)
INST(vpickev_h,         "vpickev.h",        0,  0x711e8000,0xffff8000,DF_S_3R)
INST(vpickev_w,         "vpickev.w",        0,  0x711f0000,0xffff8000,DF_S_3R)
INST(vpickev_d,         "vpickev.d",        0,  0x711f8000,0xffff8000,DF_S_3R)
INST(vpickod_b,         "vpickod.b",        0,  0x71200000,0xffff8000,DF_S_3R)
INST(vpickod_h,         "vpickod.h",        0,  0x71208000,0xffff8000,DF_S_3R)
INST(vpickod_w,         "vpickod.w",        0,  0x71210000,0xffff8000,DF_S_3R)
INST(vpickod_d,         "vpickod.d",        0,  0x71218000,0xffff8000,DF_S_3R)
INST(vand_v,            "vand.v",           0,  0x71260000,0xffff8000,DF_S_3R)
INST(vor_v,             "vor.v",            0,  0x71268000,0xffff8000,DF_S_3R)
INST(vxor_v,            "vxor.v",           0,  0x71270000,0xffff8000,DF_S_3R)
INST(vnor_v,            "vnor.v",           0,  0x71278000,0xffff8000,DF_S_3R)
INST(vandn_v,           "vandn.v",          0,  0x71280000,0xffff8000,DF_S_3R)
INST(vorn_v,            "vorn.v",           0,  0x71288000,0xffff8000,DF_S_3R)
INST(vfrstp_b,          "vfrstp.b",         0,  0x712b0000,0xffff8000,DF_S_3R)
INST(vfrstp_h,          "vfrstp.h",         0,  0x712b8000,0xffff8000,DF_S_3R)
INST(vadd_q,            "vadd.q",           0,  0x712d0000,0xffff8000,DF_S_3R)
INST(vsub_q,            "vsub.q",           0,  0x712d8000,0xffff8000,DF_S_3R)
INST(vsigncov_b,        "vsigncov.b",       0,  0x712e0000,0xffff8000,DF_S_3R)
INST(vsigncov_h,        "vsigncov.h",       0,  0x712e8000,0xffff8000,DF_S_3R)
INST(vsigncov_w,        "vsigncov.w",       0,  0x712f0000,0xffff8000,DF_S_3R)
INST(vsigncov_d,        "vsigncov.d",       0,  0x712f8000,0xffff8000,DF_S_3R)
INST(vfadd_s,           "vfadd.s",          0,  0x71308000,0xffff8000,DF_S_3R)
INST(vfadd_d,           "vfadd.d",          0,  0x71310000,0xffff8000,DF_S_3R)
INST(vfsub_s,           "vfsub.s",          0,  0x71328000,0xffff8000,DF_S_3R)
INST(vfsub_d,           "vfsub.d",          0,  0x71330000,0xffff8000,DF_S_3R)
INST(vfmul_s,           "vfmul.s",          0,  0x71388000,0xffff8000,DF_S_3R)
INST(vfmul_d,           "vfmul.d",          0,  0x71390000,0xffff8000,DF_S_3R)
INST(vfdiv_s,           "vfdiv.s",          0,  0x713a8000,0xffff8000,DF_S_3R)
INST(vfdiv_d,           "vfdiv.d",          0,  0x713b0000,0xffff8000,DF_S_3R)
INST(vfmax_s,           "vfmax.s",          0,  0x713c8000,0xffff8000,DF_S_3R)
INST(vfmax_d,           "vfmax.d",          0,  0x713d0000,0xffff8000,DF_S_3R)
INST(vfmin_s,           "vfmin.s",          0,  0x713e8000,0xffff8000,DF_S_3R)
INST(vfmin_d,           "vfmin.d",          0,  0x713f0000,0xffff8000,DF_S_3R)
INST(vfmaxa_s,          "vfmaxa.s",         0,  0x71408000,0xffff8000,DF_S_3R)
INST(vfmaxa_d,          "vfmaxa.d",         0,  0x71410000,0xffff8000,DF_S_3R)
INST(vfmina_s,          "vfmina.s",         0,  0x71428000,0xffff8000,DF_S_3R)
INST(vfmina_d,          "vfmina.d",         0,  0x71430000,0xffff8000,DF_S_3R)
INST(vfcvt_h_s,         "vfcvt.h.s",        0,  0x71460000,0xffff8000,DF_S_3R)
INST(vfcvt_s_d,         "vfcvt.s.d",        0,  0x71468000,0xffff8000,DF_S_3R)
INST(vffint_s_l,        "vffint.s.l",       0,  0x71480000,0xffff8000,DF_S_3R)
INST(vftint_w_d,        "vftint.w.d",       0,  0x71498000,0xffff8000,DF_S_3R)
INST(vftintrm_w_d,      "vftintrm.w.d",     0,  0x714a0000,0xffff8000,DF_S_3R)
INST(vftintrp_w_d,      "vftintrp.w.d",     0,  0x714a8000,0xffff8000,DF_S_3R)
INST(vftintrz_w_d,      "vftintrz.w.d",     0,  0x714b0000,0xffff8000,DF_S_3R)
INST(vftintrne_w_d,     "vftintrne.w.d",    0,  0x714b8000,0xffff8000,DF_S_3R)
INST(vshuf_h,           "vshuf.h",          0,  0x717a8000,0xffff8000,DF_S_3R)
INST(vshuf_w,           "vshuf.w",          0,  0x717b0000,0xffff8000,DF_S_3R)
INST(vshuf_d,           "vshuf.d",          0,  0x717b8000,0xffff8000,DF_S_3R)

// R_R.
// vd, rj
INST(vreplgr2vr_b,      "vreplgr2vr.b",     0,  0x729f0000,0xfffffc00,DF_S_RG)
INST(vreplgr2vr_h,      "vreplgr2vr.h",     0,  0x729f0400,0xfffffc00,DF_S_RG)
INST(vreplgr2vr_w,      "vreplgr2vr.w",     0,  0x729f0800,0xfffffc00,DF_S_RG)
INST(vreplgr2vr_d,      "vreplgr2vr.d",     0,  0x729f0c00,0xfffffc00,DF_S_RG)

// vd, vj
INST(vclo_b,            "vclo.b",           0,  0x729c0000,0xfffffc00,DF_S_2R)
INST(vclo_h,            "vclo.h",           0,  0x729c0400,0xfffffc00,DF_S_2R)
INST(vclo_w,            "vclo.w",           0,  0x729c0800,0xfffffc00,DF_S_2R)
INST(vclo_d,            "vclo.d",           0,  0x729c0c00,0xfffffc00,DF_S_2R)
INST(vclz_b,            "vclz.b",           0,  0x729c1000,0xfffffc00,DF_S_2R)
INST(vclz_h,            "vclz.h",           0,  0x729c1400,0xfffffc00,DF_S_2R)
INST(vclz_w,            "vclz.w",           0,  0x729c1800,0xfffffc00,DF_S_2R)
INST(vclz_d,            "vclz.d",           0,  0x729c1c00,0xfffffc00,DF_S_2R)
INST(vpcnt_b,           "vpcnt.b",          0,  0x729c2000,0xfffffc00,DF_S_2R)
INST(vpcnt_h,           "vpcnt.h",          0,  0x729c2400,0xfffffc00,DF_S_2R)
INST(vpcnt_w,           "vpcnt.w",          0,  0x729c2800,0xfffffc00,DF_S_2R)
INST(vpcnt_d,           "vpcnt.d",          0,  0x729c2c00,0xfffffc00,DF_S_2R)
INST(vneg_b,            "vneg.b",           0,  0x729c3000,0xfffffc00,DF_S_2R)
INST(vneg_h,            "vneg.h",           0,  0x729c3400,0xfffffc00,DF_S_2R)
INST(vneg_w,            "vneg.w",           0,  0x729c3800,0xfffffc00,DF_S_2R)
INST(vneg_d,            "vneg.d",           0,  0x729c3c00,0xfffffc00,DF_S_2R)
INST(vmskltz_b,         "vmskltz.b",        0,  0x729c4000,0xfffffc00,DF_S_2R)
INST(vmskltz_h,         "vmskltz.h",        0,  0x729c4400,0xfffffc00,DF_S_2R)
INST(vmskltz_w,         "vmskltz.w",        0,  0x729c4800,0xfffffc00,DF_S_2R)
INST(vmskltz_d,         "vmskltz.d",        0,  0x729c4c00,0xfffffc00,DF_S_2R)
INST(vmskgez_b,         "vmskgez.b",        0,  0x729c5000,0xfffffc00,DF_S_2R)
INST(vmsknz_b,          "vmsknz.b",         0,  0x729c6000,0xfffffc00,DF_S_2R)
INST(vflogb_s,          "vflogb.s",         0,  0x729cc400,0xfffffc00,DF_S_2R)
INST(vflogb_d,          "vflogb.d",         0,  0x729cc800,0xfffffc00,DF_S_2R)
INST(vfclass_s,         "vfclass.s",        0,  0x729cd400,0xfffffc00,DF_S_2R)
INST(vfclass_d,         "vfclass.d",        0,  0x729cd800,0xfffffc00,DF_S_2R)
INST(vfsqrt_s,          "vfsqrt.s",         0,  0x729ce400,0xfffffc00,DF_S_2R)
INST(vfsqrt_d,          "vfsqrt.d",         0,  0x729ce800,0xfffffc00,DF_S_2R)
INST(vfrecip_s,         "vfrecip.s",        0,  0x729cf400,0xfffffc00,DF_S_2R)
INST(vfrecip_d,         "vfrecip.d",        0,  0x729cf800,0xfffffc00,DF_S_2R)
INST(vfrsqrt_s,         "vfrsqrt.s",        0,  0x729d0400,0xfffffc00,DF_S_2R)
INST(vfrsqrt_d,         "vfrsqrt.d",        0,  0x729d0800,0xfffffc00,DF_S_2R)
INST(vfrint_s,          "vfrint.s",         0,  0x729d3400,0xfffffc00,DF_S_2R)
INST(vfrint_d,          "vfrint.d",         0,  0x729d3800,0xfffffc00,DF_S_2R)
INST(vfrintrm_s,        "vfrintrm.s",       0,  0x729d4400,0xfffffc00,DF_S_2R)
INST(vfrintrm_d,        "vfrintrm.d",       0,  0x729d4800,0xfffffc00,DF_S_2R)
INST(vfrintrp_s,        "vfrintrp.s",       0,  0x729d5400,0xfffffc00,DF_S_2R)
INST(vfrintrp_d,        "vfrintrp.d",       0,  0x729d5800,0xfffffc00,DF_S_2R)
INST(vfrintrz_s,        "vfrintrz.s",       0,  0x729d6400,0xfffffc00,DF_S_2R)
INST(vfrintrz_d,        "vfrintrz.d",       0,  0x729d6800,0xfffffc00,DF_S_2R)
INST(vfrintrne_s,       "vfrintrne.s",      0,  0x729d7400,0xfffffc00,DF_S_2R)
INST(vfrintrne_d,       "vfrintrne.d",      0,  0x729d7800,0xfffffc00,DF_S_2R)
INST(vfcvtl_s_h,        "vfcvtl.s.h",       0,  0x729de800,0xfffffc00,DF_S_2R)
INST(vfcvth_s_h,        "vfcvth.s.h",       0,  0x729dec00,0xfffffc00,DF_S_2R)
INST(vfcvtl_d_s,        "vfcvtl.d.s",       0,  0x729df000,0xfffffc00,DF_S_2R)
INST(vfcvth_d_s,        "vfcvth.d.s",       0,  0x729df400,0xfffffc00,DF_S_2R)
INST(vffint_s_w,        "vffint.s.w",       0,  0x729e0000,0xfffffc00,DF_S_2R)
INST(vffint_s_wu,       "vffint.s.wu",      0,  0x729e0400,0xfffffc00,DF_S_2R)
INST(vffint_d_l,        "vffint.d.l",       0,  0x729e0800,0xfffffc00,DF_S_2R)
INST(vffint_d_lu,       "vffint.d.lu",      0,  0x729e0c00,0xfffffc00,DF_S_2R)
INST(vffintl_d_w,       "vffintl.d.w",      0,  0x729e1000,0xfffffc00,DF_S_2R)
INST(vffinth_d_w,       "vffinth.d.w",      0,  0x729e1400,0xfffffc00,DF_S_2R)
INST(vftint_w_s,        "vftint.w.s",       0,  0x729e3000,0xfffffc00,DF_S_2R)
INST(vftint_l_d,        "vftint.l.d",       0,  0x729e3400,0xfffffc00,DF_S_2R)
INST(vftintrm_w_s,      "vftintrm.w.s",     0,  0x729e3800,0xfffffc00,DF_S_2R)
INST(vftintrm_l_d,      "vftintrm.l.d",     0,  0x729e3c00,0xfffffc00,DF_S_2R)
INST(vftintrp_w_s,      "vftintrp.w.s",     0,  0x729e4000,0xfffffc00,DF_S_2R)
INST(vftintrp_l_d,      "vftintrp.l.d",     0,  0x729e4400,0xfffffc00,DF_S_2R)
INST(vftintrz_w_s,      "vftintrz.w.s",     0,  0x729e4800,0xfffffc00,DF_S_2R)
INST(vftintrz_l_d,      "vftintrz.l.d",     0,  0x729e4c00,0xfffffc00,DF_S_2R)
INST(vftintrne_w_s,     "vftintrne.w.s",    0,  0x729e5000,0xfffffc00,DF_S_2R)
INST(vftintrne_l_d,     "vftintrne.l.d",    0,  0x729e5400,0xfffffc00,DF_S_2R)
INST(vftint_wu_s,       "vftint.wu.s",      0,  0x729e5800,0xfffffc00,DF_S_2R)
INST(vftint_lu_d,       "vftint.lu.d",      0,  0x729e5c00,0xfffffc00,DF_S_2R)
INST(vftintrz_wu_s,     "vftintrz.wu.s",    0,  0x729e7000,0xfffffc00,DF_S_2R)
INST(vftintrz_lu_d,     "vftintrz.lu.d",    0,  0x729e7400,0xfffffc00,DF_S_2R)
INST(vftintl_l_s,       "vftintl.l.s",      0,  0x729e8000,0xfffffc00,DF_S_2R)
INST(vftinth_l_s,       "vftinth.l.s",      0,  0x729e8400,0xfffffc00,DF_S_2R)
INST(vftintrml_l_s,     "vftintrml.l.s",    0,  0x729e8800,0xfffffc00,DF_S_2R)
INST(vftintrmh_l_s,     "vftintrmh.l.s",    0,  0x729e8c00,0xfffffc00,DF_S_2R)
INST(vftintrpl_l_s,     "vftintrpl.l.s",    0,  0x729e9000,0xfffffc00,DF_S_2R)
INST(vftintrph_l_s,     "vftintrph.l.s",    0,  0x729e9400,0xfffffc00,DF_S_2R)
INST(vftintrzl_l_s,     "vftintrzl.l.s",    0,  0x729e9800,0xfffffc00,DF_S_2R)
INST(vftintrzh_l_s,     "vftintrzh.l.s",    0,  0x729e9c00,0xfffffc00,DF_S_2R)
INST(vftintrnel_l_s,    "vftintrnel.l.s",   0,  0x729ea000,0xfffffc00,DF_S_2R)
INST(vftintrneh_l_s,    "vftintrneh.l.s",   0,  0x729ea400,0xfffffc00,DF_S_2R)
INST(vexth_h_b,         "vexth.h.b",        0,  0x729ee000,0xfffffc00,DF_S_2R)
INST(vexth_w_h,         "vexth.w.h",        0,  0x729ee400,0xfffffc00,DF_S_2R)
INST(vexth_d_w,         "vexth.d.w",        0,  0x729ee800,0xfffffc00,DF_S_2R)
INST(vexth_q_d,         "vexth.q.d",        0,  0x729eec00,0xfffffc00,DF_S_2R)
INST(vexth_hu_bu,       "vexth.hu.bu",      0,  0x729ef000,0xfffffc00,DF_S_2R)
INST(vexth_wu_hu,       "vexth.wu.hu",      0,  0x729ef400,0xfffffc00,DF_S_2R)
INST(vexth_du_wu,       "vexth.du.wu",      0,  0x729ef800,0xfffffc00,DF_S_2R)
INST(vexth_qu_du,       "vexth.qu.du",      0,  0x729efc00,0xfffffc00,DF_S_2R)
INST(vextl_q_d,         "vextl.q.d",        0,  0x73090000,0xfffffc00,DF_S_2R)
INST(vextl_qu_du,       "vextl.qu.du",      0,  0x730d0000,0xfffffc00,DF_S_2R)

// R_I: vd, ui13
INST(vldi,              "vldi",             LD, 0x73e00000,0xfffc0000,DF_S_R13IU)
//R_I: cd, vj
INST(vseteqz_v,         "vseteqz.v",        0,  0x729c9800,0xfffffc18,DF_S_CR)
INST(vsetnez_v,         "vsetnez.v",        0,  0x729c9c00,0xfffffc18,DF_S_CR)
INST(vsetanyeqz_b,      "vsetanyeqz.b",     0,  0x729ca000,0xfffffc18,DF_S_CR)
INST(vsetanyeqz_h,      "vsetanyeqz.h",     0,  0x729ca400,0xfffffc18,DF_S_CR)
INST(vsetanyeqz_w,      "vsetanyeqz.w",     0,  0x729ca800,0xfffffc18,DF_S_CR)
INST(vsetanyeqz_d,      "vsetanyeqz.d",     0,  0x729cac00,0xfffffc18,DF_S_CR)
INST(vsetallnez_b,      "vsetallnez.b",     0,  0x729cb000,0xfffffc18,DF_S_CR)
INST(vsetallnez_h,      "vsetallnez.h",     0,  0x729cb400,0xfffffc18,DF_S_CR)
INST(vsetallnez_w,      "vsetallnez.w",     0,  0x729cb800,0xfffffc18,DF_S_CR)
INST(vsetallnez_d,      "vsetallnez.d",     0,  0x729cbc00,0xfffffc18,DF_S_CR)

// R_R_I_I.
// vd, rj, si8, idx
INST(vstelm_b,          "vstelm.b",         0,  0x31800000,0xffc00000,DF_S_2R8IX)
INST(vstelm_h,          "vstelm.h",         0,  0x31400000,0xffe00000,DF_S_2R8IX)
INST(vstelm_w,          "vstelm.w",         0,  0x31200000,0xfff00000,DF_S_2R8IX)
INST(vstelm_d,          "vstelm.d",         0,  0x31100000,0xfff80000,DF_S_2R8IX)

// R_R_I.
// si5
INST(vseqi_b,           "vseqi.b",          0,  0x72800000,0xffff8000,DF_S_2R5I)  //vd, vj,si5
INST(vseqi_h,           "vseqi.h",          0,  0x72808000,0xffff8000,DF_S_2R5I)
INST(vseqi_w,           "vseqi.w",          0,  0x72810000,0xffff8000,DF_S_2R5I)
INST(vseqi_d,           "vseqi.d",          0,  0x72818000,0xffff8000,DF_S_2R5I)
INST(vslei_b,           "vslei.b",          0,  0x72820000,0xffff8000,DF_S_2R5I)
INST(vslei_h,           "vslei.h",          0,  0x72828000,0xffff8000,DF_S_2R5I)
INST(vslei_w,           "vslei.w",          0,  0x72830000,0xffff8000,DF_S_2R5I)
INST(vslei_d,           "vslei.d",          0,  0x72838000,0xffff8000,DF_S_2R5I)
INST(vslti_b,           "vslti.b",          0,  0x72860000,0xffff8000,DF_S_2R5I)
INST(vslti_h,           "vslti.h",          0,  0x72868000,0xffff8000,DF_S_2R5I)
INST(vslti_w,           "vslti.w",          0,  0x72870000,0xffff8000,DF_S_2R5I)
INST(vslti_d,           "vslti.d",          0,  0x72878000,0xffff8000,DF_S_2R5I)
INST(vslti_bu,          "vslti.bu",         0,  0x72880000,0xffff8000,DF_S_2R5I)
INST(vmaxi_b,           "vmaxi.b",          0,  0x72900000,0xffff8000,DF_S_2R5I)
INST(vmaxi_h,           "vmaxi.h",          0,  0x72908000,0xffff8000,DF_S_2R5I)
INST(vmaxi_w,           "vmaxi.w",          0,  0x72910000,0xffff8000,DF_S_2R5I)
INST(vmaxi_d,           "vmaxi.d",          0,  0x72918000,0xffff8000,DF_S_2R5I)
INST(vmini_b,           "vmini.b",          0,  0x72920000,0xffff8000,DF_S_2R5I)
INST(vmini_h,           "vmini.h",          0,  0x72928000,0xffff8000,DF_S_2R5I)
INST(vmini_w,           "vmini.w",          0,  0x72930000,0xffff8000,DF_S_2R5I)
INST(vmini_d,           "vmini.d",          0,  0x72938000,0xffff8000,DF_S_2R5I)
// si9
INST(vldrepl_d,         "vldrepl.d",        0,  0x30100000,0xfff80000,DF_S_RGX) // vd,rj,si9
// si10
INST(vldrepl_w,         "vldrepl.w",        0,  0x30200000,0xfff00000,DF_S_RGX) // vd,rj,si10
// si11
INST(vldrepl_h,         "vldrepl.h",        0,  0x30400000,0xffe00000,DF_S_RGX) // vd,rj,si11
// si12
INST(vld,               "vld",              LD, 0x2c000000,0xffc00000,DF_S_RGX) // vd,rj,si12
INST(vldrepl_b,         "vldrepl.b",        0,  0x30800000,0xffc00000,DF_S_RGX)
INST(vst,               "vst",              ST, 0x2c400000,0xffc00000,DF_S_RGX)
// ui1
INST(vinsgr2vr_d,       "vinsgr2vr.d",      0,  0x72ebf000,0xfffff800,DF_S_RGX) // vd,rj,ui1
INST(vpickve2gr_d,      "vpickve2gr.d",     0,  0x72eff000,0xfffff800,DF_S_GRX) // rd,vj,ui1
INST(vpickve2gr_du,     "vpickve2gr.du",    0,  0x72f3f000,0xfffff800,DF_S_GRX) // rd,vj,ui1
INST(vreplvei_d,        "vreplvei.d",       0,  0x72f7f000,0xfffff800,DF_S_2RX) // vd,vj,ui1
// ui2
INST(vinsgr2vr_w,       "vinsgr2vr.w",      0,  0x72ebe000,0xfffff000,DF_S_RGX) // vd,rj,ui2
INST(vpickve2gr_w,      "vpickve2gr.w",     0,  0x72efe000,0xfffff000,DF_S_GRX) // rd,vj,ui2
INST(vpickve2gr_wu,     "vpickve2gr.wu",    0,  0x72f3e000,0xfffff000,DF_S_GRX) // rd,vj,ui2
INST(vreplvei_w,        "vreplvei.w",       0,  0x72f7e000,0xfffff000,DF_S_2RX) // vd,vj,ui2
// ui3
INST(vslli_b,           "vslli.b",          0,  0x732c2000,0xffffe000,DF_S_2R3IU) // vd,vj,ui3
INST(vsrli_b,           "vsrli.b",          0,  0x73302000,0xffffe000,DF_S_2R3IU)
INST(vsrai_b,           "vsrai.b",          0,  0x73342000,0xffffe000,DF_S_2R3IU)
INST(vsrlri_b,          "vsrlri.b",         0,  0x72a42000,0xffffe000,DF_S_2R3IU)
INST(vsrari_b,          "vsrari.b",         0,  0x72a82000,0xffffe000,DF_S_2R3IU)
INST(vrotri_b,          "vrotri.b",         0,  0x72a02000,0xffffe000,DF_S_2R3IU)
INST(vsllwil_h_b,       "vsllwil.h.b",      0,  0x73082000,0xffffe000,DF_S_2R3IU)
INST(vsllwil_hu_bu,     "vsllwil.hu.bu",    0,  0x730c2000,0xffffe000,DF_S_2R3IU)
INST(vbitclri_b,        "vbitclri.b",       0,  0x73102000,0xffffe000,DF_S_2R3IU)
INST(vbitseti_b,        "vbitseti.b",       0,  0x73142000,0xffffe000,DF_S_2R3IU)
INST(vbitrevi_b,        "vbitrevi.b",       0,  0x73182000,0xffffe000,DF_S_2R3IU)
INST(vsat_b,            "vsat.b",           0,  0x73242000,0xffffe000,DF_S_2R3IU)
INST(vsat_bu,           "vsat.bu",          0,  0x73282000,0xffffe000,DF_S_2R3IU)
INST(vinsgr2vr_h,       "vinsgr2vr.h",      0,  0x72ebc000,0xffffe000,DF_S_RGX)   // vd,rj,ui3
INST(vpickve2gr_h,      "vpickve2gr.h",     0,  0x72efc000,0xffffe000,DF_S_GRX)   // rd,vj,ui3
INST(vpickve2gr_hu,     "vpickve2gr.hu",    0,  0x72f3c000,0xffffe000,DF_S_GRX)   // rd,vj,ui3
INST(vreplvei_h,        "vreplvei.h",       0,  0x72f7c000,0xffffe000,DF_S_2R3IU) // vd,vj,ui3
// ui4
INST(vslli_h,           "vslli.h",          0,  0x732c4000,0xffffc000,DF_S_2R4IU) // vd,vj,ui4
INST(vsrli_h,           "vsrli.h",          0,  0x73304000,0xffffc000,DF_S_2R4IU)
INST(vsrai_h,           "vsrai.h",          0,  0x73344000,0xffffc000,DF_S_2R4IU)
INST(vsrlri_h,          "vsrlri.h",         0,  0x72a44000,0xffffc000,DF_S_2R4IU)
INST(vsrari_h,          "vsrari.h",         0,  0x72a84000,0xffffc000,DF_S_2R4IU)
INST(vrotri_h,          "vrotri.h",         0,  0x72a04000,0xffffc000,DF_S_2R4IU)
INST(vsllwil_w_h,       "vsllwil.w.h",      0,  0x73084000,0xffffc000,DF_S_2R4IU)
INST(vsllwil_wu_hu,     "vsllwil.wu.hu",    0,  0x730c4000,0xffffc000,DF_S_2R4IU)
INST(vsrlni_b_h,        "vsrlni.b.h",       0,  0x73404000,0xffffc000,DF_S_2R4IU)
INST(vsrlrni_b_h,       "vsrlrni.b.h",      0,  0x73444000,0xffffc000,DF_S_2R4IU)
INST(vssrlni_b_h,       "vssrlni.b.h",      0,  0x73484000,0xffffc000,DF_S_2R4IU)
INST(vssrlni_bu_h,      "vssrlni.bu.h",     0,  0x734c4000,0xffffc000,DF_S_2R4IU)
INST(vssrlrni_b_h,      "vssrlrni.b.h",     0,  0x73504000,0xffffc000,DF_S_2R4IU)
INST(vssrlrni_bu_h,     "vssrlrni.bu.h",    0,  0x73544000,0xffffc000,DF_S_2R4IU)
INST(vsrani_b_h,        "vsrani.b.h",       0,  0x73584000,0xffffc000,DF_S_2R4IU)
INST(vsrarni_b_h,       "vsrarni.b.h",      0,  0x735c4000,0xffffc000,DF_S_2R4IU)
INST(vssrani_b_h,       "vssrani.b.h",      0,  0x73604000,0xffffc000,DF_S_2R4IU)
INST(vssrani_bu_h,      "vssrani.bu.h",     0,  0x73644000,0xffffc000,DF_S_2R4IU)
INST(vssrarni_b_h,      "vssrarni.b.h",     0,  0x73684000,0xffffc000,DF_S_2R4IU)
INST(vssrarni_bu_h,     "vssrarni.bu.h",    0,  0x736c4000,0xffffc000,DF_S_2R4IU)
INST(vbitclri_h,        "vbitclri.h",       0,  0x73104000,0xffffc000,DF_S_2R4IU)
INST(vbitseti_h,        "vbitseti.h",       0,  0x73144000,0xffffc000,DF_S_2R4IU)
INST(vbitrevi_h,        "vbitrevi.h",       0,  0x73184000,0xffffc000,DF_S_2R4IU)
INST(vsat_h,            "vsat.h",           0,  0x73244000,0xffffc000,DF_S_2R4IU)
INST(vsat_hu,           "vsat.hu",          0,  0x73284000,0xffffc000,DF_S_2R4IU)
INST(vinsgr2vr_b,       "vinsgr2vr.b",      0,  0x72eb8000,0xffffc000,DF_S_RGX)   // vd,rj,ui4
INST(vpickve2gr_b,      "vpickve2gr.b",     0,  0x72ef8000,0xffffc000,DF_S_GRX)   // rd,vj,ui4
INST(vpickve2gr_bu,     "vpickve2gr.bu",    0,  0x72f38000,0xffffc000,DF_S_GRX)   // rd,vj,ui4
INST(vreplvei_b,        "vreplvei.b",       0,  0x72f78000,0xffffc000,DF_S_2R4IU) // vd,vj,ui4
// ui5
INST(vslei_bu,          "vslei.bu",         0,  0x72840000,0xffff8000,DF_S_2R5IU) // vd,vj,ui5
INST(vslei_hu,          "vslei.hu",         0,  0x72848000,0xffff8000,DF_S_2R5IU)
INST(vslei_wu,          "vslei.wu",         0,  0x72850000,0xffff8000,DF_S_2R5IU)
INST(vslei_du,          "vslei.du",         0,  0x72858000,0xffff8000,DF_S_2R5IU)
INST(vslti_hu,          "vslti.hu",         0,  0x72888000,0xffff8000,DF_S_2R5IU)
INST(vslti_wu,          "vslti.wu",         0,  0x72890000,0xffff8000,DF_S_2R5IU)
INST(vslti_du,          "vslti.du",         0,  0x72898000,0xffff8000,DF_S_2R5IU)
INST(vaddi_bu,          "vaddi.bu",         0,  0x728a0000,0xffff8000,DF_S_2R5IU)
INST(vaddi_hu,          "vaddi.hu",         0,  0x728a8000,0xffff8000,DF_S_2R5IU)
INST(vaddi_wu,          "vaddi.wu",         0,  0x728b0000,0xffff8000,DF_S_2R5IU)
INST(vaddi_du,          "vaddi.du",         0,  0x728b8000,0xffff8000,DF_S_2R5IU)
INST(vsubi_bu,          "vsubi.bu",         0,  0x728c0000,0xffff8000,DF_S_2R5IU)
INST(vsubi_hu,          "vsubi.hu",         0,  0x728c8000,0xffff8000,DF_S_2R5IU)
INST(vsubi_wu,          "vsubi.wu",         0,  0x728d0000,0xffff8000,DF_S_2R5IU)
INST(vsubi_du,          "vsubi.du",         0,  0x728d8000,0xffff8000,DF_S_2R5IU)
INST(vslli_w,           "vslli.w",          0,  0x732c8000,0xffff8000,DF_S_2R5IU)
INST(vsrli_w,           "vsrli.w",          0,  0x73308000,0xffff8000,DF_S_2R5IU)
INST(vsrai_w,           "vsrai.w",          0,  0x73348000,0xffff8000,DF_S_2R5IU)
INST(vbsll_v,           "vbsll.v",          0,  0x728e0000,0xffff8000,DF_S_2R5IU)
INST(vbsrl_v,           "vbsrl.v",          0,  0x728e8000,0xffff8000,DF_S_2R5IU)
INST(vsrlri_w,          "vsrlri.w",         0,  0x72a48000,0xffff8000,DF_S_2R5IU)
INST(vsrari_w,          "vsrari.w",         0,  0x72a88000,0xffff8000,DF_S_2R5IU)
INST(vrotri_w,          "vrotri.w",         0,  0x72a08000,0xffff8000,DF_S_2R5IU)
INST(vsllwil_d_w,       "vsllwil.d.w",      0,  0x73088000,0xffff8000,DF_S_2R5IU)
INST(vsllwil_du_wu,     "vsllwil.du.wu",    0,  0x730c8000,0xffff8000,DF_S_2R5IU)
INST(vsrlni_h_w,        "vsrlni.h.w",       0,  0x73408000,0xffff8000,DF_S_2R5IU)
INST(vsrlrni_h_w,       "vsrlrni.h.w",      0,  0x73448000,0xffff8000,DF_S_2R5IU)
INST(vssrlni_h_w,       "vssrlni.h.w",      0,  0x73488000,0xffff8000,DF_S_2R5IU)
INST(vssrlni_hu_w,      "vssrlni.hu.w",     0,  0x734c8000,0xffff8000,DF_S_2R5IU)
INST(vssrlrni_h_w,      "vssrlrni.h.w",     0,  0x73508000,0xffff8000,DF_S_2R5IU)
INST(vssrlrni_hu_w,     "vssrlrni.hu.w",    0,  0x73548000,0xffff8000,DF_S_2R5IU)
INST(vsrani_h_w,        "vsrani.h.w",       0,  0x73588000,0xffff8000,DF_S_2R5IU)
INST(vsrarni_h_w,       "vsrarni.h.w",      0,  0x735c8000,0xffff8000,DF_S_2R5IU)
INST(vssrani_h_w,       "vssrani.h.w",      0,  0x73608000,0xffff8000,DF_S_2R5IU)
INST(vssrani_hu_w,      "vssrani.hu.w",     0,  0x73648000,0xffff8000,DF_S_2R5IU)
INST(vssrarni_h_w,      "vssrarni.h.w",     0,  0x73688000,0xffff8000,DF_S_2R5IU)
INST(vssrarni_hu_w,     "vssrarni.hu.w",    0,  0x736c8000,0xffff8000,DF_S_2R5IU)
INST(vbitclri_w,        "vbitclri.w",       0,  0x73108000,0xffff8000,DF_S_2R5IU)
INST(vbitseti_w,        "vbitseti.w",       0,  0x73148000,0xffff8000,DF_S_2R5IU)
INST(vbitrevi_w,        "vbitrevi.w",       0,  0x73188000,0xffff8000,DF_S_2R5IU)
INST(vmaxi_bu,          "vmaxi.bu",         0,  0x72940000,0xffff8000,DF_S_2R5IU)
INST(vmaxi_hu,          "vmaxi.hu",         0,  0x72948000,0xffff8000,DF_S_2R5IU)
INST(vmaxi_wu,          "vmaxi.wu",         0,  0x72950000,0xffff8000,DF_S_2R5IU)
INST(vmaxi_du,          "vmaxi.du",         0,  0x72958000,0xffff8000,DF_S_2R5IU)
INST(vmini_bu,          "vmini.bu",         0,  0x72960000,0xffff8000,DF_S_2R5IU)
INST(vmini_hu,          "vmini.hu",         0,  0x72968000,0xffff8000,DF_S_2R5IU)
INST(vmini_wu,          "vmini.wu",         0,  0x72970000,0xffff8000,DF_S_2R5IU)
INST(vmini_du,          "vmini.du",         0,  0x72978000,0xffff8000,DF_S_2R5IU)
INST(vfrstpi_b,         "vfrstpi.b",        0,  0x729a0000,0xffff8000,DF_S_2R5IU)
INST(vfrstpi_h,         "vfrstpi.h",        0,  0x729a8000,0xffff8000,DF_S_2R5IU)
INST(vsat_w,            "vsat.w",           0,  0x73248000,0xffff8000,DF_S_2R5IU)
INST(vsat_wu,           "vsat.wu",          0,  0x73288000,0xffff8000,DF_S_2R5IU)
// ui6
INST(vslli_d,           "vslli.d",          0,  0x732d0000,0xffff0000,DF_S_2R6IU) // vd, vj, ui6
INST(vsrli_d,           "vsrli.d",          0,  0x73310000,0xffff0000,DF_S_2R6IU)
INST(vsrai_d,           "vsrai.d",          0,  0x73350000,0xffff0000,DF_S_2R6IU)
INST(vrotri_d,          "vrotri.d",         0,  0x72a10000,0xffff0000,DF_S_2R6IU)
INST(vsrlri_d,          "vsrlri.d",         0,  0x72a50000,0xffff0000,DF_S_2R6IU)
INST(vsrari_d,          "vsrari.d",         0,  0x72a90000,0xffff0000,DF_S_2R6IU)
INST(vsrlni_w_d,        "vsrlni.w.d",       0,  0x73410000,0xffff0000,DF_S_2R6IU)
INST(vsrlrni_w_d,       "vsrlrni.w.d",      0,  0x73450000,0xffff0000,DF_S_2R6IU)
INST(vssrlni_w_d,       "vssrlni.w.d",      0,  0x73490000,0xffff0000,DF_S_2R6IU)
INST(vssrlni_wu_d,      "vssrlni.wu.d",     0,  0x734d0000,0xffff0000,DF_S_2R6IU)
INST(vssrlrni_w_d,      "vssrlrni.w.d",     0,  0x73510000,0xffff0000,DF_S_2R6IU)
INST(vssrlrni_wu_d,     "vssrlrni.wu.d",    0,  0x73550000,0xffff0000,DF_S_2R6IU)
INST(vsrani_w_d,        "vsrani.w.d",       0,  0x73590000,0xffff0000,DF_S_2R6IU)
INST(vsrarni_w_d,       "vsrarni.w.d",      0,  0x735d0000,0xffff0000,DF_S_2R6IU)
INST(vssrani_w_d,       "vssrani.w.d",      0,  0x73610000,0xffff0000,DF_S_2R6IU)
INST(vssrani_wu_d,      "vssrani.wu.d",     0,  0x73650000,0xffff0000,DF_S_2R6IU)
INST(vssrarni_w_d,      "vssrarni.w.d",     0,  0x73690000,0xffff0000,DF_S_2R6IU)
INST(vssrarni_wu_d,     "vssrarni.wu.d",    0,  0x736d0000,0xffff0000,DF_S_2R6IU)
INST(vbitclri_d,        "vbitclri.d",       0,  0x73110000,0xffff0000,DF_S_2R6IU)
INST(vbitseti_d,        "vbitseti.d",       0,  0x73150000,0xffff0000,DF_S_2R6IU)
INST(vbitrevi_d,        "vbitrevi.d",       0,  0x73190000,0xffff0000,DF_S_2R6IU)
INST(vsat_d,            "vsat.d",           0,  0x73250000,0xffff0000,DF_S_2R6IU)
INST(vsat_du,           "vsat.du",          0,  0x73290000,0xffff0000,DF_S_2R6IU)
// ui7
INST(vsrlni_d_q,        "vsrlni.d.q",       0,  0x73420000,0xfffe0000,DF_S_2R7IU)
INST(vsrlrni_d_q,       "vsrlrni.d.q",      0,  0x73460000,0xfffe0000,DF_S_2R7IU)
INST(vssrlni_d_q,       "vssrlni.d.q",      0,  0x734a0000,0xfffe0000,DF_S_2R7IU)
INST(vssrlni_du_q,      "vssrlni.du.q",     0,  0x734e0000,0xfffe0000,DF_S_2R7IU)
INST(vssrlrni_d_q,      "vssrlrni.d.q",     0,  0x73520000,0xfffe0000,DF_S_2R7IU)
INST(vssrlrni_du_q,     "vssrlrni.du.q",    0,  0x73560000,0xfffe0000,DF_S_2R7IU)
INST(vsrani_d_q,        "vsrani.d.q",       0,  0x735a0000,0xfffe0000,DF_S_2R7IU)
INST(vsrarni_d_q,       "vsrarni.d.q",      0,  0x735e0000,0xfffe0000,DF_S_2R7IU)
INST(vssrani_d_q,       "vssrani.d.q",      0,  0x73620000,0xfffe0000,DF_S_2R7IU)
INST(vssrani_du_q,      "vssrani.du.q",     0,  0x73660000,0xfffe0000,DF_S_2R7IU)
INST(vssrarni_d_q,      "vssrarni.d.q",     0,  0x736a0000,0xfffe0000,DF_S_2R7IU)
INST(vssrarni_du_q,     "vssrarni.du.q",    0,  0x736e0000,0xfffe0000,DF_S_2R7IU)
// ui8
INST(vextrins_d,        "vextrins.d",       0,  0x73800000,0xfffc0000,DF_S_2R8IU)
INST(vextrins_w,        "vextrins.w",       0,  0x73840000,0xfffc0000,DF_S_2R8IU)
INST(vextrins_h,        "vextrins.h",       0,  0x73880000,0xfffc0000,DF_S_2R8IU)
INST(vextrins_b,        "vextrins.b",       0,  0x738c0000,0xfffc0000,DF_S_2R8IU)
INST(vshuf4i_b,         "vshuf4i.b",        0,  0x73900000,0xfffc0000,DF_S_2R8IU)
INST(vshuf4i_h,         "vshuf4i.h",        0,  0x73940000,0xfffc0000,DF_S_2R8IU)
INST(vshuf4i_w,         "vshuf4i.w",        0,  0x73980000,0xfffc0000,DF_S_2R8IU)
INST(vshuf4i_d,         "vshuf4i.d",        0,  0x739c0000,0xfffc0000,DF_S_2R8IU)
INST(vbitseli_b,        "vbitseli.b",       0,  0x73c40000,0xfffc0000,DF_S_2R8IU)
INST(vandi_b,           "vandi.b",          0,  0x73d00000,0xfffc0000,DF_S_2R8IU)
INST(vori_b,            "vori.b",           0,  0x73d40000,0xfffc0000,DF_S_2R8IU)
INST(vxori_b,           "vxori.b",          0,  0x73d80000,0xfffc0000,DF_S_2R8IU)
INST(vnori_b,           "vnori.b",          0,  0x73dc0000,0xfffc0000,DF_S_2R8IU)
INST(vpermi_w,          "vpermi.w",         0,  0x73e40000,0xfffc0000,DF_S_2R8IU)

// ---- PartII-LASX:256bits-SIMD
// R_R_R_R.
// xd, xj, xk, xa
INST(xvfmadd_s,         "xvfmadd.s",        0,  0x0a100000,0xfff00000,DF_A_4R)
INST(xvfmadd_d,         "xvfmadd.d",        0,  0x0a200000,0xfff00000,DF_A_4R)
INST(xvfmsub_s,         "xvfmsub.s",        0,  0x0a500000,0xfff00000,DF_A_4R)
INST(xvfmsub_d,         "xvfmsub.d",        0,  0x0a600000,0xfff00000,DF_A_4R)
INST(xvfnmadd_s,        "xvfnmadd.s",       0,  0x0a900000,0xfff00000,DF_A_4R)
INST(xvfnmadd_d,        "xvfnmadd.d",       0,  0x0aa00000,0xfff00000,DF_A_4R)
INST(xvfnmsub_s,        "xvfnmsub.s",       0,  0x0ad00000,0xfff00000,DF_A_4R)
INST(xvfnmsub_d,        "xvfnmsub.d",       0,  0x0ae00000,0xfff00000,DF_A_4R)
INST(xvbitsel_v,        "xvbitsel.v",       0,  0x0d200000,0xfff00000,DF_A_4R)
INST(xvshuf_b,          "xvshuf.b",         0,  0x0d600000,0xfff00000,DF_A_4R)

// R_R_R.
// xd, rj, rk
INST(xvldx,             "xvldx",            LD, 0x38480000,0xffff8000,DF_A_3R)
INST(xvstx,             "xvstx",            ST, 0x384c0000,0xffff8000,DF_A_3R)
// xd, xj, rk
INST(xvreplve_b,        "xvreplve.b",       0,  0x75220000,0xffff8000,DF_A_2RG)
INST(xvreplve_h,        "xvreplve.h",       0,  0x75228000,0xffff8000,DF_A_2RG)
INST(xvreplve_w,        "xvreplve.w",       0,  0x75230000,0xffff8000,DF_A_2RG)
INST(xvreplve_d,        "xvreplve.d",       0,  0x75238000,0xffff8000,DF_A_2RG)
// xd, xj, xk
INST(xvfcmp_caf_s,      "xvfcmp.caf.s",     0,  0x0c900000,0xffff8000,DF_A_3R)
INST(xvfcmp_cun_s,      "xvfcmp.cun.s",     0,  0x0c940000,0xffff8000,DF_A_3R)
INST(xvfcmp_ceq_s,      "xvfcmp.ceq.s",     0,  0x0c920000,0xffff8000,DF_A_3R)
INST(xvfcmp_cueq_s,     "xvfcmp.cueq.s",    0,  0x0c960000,0xffff8000,DF_A_3R)
INST(xvfcmp_clt_s,      "xvfcmp.clt.s",     0,  0x0c910000,0xffff8000,DF_A_3R)
INST(xvfcmp_cult_s,     "xvfcmp.cult.s",    0,  0x0c950000,0xffff8000,DF_A_3R)
INST(xvfcmp_cle_s,      "xvfcmp.cle.s",     0,  0x0c930000,0xffff8000,DF_A_3R)
INST(xvfcmp_cule_s,     "xvfcmp.cule.s",    0,  0x0c970000,0xffff8000,DF_A_3R)
INST(xvfcmp_cne_s,      "xvfcmp.cne.s",     0,  0x0c980000,0xffff8000,DF_A_3R)
INST(xvfcmp_cor_s,      "xvfcmp.cor.s",     0,  0x0c9a0000,0xffff8000,DF_A_3R)
INST(xvfcmp_cune_s,     "xvfcmp.cune.s",    0,  0x0c9c0000,0xffff8000,DF_A_3R)
INST(xvfcmp_saf_s,      "xvfcmp.saf.s",     0,  0x0c908000,0xffff8000,DF_A_3R)
INST(xvfcmp_sun_s,      "xvfcmp.sun.s",     0,  0x0c948000,0xffff8000,DF_A_3R)
INST(xvfcmp_seq_s,      "xvfcmp.seq.s",     0,  0x0c928000,0xffff8000,DF_A_3R)
INST(xvfcmp_sueq_s,     "xvfcmp.sueq.s",    0,  0x0c968000,0xffff8000,DF_A_3R)
INST(xvfcmp_slt_s,      "xvfcmp.slt.s",     0,  0x0c918000,0xffff8000,DF_A_3R)
INST(xvfcmp_sult_s,     "xvfcmp.sult.s",    0,  0x0c958000,0xffff8000,DF_A_3R)
INST(xvfcmp_sle_s,      "xvfcmp.sle.s",     0,  0x0c938000,0xffff8000,DF_A_3R)
INST(xvfcmp_sule_s,     "xvfcmp.sule.s",    0,  0x0c978000,0xffff8000,DF_A_3R)
INST(xvfcmp_sne_s,      "xvfcmp.sne.s",     0,  0x0c988000,0xffff8000,DF_A_3R)
INST(xvfcmp_sor_s,      "xvfcmp.sor.s",     0,  0x0c9a8000,0xffff8000,DF_A_3R)
INST(xvfcmp_sune_s,     "xvfcmp.sune.s",    0,  0x0c9c8000,0xffff8000,DF_A_3R)
INST(xvfcmp_caf_d,      "xvfcmp.caf.d",     0,  0x0ca00000,0xffff8000,DF_A_3R)
INST(xvfcmp_cun_d,      "xvfcmp.cun.d",     0,  0x0ca40000,0xffff8000,DF_A_3R)
INST(xvfcmp_ceq_d,      "xvfcmp.ceq.d",     0,  0x0ca20000,0xffff8000,DF_A_3R)
INST(xvfcmp_cueq_d,     "xvfcmp.cueq.d",    0,  0x0ca60000,0xffff8000,DF_A_3R)
INST(xvfcmp_clt_d,      "xvfcmp.clt.d",     0,  0x0ca10000,0xffff8000,DF_A_3R)
INST(xvfcmp_cult_d,     "xvfcmp.cult.d",    0,  0x0ca50000,0xffff8000,DF_A_3R)
INST(xvfcmp_cle_d,      "xvfcmp.cle.d",     0,  0x0ca30000,0xffff8000,DF_A_3R)
INST(xvfcmp_cule_d,     "xvfcmp.cule.d",    0,  0x0ca70000,0xffff8000,DF_A_3R)
INST(xvfcmp_cne_d,      "xvfcmp.cne.d",     0,  0x0ca80000,0xffff8000,DF_A_3R)
INST(xvfcmp_cor_d,      "xvfcmp.cor.d",     0,  0x0caa0000,0xffff8000,DF_A_3R)
INST(xvfcmp_cune_d,     "xvfcmp.cune.d",    0,  0x0cac0000,0xffff8000,DF_A_3R)
INST(xvfcmp_saf_d,      "xvfcmp.saf.d",     0,  0x0ca08000,0xffff8000,DF_A_3R)
INST(xvfcmp_sun_d,      "xvfcmp.sun.d",     0,  0x0ca48000,0xffff8000,DF_A_3R)
INST(xvfcmp_seq_d,      "xvfcmp.seq.d",     0,  0x0ca28000,0xffff8000,DF_A_3R)
INST(xvfcmp_sueq_d,     "xvfcmp.sueq.d",    0,  0x0ca68000,0xffff8000,DF_A_3R)
INST(xvfcmp_slt_d,      "xvfcmp.slt.d",     0,  0x0ca18000,0xffff8000,DF_A_3R)
INST(xvfcmp_sult_d,     "xvfcmp.sult.d",    0,  0x0ca58000,0xffff8000,DF_A_3R)
INST(xvfcmp_sle_d,      "xvfcmp.sle.d",     0,  0x0ca38000,0xffff8000,DF_A_3R)
INST(xvfcmp_sule_d,     "xvfcmp.sule.d",    0,  0x0ca78000,0xffff8000,DF_A_3R)
INST(xvfcmp_sne_d,      "xvfcmp.sne.d",     0,  0x0ca88000,0xffff8000,DF_A_3R)
INST(xvfcmp_sor_d,      "xvfcmp.sor.d",     0,  0x0caa8000,0xffff8000,DF_A_3R)
INST(xvfcmp_sune_d,     "xvfcmp.sune.d",    0,  0x0cac8000,0xffff8000,DF_A_3R)
INST(xvseq_b,           "xvseq.b",          0,  0x74000000,0xffff8000,DF_A_3R)
INST(xvseq_h,           "xvseq.h",          0,  0x74008000,0xffff8000,DF_A_3R)
INST(xvseq_w,           "xvseq.w",          0,  0x74010000,0xffff8000,DF_A_3R)
INST(xvseq_d,           "xvseq.d",          0,  0x74018000,0xffff8000,DF_A_3R)
INST(xvsle_b,           "xvsle.b",          0,  0x74020000,0xffff8000,DF_A_3R)
INST(xvsle_h,           "xvsle.h",          0,  0x74028000,0xffff8000,DF_A_3R)
INST(xvsle_w,           "xvsle.w",          0,  0x74030000,0xffff8000,DF_A_3R)
INST(xvsle_d,           "xvsle.d",          0,  0x74038000,0xffff8000,DF_A_3R)
INST(xvsle_bu,          "xvsle.bu",         0,  0x74040000,0xffff8000,DF_A_3R)
INST(xvsle_hu,          "xvsle.hu",         0,  0x74048000,0xffff8000,DF_A_3R)
INST(xvsle_wu,          "xvsle.wu",         0,  0x74050000,0xffff8000,DF_A_3R)
INST(xvsle_du,          "xvsle.du",         0,  0x74058000,0xffff8000,DF_A_3R)
INST(xvslt_b,           "xvslt.b",          0,  0x74060000,0xffff8000,DF_A_3R)
INST(xvslt_h,           "xvslt.h",          0,  0x74068000,0xffff8000,DF_A_3R)
INST(xvslt_w,           "xvslt.w",          0,  0x74070000,0xffff8000,DF_A_3R)
INST(xvslt_d,           "xvslt.d",          0,  0x74078000,0xffff8000,DF_A_3R)
INST(xvslt_bu,          "xvslt.bu",         0,  0x74080000,0xffff8000,DF_A_3R)
INST(xvslt_hu,          "xvslt.hu",         0,  0x74088000,0xffff8000,DF_A_3R)
INST(xvslt_wu,          "xvslt.wu",         0,  0x74090000,0xffff8000,DF_A_3R)
INST(xvslt_du,          "xvslt.du",         0,  0x74098000,0xffff8000,DF_A_3R)
INST(xvadd_b,           "xvadd.b",          0,  0x740a0000,0xffff8000,DF_A_3R)
INST(xvadd_h,           "xvadd.h",          0,  0x740a8000,0xffff8000,DF_A_3R)
INST(xvadd_w,           "xvadd.w",          0,  0x740b0000,0xffff8000,DF_A_3R)
INST(xvadd_d,           "xvadd.d",          0,  0x740b8000,0xffff8000,DF_A_3R)
INST(xvsub_b,           "xvsub.b",          0,  0x740c0000,0xffff8000,DF_A_3R)
INST(xvsub_h,           "xvsub.h",          0,  0x740c8000,0xffff8000,DF_A_3R)
INST(xvsub_w,           "xvsub.w",          0,  0x740d0000,0xffff8000,DF_A_3R)
INST(xvsub_d,           "xvsub.d",          0,  0x740d8000,0xffff8000,DF_A_3R)
INST(xvaddwev_h_b,      "xvaddwev.h.b",     0,  0x741e0000,0xffff8000,DF_A_3R)
INST(xvaddwev_w_h,      "xvaddwev.w.h",     0,  0x741e8000,0xffff8000,DF_A_3R)
INST(xvaddwev_d_w,      "xvaddwev.d.w",     0,  0x741f0000,0xffff8000,DF_A_3R)
INST(xvaddwev_q_d,      "xvaddwev.q.d",     0,  0x741f8000,0xffff8000,DF_A_3R)
INST(xvsubwev_h_b,      "xvsubwev.h.b",     0,  0x74200000,0xffff8000,DF_A_3R)
INST(xvsubwev_w_h,      "xvsubwev.w.h",     0,  0x74208000,0xffff8000,DF_A_3R)
INST(xvsubwev_d_w,      "xvsubwev.d.w",     0,  0x74210000,0xffff8000,DF_A_3R)
INST(xvsubwev_q_d,      "xvsubwev.q.d",     0,  0x74218000,0xffff8000,DF_A_3R)
INST(xvaddwod_h_b,      "xvaddwod.h.b",     0,  0x74220000,0xffff8000,DF_A_3R)
INST(xvaddwod_w_h,      "xvaddwod.w.h",     0,  0x74228000,0xffff8000,DF_A_3R)
INST(xvaddwod_d_w,      "xvaddwod.d.w",     0,  0x74230000,0xffff8000,DF_A_3R)
INST(xvaddwod_q_d,      "xvaddwod.q.d",     0,  0x74238000,0xffff8000,DF_A_3R)
INST(xvsubwod_h_b,      "xvsubwod.h.b",     0,  0x74240000,0xffff8000,DF_A_3R)
INST(xvsubwod_w_h,      "xvsubwod.w.h",     0,  0x74248000,0xffff8000,DF_A_3R)
INST(xvsubwod_d_w,      "xvsubwod.d.w",     0,  0x74250000,0xffff8000,DF_A_3R)
INST(xvsubwod_q_d,      "xvsubwod.q.d",     0,  0x74258000,0xffff8000,DF_A_3R)
INST(xvaddwev_h_bu,     "xvaddwev.h.bu",    0,  0x742e0000,0xffff8000,DF_A_3R)
INST(xvaddwev_w_hu,     "xvaddwev.w.hu",    0,  0x742e8000,0xffff8000,DF_A_3R)
INST(xvaddwev_d_wu,     "xvaddwev.d.wu",    0,  0x742f0000,0xffff8000,DF_A_3R)
INST(xvaddwev_q_du,     "xvaddwev.q.du",    0,  0x742f8000,0xffff8000,DF_A_3R)
INST(xvsubwev_h_bu,     "xvsubwev.h.bu",    0,  0x74300000,0xffff8000,DF_A_3R)
INST(xvsubwev_w_hu,     "xvsubwev.w.hu",    0,  0x74308000,0xffff8000,DF_A_3R)
INST(xvsubwev_d_wu,     "xvsubwev.d.wu",    0,  0x74310000,0xffff8000,DF_A_3R)
INST(xvsubwev_q_du,     "xvsubwev.q.du",    0,  0x74318000,0xffff8000,DF_A_3R)
INST(xvaddwod_h_bu,     "xvaddwod.h.bu",    0,  0x74320000,0xffff8000,DF_A_3R)
INST(xvaddwod_w_hu,     "xvaddwod.w.hu",    0,  0x74328000,0xffff8000,DF_A_3R)
INST(xvaddwod_d_wu,     "xvaddwod.d.wu",    0,  0x74330000,0xffff8000,DF_A_3R)
INST(xvaddwod_q_du,     "xvaddwod.q.du",    0,  0x74338000,0xffff8000,DF_A_3R)
INST(xvsubwod_h_bu,     "xvsubwod.h.bu",    0,  0x74340000,0xffff8000,DF_A_3R)
INST(xvsubwod_w_hu,     "xvsubwod.w.hu",    0,  0x74348000,0xffff8000,DF_A_3R)
INST(xvsubwod_d_wu,     "xvsubwod.d.wu",    0,  0x74350000,0xffff8000,DF_A_3R)
INST(xvsubwod_q_du,     "xvsubwod.q.du",    0,  0x74358000,0xffff8000,DF_A_3R)
INST(xvaddwev_h_bu_b,   "xvaddwev.h.bu.b",  0,  0x743e0000,0xffff8000,DF_A_3R)
INST(xvaddwev_w_hu_h,   "xvaddwev.w.hu.h",  0,  0x743e8000,0xffff8000,DF_A_3R)
INST(xvaddwev_d_wu_w,   "xvaddwev.d.wu.w",  0,  0x743f0000,0xffff8000,DF_A_3R)
INST(xvaddwev_q_du_d,   "xvaddwev.q.du.d",  0,  0x743f8000,0xffff8000,DF_A_3R)
INST(xvaddwod_h_bu_b,   "xvaddwod.h.bu.b",  0,  0x74400000,0xffff8000,DF_A_3R)
INST(xvaddwod_w_hu_h,   "xvaddwod.w.hu.h",  0,  0x74408000,0xffff8000,DF_A_3R)
INST(xvaddwod_d_wu_w,   "xvaddwod.d.wu.w",  0,  0x74410000,0xffff8000,DF_A_3R)
INST(xvaddwod_q_du_d,   "xvaddwod.q.du.d",  0,  0x74418000,0xffff8000,DF_A_3R)
INST(xvsadd_b,          "xvsadd.b",         0,  0x74460000,0xffff8000,DF_A_3R)
INST(xvsadd_h,          "xvsadd.h",         0,  0x74468000,0xffff8000,DF_A_3R)
INST(xvsadd_w,          "xvsadd.w",         0,  0x74470000,0xffff8000,DF_A_3R)
INST(xvsadd_d,          "xvsadd.d",         0,  0x74478000,0xffff8000,DF_A_3R)
INST(xvssub_b,          "xvssub.b",         0,  0x74480000,0xffff8000,DF_A_3R)
INST(xvssub_h,          "xvssub.h",         0,  0x74488000,0xffff8000,DF_A_3R)
INST(xvssub_w,          "xvssub.w",         0,  0x74490000,0xffff8000,DF_A_3R)
INST(xvssub_d,          "xvssub.d",         0,  0x74498000,0xffff8000,DF_A_3R)
INST(xvsadd_bu,         "xvsadd.bu",        0,  0x744a0000,0xffff8000,DF_A_3R)
INST(xvsadd_hu,         "xvsadd.hu",        0,  0x744a8000,0xffff8000,DF_A_3R)
INST(xvsadd_wu,         "xvsadd.wu",        0,  0x744b0000,0xffff8000,DF_A_3R)
INST(xvsadd_du,         "xvsadd.du",        0,  0x744b8000,0xffff8000,DF_A_3R)
INST(xvssub_bu,         "xvssub.bu",        0,  0x744c0000,0xffff8000,DF_A_3R)
INST(xvssub_hu,         "xvssub.hu",        0,  0x744c8000,0xffff8000,DF_A_3R)
INST(xvssub_wu,         "xvssub.wu",        0,  0x744d0000,0xffff8000,DF_A_3R)
INST(xvssub_du,         "xvssub.du",        0,  0x744d8000,0xffff8000,DF_A_3R)
INST(xvhaddw_h_b,       "xvhaddw.h.b",      0,  0x74540000,0xffff8000,DF_A_3R)
INST(xvhaddw_w_h,       "xvhaddw.w.h",      0,  0x74548000,0xffff8000,DF_A_3R)
INST(xvhaddw_d_w,       "xvhaddw.d.w",      0,  0x74550000,0xffff8000,DF_A_3R)
INST(xvhaddw_q_d,       "xvhaddw.q.d",      0,  0x74558000,0xffff8000,DF_A_3R)
INST(xvhsubw_h_b,       "xvhsubw.h.b",      0,  0x74560000,0xffff8000,DF_A_3R)
INST(xvhsubw_w_h,       "xvhsubw.w.h",      0,  0x74568000,0xffff8000,DF_A_3R)
INST(xvhsubw_d_w,       "xvhsubw.d.w",      0,  0x74570000,0xffff8000,DF_A_3R)
INST(xvhsubw_q_d,       "xvhsubw.q.d",      0,  0x74578000,0xffff8000,DF_A_3R)
INST(xvhaddw_hu_bu,     "xvhaddw.hu.bu",    0,  0x74580000,0xffff8000,DF_A_3R)
INST(xvhaddw_wu_hu,     "xvhaddw.wu.hu",    0,  0x74588000,0xffff8000,DF_A_3R)
INST(xvhaddw_du_wu,     "xvhaddw.du.wu",    0,  0x74590000,0xffff8000,DF_A_3R)
INST(xvhaddw_qu_du,     "xvhaddw.qu.du",    0,  0x74598000,0xffff8000,DF_A_3R)
INST(xvhsubw_hu_bu,     "xvhsubw.hu.bu",    0,  0x745a0000,0xffff8000,DF_A_3R)
INST(xvhsubw_wu_hu,     "xvhsubw.wu.hu",    0,  0x745a8000,0xffff8000,DF_A_3R)
INST(xvhsubw_du_wu,     "xvhsubw.du.wu",    0,  0x745b0000,0xffff8000,DF_A_3R)
INST(xvhsubw_qu_du,     "xvhsubw.qu.du",    0,  0x745b8000,0xffff8000,DF_A_3R)
INST(xvadda_b,          "xvadda.b",         0,  0x745c0000,0xffff8000,DF_A_3R)
INST(xvadda_h,          "xvadda.h",         0,  0x745c8000,0xffff8000,DF_A_3R)
INST(xvadda_w,          "xvadda.w",         0,  0x745d0000,0xffff8000,DF_A_3R)
INST(xvadda_d,          "xvadda.d",         0,  0x745d8000,0xffff8000,DF_A_3R)
INST(xvabsd_b,          "xvabsd.b",         0,  0x74600000,0xffff8000,DF_A_3R)
INST(xvabsd_h,          "xvabsd.h",         0,  0x74608000,0xffff8000,DF_A_3R)
INST(xvabsd_w,          "xvabsd.w",         0,  0x74610000,0xffff8000,DF_A_3R)
INST(xvabsd_d,          "xvabsd.d",         0,  0x74618000,0xffff8000,DF_A_3R)
INST(xvabsd_bu,         "xvabsd.bu",        0,  0x74620000,0xffff8000,DF_A_3R)
INST(xvabsd_hu,         "xvabsd.hu",        0,  0x74628000,0xffff8000,DF_A_3R)
INST(xvabsd_wu,         "xvabsd.wu",        0,  0x74630000,0xffff8000,DF_A_3R)
INST(xvabsd_du,         "xvabsd.du",        0,  0x74638000,0xffff8000,DF_A_3R)
INST(xvavg_b,           "xvavg.b",          0,  0x74640000,0xffff8000,DF_A_3R)
INST(xvavg_h,           "xvavg.h",          0,  0x74648000,0xffff8000,DF_A_3R)
INST(xvavg_w,           "xvavg.w",          0,  0x74650000,0xffff8000,DF_A_3R)
INST(xvavg_d,           "xvavg.d",          0,  0x74658000,0xffff8000,DF_A_3R)
INST(xvavg_bu,          "xvavg.bu",         0,  0x74660000,0xffff8000,DF_A_3R)
INST(xvavg_hu,          "xvavg.hu",         0,  0x74668000,0xffff8000,DF_A_3R)
INST(xvavg_wu,          "xvavg.wu",         0,  0x74670000,0xffff8000,DF_A_3R)
INST(xvavg_du,          "xvavg.du",         0,  0x74678000,0xffff8000,DF_A_3R)
INST(xvavgr_b,          "xvavgr.b",         0,  0x74680000,0xffff8000,DF_A_3R)
INST(xvavgr_h,          "xvavgr.h",         0,  0x74688000,0xffff8000,DF_A_3R)
INST(xvavgr_w,          "xvavgr.w",         0,  0x74690000,0xffff8000,DF_A_3R)
INST(xvavgr_d,          "xvavgr.d",         0,  0x74698000,0xffff8000,DF_A_3R)
INST(xvavgr_bu,         "xvavgr.bu",        0,  0x746a0000,0xffff8000,DF_A_3R)
INST(xvavgr_hu,         "xvavgr.hu",        0,  0x746a8000,0xffff8000,DF_A_3R)
INST(xvavgr_wu,         "xvavgr.wu",        0,  0x746b0000,0xffff8000,DF_A_3R)
INST(xvavgr_du,         "xvavgr.du",        0,  0x746b8000,0xffff8000,DF_A_3R)
INST(xvmax_b,           "xvmax.b",          0,  0x74700000,0xffff8000,DF_A_3R)
INST(xvmax_h,           "xvmax.h",          0,  0x74708000,0xffff8000,DF_A_3R)
INST(xvmax_w,           "xvmax.w",          0,  0x74710000,0xffff8000,DF_A_3R)
INST(xvmax_d,           "xvmax.d",          0,  0x74718000,0xffff8000,DF_A_3R)
INST(xvmin_b,           "xvmin.b",          0,  0x74720000,0xffff8000,DF_A_3R)
INST(xvmin_h,           "xvmin.h",          0,  0x74728000,0xffff8000,DF_A_3R)
INST(xvmin_w,           "xvmin.w",          0,  0x74730000,0xffff8000,DF_A_3R)
INST(xvmin_d,           "xvmin.d",          0,  0x74738000,0xffff8000,DF_A_3R)
INST(xvmax_bu,          "xvmax.bu",         0,  0x74740000,0xffff8000,DF_A_3R)
INST(xvmax_hu,          "xvmax.hu",         0,  0x74748000,0xffff8000,DF_A_3R)
INST(xvmax_wu,          "xvmax.wu",         0,  0x74750000,0xffff8000,DF_A_3R)
INST(xvmax_du,          "xvmax.du",         0,  0x74758000,0xffff8000,DF_A_3R)
INST(xvmin_bu,          "xvmin.bu",         0,  0x74760000,0xffff8000,DF_A_3R)
INST(xvmin_hu,          "xvmin.hu",         0,  0x74768000,0xffff8000,DF_A_3R)
INST(xvmin_wu,          "xvmin.wu",         0,  0x74770000,0xffff8000,DF_A_3R)
INST(xvmin_du,          "xvmin.du",         0,  0x74778000,0xffff8000,DF_A_3R)
INST(xvmul_b,           "xvmul.b",          0,  0x74840000,0xffff8000,DF_A_3R)
INST(xvmul_h,           "xvmul.h",          0,  0x74848000,0xffff8000,DF_A_3R)
INST(xvmul_w,           "xvmul.w",          0,  0x74850000,0xffff8000,DF_A_3R)
INST(xvmul_d,           "xvmul.d",          0,  0x74858000,0xffff8000,DF_A_3R)
INST(xvmuh_b,           "xvmuh.b",          0,  0x74860000,0xffff8000,DF_A_3R)
INST(xvmuh_h,           "xvmuh.h",          0,  0x74868000,0xffff8000,DF_A_3R)
INST(xvmuh_w,           "xvmuh.w",          0,  0x74870000,0xffff8000,DF_A_3R)
INST(xvmuh_d,           "xvmuh.d",          0,  0x74878000,0xffff8000,DF_A_3R)
INST(xvmuh_bu,          "xvmuh.bu",         0,  0x74880000,0xffff8000,DF_A_3R)
INST(xvmuh_hu,          "xvmuh.hu",         0,  0x74888000,0xffff8000,DF_A_3R)
INST(xvmuh_wu,          "xvmuh.wu",         0,  0x74890000,0xffff8000,DF_A_3R)
INST(xvmuh_du,          "xvmuh.du",         0,  0x74898000,0xffff8000,DF_A_3R)
INST(xvmulwev_h_b,      "xvmulwev.h.b",     0,  0x74900000,0xffff8000,DF_A_3R)
INST(xvmulwev_w_h,      "xvmulwev.w.h",     0,  0x74908000,0xffff8000,DF_A_3R)
INST(xvmulwev_d_w,      "xvmulwev.d.w",     0,  0x74910000,0xffff8000,DF_A_3R)
INST(xvmulwev_q_d,      "xvmulwev.q.d",     0,  0x74918000,0xffff8000,DF_A_3R)
INST(xvmulwod_h_b,      "xvmulwod.h.b",     0,  0x74920000,0xffff8000,DF_A_3R)
INST(xvmulwod_w_h,      "xvmulwod.w.h",     0,  0x74928000,0xffff8000,DF_A_3R)
INST(xvmulwod_d_w,      "xvmulwod.d.w",     0,  0x74930000,0xffff8000,DF_A_3R)
INST(xvmulwod_q_d,      "xvmulwod.q.d",     0,  0x74938000,0xffff8000,DF_A_3R)
INST(xvmulwev_h_bu,     "xvmulwev.h.bu",    0,  0x74980000,0xffff8000,DF_A_3R)
INST(xvmulwev_w_hu,     "xvmulwev.w.hu",    0,  0x74988000,0xffff8000,DF_A_3R)
INST(xvmulwev_d_wu,     "xvmulwev.d.wu",    0,  0x74990000,0xffff8000,DF_A_3R)
INST(xvmulwev_q_du,     "xvmulwev.q.du",    0,  0x74998000,0xffff8000,DF_A_3R)
INST(xvmulwod_h_bu,     "xvmulwod.h.bu",    0,  0x749a0000,0xffff8000,DF_A_3R)
INST(xvmulwod_w_hu,     "xvmulwod.w.hu",    0,  0x749a8000,0xffff8000,DF_A_3R)
INST(xvmulwod_d_wu,     "xvmulwod.d.wu",    0,  0x749b0000,0xffff8000,DF_A_3R)
INST(xvmulwod_q_du,     "xvmulwod.q.du",    0,  0x749b8000,0xffff8000,DF_A_3R)
INST(xvmulwev_h_bu_b,   "xvmulwev.h.bu.b",  0,  0x74a00000,0xffff8000,DF_A_3R)
INST(xvmulwev_w_hu_h,   "xvmulwev.w.hu.h",  0,  0x74a08000,0xffff8000,DF_A_3R)
INST(xvmulwev_d_wu_w,   "xvmulwev.d.wu.w",  0,  0x74a10000,0xffff8000,DF_A_3R)
INST(xvmulwev_q_du_d,   "xvmulwev.q.du.d",  0,  0x74a18000,0xffff8000,DF_A_3R)
INST(xvmulwod_h_bu_b,   "xvmulwod.h.bu.b",  0,  0x74a20000,0xffff8000,DF_A_3R)
INST(xvmulwod_w_hu_h,   "xvmulwod.w.hu.h",  0,  0x74a28000,0xffff8000,DF_A_3R)
INST(xvmulwod_d_wu_w,   "xvmulwod.d.wu.w",  0,  0x74a30000,0xffff8000,DF_A_3R)
INST(xvmulwod_q_du_d,   "xvmulwod.q.du.d",  0,  0x74a38000,0xffff8000,DF_A_3R)
INST(xvmadd_b,          "xvmadd.b",         0,  0x74a80000,0xffff8000,DF_A_3R)
INST(xvmadd_h,          "xvmadd.h",         0,  0x74a88000,0xffff8000,DF_A_3R)
INST(xvmadd_w,          "xvmadd.w",         0,  0x74a90000,0xffff8000,DF_A_3R)
INST(xvmadd_d,          "xvmadd.d",         0,  0x74a98000,0xffff8000,DF_A_3R)
INST(xvmsub_b,          "xvmsub.b",         0,  0x74aa0000,0xffff8000,DF_A_3R)
INST(xvmsub_h,          "xvmsub.h",         0,  0x74aa8000,0xffff8000,DF_A_3R)
INST(xvmsub_w,          "xvmsub.w",         0,  0x74ab0000,0xffff8000,DF_A_3R)
INST(xvmsub_d,          "xvmsub.d",         0,  0x74ab8000,0xffff8000,DF_A_3R)
INST(xvmaddwev_h_b,     "xvmaddwev.h.b",    0,  0x74ac0000,0xffff8000,DF_A_3R)
INST(xvmaddwev_w_h,     "xvmaddwev.w.h",    0,  0x74ac8000,0xffff8000,DF_A_3R)
INST(xvmaddwev_d_w,     "xvmaddwev.d.w",    0,  0x74ad0000,0xffff8000,DF_A_3R)
INST(xvmaddwev_q_d,     "xvmaddwev.q.d",    0,  0x74ad8000,0xffff8000,DF_A_3R)
INST(xvmaddwod_h_b,     "xvmaddwod.h.b",    0,  0x74ae0000,0xffff8000,DF_A_3R)
INST(xvmaddwod_w_h,     "xvmaddwod.w.h",    0,  0x74ae8000,0xffff8000,DF_A_3R)
INST(xvmaddwod_d_w,     "xvmaddwod.d.w",    0,  0x74af0000,0xffff8000,DF_A_3R)
INST(xvmaddwod_q_d,     "xvmaddwod.q.d",    0,  0x74af8000,0xffff8000,DF_A_3R)
INST(xvmaddwev_h_bu,    "xvmaddwev.h.bu",   0,  0x74b40000,0xffff8000,DF_A_3R)
INST(xvmaddwev_w_hu,    "xvmaddwev.w.hu",   0,  0x74b48000,0xffff8000,DF_A_3R)
INST(xvmaddwev_d_wu,    "xvmaddwev.d.wu",   0,  0x74b50000,0xffff8000,DF_A_3R)
INST(xvmaddwev_q_du,    "xvmaddwev.q.du",   0,  0x74b58000,0xffff8000,DF_A_3R)
INST(xvmaddwod_h_bu,    "xvmaddwod.h.bu",   0,  0x74b60000,0xffff8000,DF_A_3R)
INST(xvmaddwod_w_hu,    "xvmaddwod.w.hu",   0,  0x74b68000,0xffff8000,DF_A_3R)
INST(xvmaddwod_d_wu,    "xvmaddwod.d.wu",   0,  0x74b70000,0xffff8000,DF_A_3R)
INST(xvmaddwod_q_du,    "xvmaddwod.q.du",   0,  0x74b78000,0xffff8000,DF_A_3R)
INST(xvmaddwev_h_bu_b,  "xvmaddwev.h.bu.b", 0,  0x74bc0000,0xffff8000,DF_A_3R)
INST(xvmaddwev_w_hu_h,  "xvmaddwev.w.hu.h", 0,  0x74bc8000,0xffff8000,DF_A_3R)
INST(xvmaddwev_d_wu_w,  "xvmaddwev.d.wu.w", 0,  0x74bd0000,0xffff8000,DF_A_3R)
INST(xvmaddwev_q_du_d,  "xvmaddwev.q.du.d", 0,  0x74bd8000,0xffff8000,DF_A_3R)
INST(xvmaddwod_h_bu_b,  "xvmaddwod.h.bu.b", 0,  0x74be0000,0xffff8000,DF_A_3R)
INST(xvmaddwod_w_hu_h,  "xvmaddwod.w.hu.h", 0,  0x74be8000,0xffff8000,DF_A_3R)
INST(xvmaddwod_d_wu_w,  "xvmaddwod.d.wu.w", 0,  0x74bf0000,0xffff8000,DF_A_3R)
INST(xvmaddwod_q_du_d,  "xvmaddwod.q.du.d", 0,  0x74bf8000,0xffff8000,DF_A_3R)
INST(xvdiv_b,           "xvdiv.b",          0,  0x74e00000,0xffff8000,DF_A_3R)
INST(xvdiv_h,           "xvdiv.h",          0,  0x74e08000,0xffff8000,DF_A_3R)
INST(xvdiv_w,           "xvdiv.w",          0,  0x74e10000,0xffff8000,DF_A_3R)
INST(xvdiv_d,           "xvdiv.d",          0,  0x74e18000,0xffff8000,DF_A_3R)
INST(xvmod_b,           "xvmod.b",          0,  0x74e20000,0xffff8000,DF_A_3R)
INST(xvmod_h,           "xvmod.h",          0,  0x74e28000,0xffff8000,DF_A_3R)
INST(xvmod_w,           "xvmod.w",          0,  0x74e30000,0xffff8000,DF_A_3R)
INST(xvmod_d,           "xvmod.d",          0,  0x74e38000,0xffff8000,DF_A_3R)
INST(xvdiv_bu,          "xvdiv.bu",         0,  0x74e40000,0xffff8000,DF_A_3R)
INST(xvdiv_hu,          "xvdiv.hu",         0,  0x74e48000,0xffff8000,DF_A_3R)
INST(xvdiv_wu,          "xvdiv.wu",         0,  0x74e50000,0xffff8000,DF_A_3R)
INST(xvdiv_du,          "xvdiv.du",         0,  0x74e58000,0xffff8000,DF_A_3R)
INST(xvmod_bu,          "xvmod.bu",         0,  0x74e60000,0xffff8000,DF_A_3R)
INST(xvmod_hu,          "xvmod.hu",         0,  0x74e68000,0xffff8000,DF_A_3R)
INST(xvmod_wu,          "xvmod.wu",         0,  0x74e70000,0xffff8000,DF_A_3R)
INST(xvmod_du,          "xvmod.du",         0,  0x74e78000,0xffff8000,DF_A_3R)
INST(xvsll_b,           "xvsll.b",          0,  0x74e80000,0xffff8000,DF_A_3R)
INST(xvsll_h,           "xvsll.h",          0,  0x74e88000,0xffff8000,DF_A_3R)
INST(xvsll_w,           "xvsll.w",          0,  0x74e90000,0xffff8000,DF_A_3R)
INST(xvsll_d,           "xvsll.d",          0,  0x74e98000,0xffff8000,DF_A_3R)
INST(xvsrl_b,           "xvsrl.b",          0,  0x74ea0000,0xffff8000,DF_A_3R)
INST(xvsrl_h,           "xvsrl.h",          0,  0x74ea8000,0xffff8000,DF_A_3R)
INST(xvsrl_w,           "xvsrl.w",          0,  0x74eb0000,0xffff8000,DF_A_3R)
INST(xvsrl_d,           "xvsrl.d",          0,  0x74eb8000,0xffff8000,DF_A_3R)
INST(xvsra_b,           "xvsra.b",          0,  0x74ec0000,0xffff8000,DF_A_3R)
INST(xvsra_h,           "xvsra.h",          0,  0x74ec8000,0xffff8000,DF_A_3R)
INST(xvsra_w,           "xvsra.w",          0,  0x74ed0000,0xffff8000,DF_A_3R)
INST(xvsra_d,           "xvsra.d",          0,  0x74ed8000,0xffff8000,DF_A_3R)
INST(xvrotr_b,          "xvrotr.b",         0,  0x74ee0000,0xffff8000,DF_A_3R)
INST(xvrotr_h,          "xvrotr.h",         0,  0x74ee8000,0xffff8000,DF_A_3R)
INST(xvrotr_w,          "xvrotr.w",         0,  0x74ef0000,0xffff8000,DF_A_3R)
INST(xvrotr_d,          "xvrotr.d",         0,  0x74ef8000,0xffff8000,DF_A_3R)
INST(xvsrlr_b,          "xvsrlr.b",         0,  0x74f00000,0xffff8000,DF_A_3R)
INST(xvsrlr_h,          "xvsrlr.h",         0,  0x74f08000,0xffff8000,DF_A_3R)
INST(xvsrlr_w,          "xvsrlr.w",         0,  0x74f10000,0xffff8000,DF_A_3R)
INST(xvsrlr_d,          "xvsrlr.d",         0,  0x74f18000,0xffff8000,DF_A_3R)
INST(xvsrar_b,          "xvsrar.b",         0,  0x74f20000,0xffff8000,DF_A_3R)
INST(xvsrar_h,          "xvsrar.h",         0,  0x74f28000,0xffff8000,DF_A_3R)
INST(xvsrar_w,          "xvsrar.w",         0,  0x74f30000,0xffff8000,DF_A_3R)
INST(xvsrar_d,          "xvsrar.d",         0,  0x74f38000,0xffff8000,DF_A_3R)
INST(xvsrln_b_h,        "xvsrln.b.h",       0,  0x74f48000,0xffff8000,DF_A_3R)
INST(xvsrln_h_w,        "xvsrln.h.w",       0,  0x74f50000,0xffff8000,DF_A_3R)
INST(xvsrln_w_d,        "xvsrln.w.d",       0,  0x74f58000,0xffff8000,DF_A_3R)
INST(xvsran_b_h,        "xvsran.b.h",       0,  0x74f68000,0xffff8000,DF_A_3R)
INST(xvsran_h_w,        "xvsran.h.w",       0,  0x74f70000,0xffff8000,DF_A_3R)
INST(xvsran_w_d,        "xvsran.w.d",       0,  0x74f78000,0xffff8000,DF_A_3R)
INST(xvsrlrn_b_h,       "xvsrlrn.b.h",      0,  0x74f88000,0xffff8000,DF_A_3R)
INST(xvsrlrn_h_w,       "xvsrlrn.h.w",      0,  0x74f90000,0xffff8000,DF_A_3R)
INST(xvsrlrn_w_d,       "xvsrlrn.w.d",      0,  0x74f98000,0xffff8000,DF_A_3R)
INST(xvsrarn_b_h,       "xvsrarn.b.h",      0,  0x74fa8000,0xffff8000,DF_A_3R)
INST(xvsrarn_h_w,       "xvsrarn.h.w",      0,  0x74fb0000,0xffff8000,DF_A_3R)
INST(xvsrarn_w_d,       "xvsrarn.w.d",      0,  0x74fb8000,0xffff8000,DF_A_3R)
INST(xvssrln_b_h,       "xvssrln.b.h",      0,  0x74fc8000,0xffff8000,DF_A_3R)
INST(xvssrln_h_w,       "xvssrln.h.w",      0,  0x74fd0000,0xffff8000,DF_A_3R)
INST(xvssrln_w_d,       "xvssrln.w.d",      0,  0x74fd8000,0xffff8000,DF_A_3R)
INST(xvssran_b_h,       "xvssran.b.h",      0,  0x74fe8000,0xffff8000,DF_A_3R)
INST(xvssran_h_w,       "xvssran.h.w",      0,  0x74ff0000,0xffff8000,DF_A_3R)
INST(xvssran_w_d,       "xvssran.w.d",      0,  0x74ff8000,0xffff8000,DF_A_3R)
INST(xvssrlrn_b_h,      "xvssrlrn.b.h",     0,  0x75008000,0xffff8000,DF_A_3R)
INST(xvssrlrn_h_w,      "xvssrlrn.h.w",     0,  0x75010000,0xffff8000,DF_A_3R)
INST(xvssrlrn_w_d,      "xvssrlrn.w.d",     0,  0x75018000,0xffff8000,DF_A_3R)
INST(xvssrarn_b_h,      "xvssrarn.b.h",     0,  0x75028000,0xffff8000,DF_A_3R)
INST(xvssrarn_h_w,      "xvssrarn.h.w",     0,  0x75030000,0xffff8000,DF_A_3R)
INST(xvssrarn_w_d,      "xvssrarn.w.d",     0,  0x75038000,0xffff8000,DF_A_3R)
INST(xvssrln_bu_h,      "xvssrln.bu.h",     0,  0x75048000,0xffff8000,DF_A_3R)
INST(xvssrln_hu_w,      "xvssrln.hu.w",     0,  0x75050000,0xffff8000,DF_A_3R)
INST(xvssrln_wu_d,      "xvssrln.wu.d",     0,  0x75058000,0xffff8000,DF_A_3R)
INST(xvssran_bu_h,      "xvssran.bu.h",     0,  0x75068000,0xffff8000,DF_A_3R)
INST(xvssran_hu_w,      "xvssran.hu.w",     0,  0x75070000,0xffff8000,DF_A_3R)
INST(xvssran_wu_d,      "xvssran.wu.d",     0,  0x75078000,0xffff8000,DF_A_3R)
INST(xvssrlrn_bu_h,     "xvssrlrn.bu.h",    0,  0x75088000,0xffff8000,DF_A_3R)
INST(xvssrlrn_hu_w,     "xvssrlrn.hu.w",    0,  0x75090000,0xffff8000,DF_A_3R)
INST(xvssrlrn_wu_d,     "xvssrlrn.wu.d",    0,  0x75098000,0xffff8000,DF_A_3R)
INST(xvssrarn_bu_h,     "xvssrarn.bu.h",    0,  0x750a8000,0xffff8000,DF_A_3R)
INST(xvssrarn_hu_w,     "xvssrarn.hu.w",    0,  0x750b0000,0xffff8000,DF_A_3R)
INST(xvssrarn_wu_d,     "xvssrarn.wu.d",    0,  0x750b8000,0xffff8000,DF_A_3R)
INST(xvbitclr_b,        "xvbitclr.b",       0,  0x750c0000,0xffff8000,DF_A_3R)
INST(xvbitclr_h,        "xvbitclr.h",       0,  0x750c8000,0xffff8000,DF_A_3R)
INST(xvbitclr_w,        "xvbitclr.w",       0,  0x750d0000,0xffff8000,DF_A_3R)
INST(xvbitclr_d,        "xvbitclr.d",       0,  0x750d8000,0xffff8000,DF_A_3R)
INST(xvbitset_b,        "xvbitset.b",       0,  0x750e0000,0xffff8000,DF_A_3R)
INST(xvbitset_h,        "xvbitset.h",       0,  0x750e8000,0xffff8000,DF_A_3R)
INST(xvbitset_w,        "xvbitset.w",       0,  0x750f0000,0xffff8000,DF_A_3R)
INST(xvbitset_d,        "xvbitset.d",       0,  0x750f8000,0xffff8000,DF_A_3R)
INST(xvbitrev_b,        "xvbitrev.b",       0,  0x75100000,0xffff8000,DF_A_3R)
INST(xvbitrev_h,        "xvbitrev.h",       0,  0x75108000,0xffff8000,DF_A_3R)
INST(xvbitrev_w,        "xvbitrev.w",       0,  0x75110000,0xffff8000,DF_A_3R)
INST(xvbitrev_d,        "xvbitrev.d",       0,  0x75118000,0xffff8000,DF_A_3R)
INST(xvpackev_b,        "xvpackev.b",       0,  0x75160000,0xffff8000,DF_A_3R)
INST(xvpackev_h,        "xvpackev.h",       0,  0x75168000,0xffff8000,DF_A_3R)
INST(xvpackev_w,        "xvpackev.w",       0,  0x75170000,0xffff8000,DF_A_3R)
INST(xvpackev_d,        "xvpackev.d",       0,  0x75178000,0xffff8000,DF_A_3R)
INST(xvpackod_b,        "xvpackod.b",       0,  0x75180000,0xffff8000,DF_A_3R)
INST(xvpackod_h,        "xvpackod.h",       0,  0x75188000,0xffff8000,DF_A_3R)
INST(xvpackod_w,        "xvpackod.w",       0,  0x75190000,0xffff8000,DF_A_3R)
INST(xvpackod_d,        "xvpackod.d",       0,  0x75198000,0xffff8000,DF_A_3R)
INST(xvilvl_b,          "xvilvl.b",         0,  0x751a0000,0xffff8000,DF_A_3R)
INST(xvilvl_h,          "xvilvl.h",         0,  0x751a8000,0xffff8000,DF_A_3R)
INST(xvilvl_w,          "xvilvl.w",         0,  0x751b0000,0xffff8000,DF_A_3R)
INST(xvilvl_d,          "xvilvl.d",         0,  0x751b8000,0xffff8000,DF_A_3R)
INST(xvilvh_b,          "xvilvh.b",         0,  0x751c0000,0xffff8000,DF_A_3R)
INST(xvilvh_h,          "xvilvh.h",         0,  0x751c8000,0xffff8000,DF_A_3R)
INST(xvilvh_w,          "xvilvh.w",         0,  0x751d0000,0xffff8000,DF_A_3R)
INST(xvilvh_d,          "xvilvh.d",         0,  0x751d8000,0xffff8000,DF_A_3R)
INST(xvpickev_b,        "xvpickev.b",       0,  0x751e0000,0xffff8000,DF_A_3R)
INST(xvpickev_h,        "xvpickev.h",       0,  0x751e8000,0xffff8000,DF_A_3R)
INST(xvpickev_w,        "xvpickev.w",       0,  0x751f0000,0xffff8000,DF_A_3R)
INST(xvpickev_d,        "xvpickev.d",       0,  0x751f8000,0xffff8000,DF_A_3R)
INST(xvpickod_b,        "xvpickod.b",       0,  0x75200000,0xffff8000,DF_A_3R)
INST(xvpickod_h,        "xvpickod.h",       0,  0x75208000,0xffff8000,DF_A_3R)
INST(xvpickod_w,        "xvpickod.w",       0,  0x75210000,0xffff8000,DF_A_3R)
INST(xvpickod_d,        "xvpickod.d",       0,  0x75218000,0xffff8000,DF_A_3R)
INST(xvand_v,           "xvand.v",          0,  0x75260000,0xffff8000,DF_A_3R)
INST(xvor_v,            "xvor.v",           0,  0x75268000,0xffff8000,DF_A_3R)
INST(xvxor_v,           "xvxor.v",          0,  0x75270000,0xffff8000,DF_A_3R)
INST(xvnor_v,           "xvnor.v",          0,  0x75278000,0xffff8000,DF_A_3R)
INST(xvandn_v,          "xvandn.v",         0,  0x75280000,0xffff8000,DF_A_3R)
INST(xvorn_v,           "xvorn.v",          0,  0x75288000,0xffff8000,DF_A_3R)
INST(xvfrstp_b,         "xvfrstp.b",        0,  0x752b0000,0xffff8000,DF_A_3R)
INST(xvfrstp_h,         "xvfrstp.h",        0,  0x752b8000,0xffff8000,DF_A_3R)
INST(xvadd_q,           "xvadd.q",          0,  0x752d0000,0xffff8000,DF_A_3R)
INST(xvsub_q,           "xvsub.q",          0,  0x752d8000,0xffff8000,DF_A_3R)
INST(xvsigncov_b,       "xvsigncov.b",      0,  0x752e0000,0xffff8000,DF_A_3R)
INST(xvsigncov_h,       "xvsigncov.h",      0,  0x752e8000,0xffff8000,DF_A_3R)
INST(xvsigncov_w,       "xvsigncov.w",      0,  0x752f0000,0xffff8000,DF_A_3R)
INST(xvsigncov_d,       "xvsigncov.d",      0,  0x752f8000,0xffff8000,DF_A_3R)
INST(xvfadd_s,          "xvfadd.s",         0,  0x75308000,0xffff8000,DF_A_3R)
INST(xvfadd_d,          "xvfadd.d",         0,  0x75310000,0xffff8000,DF_A_3R)
INST(xvfsub_s,          "xvfsub.s",         0,  0x75328000,0xffff8000,DF_A_3R)
INST(xvfsub_d,          "xvfsub.d",         0,  0x75330000,0xffff8000,DF_A_3R)
INST(xvfmul_s,          "xvfmul.s",         0,  0x75388000,0xffff8000,DF_A_3R)
INST(xvfmul_d,          "xvfmul.d",         0,  0x75390000,0xffff8000,DF_A_3R)
INST(xvfdiv_s,          "xvfdiv.s",         0,  0x753a8000,0xffff8000,DF_A_3R)
INST(xvfdiv_d,          "xvfdiv.d",         0,  0x753b0000,0xffff8000,DF_A_3R)
INST(xvfmax_s,          "xvfmax.s",         0,  0x753c8000,0xffff8000,DF_A_3R)
INST(xvfmax_d,          "xvfmax.d",         0,  0x753d0000,0xffff8000,DF_A_3R)
INST(xvfmin_s,          "xvfmin.s",         0,  0x753e8000,0xffff8000,DF_A_3R)
INST(xvfmin_d,          "xvfmin.d",         0,  0x753f0000,0xffff8000,DF_A_3R)
INST(xvfmaxa_s,         "xvfmaxa.s",        0,  0x75408000,0xffff8000,DF_A_3R)
INST(xvfmaxa_d,         "xvfmaxa.d",        0,  0x75410000,0xffff8000,DF_A_3R)
INST(xvfmina_s,         "xvfmina.s",        0,  0x75428000,0xffff8000,DF_A_3R)
INST(xvfmina_d,         "xvfmina.d",        0,  0x75430000,0xffff8000,DF_A_3R)
INST(xvfcvt_h_s,        "xvfcvt.h.s",       0,  0x75460000,0xffff8000,DF_A_3R)
INST(xvfcvt_s_d,        "xvfcvt.s.d",       0,  0x75468000,0xffff8000,DF_A_3R)
INST(xvffint_s_l,       "xvffint.s.l",      0,  0x75480000,0xffff8000,DF_A_3R)
INST(xvftint_w_d,       "xvftint.w.d",      0,  0x75498000,0xffff8000,DF_A_3R)
INST(xvftintrm_w_d,     "xvftintrm.w.d",    0,  0x754a0000,0xffff8000,DF_A_3R)
INST(xvftintrp_w_d,     "xvftintrp.w.d",    0,  0x754a8000,0xffff8000,DF_A_3R)
INST(xvftintrz_w_d,     "xvftintrz.w.d",    0,  0x754b0000,0xffff8000,DF_A_3R)
INST(xvftintrne_w_d,    "xvftintrne.w.d",   0,  0x754b8000,0xffff8000,DF_A_3R)
INST(xvshuf_h,          "xvshuf.h",         0,  0x757a8000,0xffff8000,DF_A_3R)
INST(xvshuf_w,          "xvshuf.w",         0,  0x757b0000,0xffff8000,DF_A_3R)
INST(xvshuf_d,          "xvshuf.d",         0,  0x757b8000,0xffff8000,DF_A_3R)
INST(xvperm_w,          "xvperm.w",         0,  0x757d0000,0xffff8000,DF_A_3R)

// R_R.
// xd, rj
INST(xvreplgr2vr_b,     "xvreplgr2vr.b",    0,  0x769f0000,0xfffffc00,DF_A_RG)
INST(xvreplgr2vr_h,     "xvreplgr2vr.h",    0,  0x769f0400,0xfffffc00,DF_A_RG)
INST(xvreplgr2vr_w,     "xvreplgr2vr.w",    0,  0x769f0800,0xfffffc00,DF_A_RG)
INST(xvreplgr2vr_d,     "xvreplgr2vr.d",    0,  0x769f0c00,0xfffffc00,DF_A_RG)
// xd, xj
INST(xvclo_b,           "xvclo.b",          0,  0x769c0000,0xfffffc00,DF_A_2R)
INST(xvclo_h,           "xvclo.h",          0,  0x769c0400,0xfffffc00,DF_A_2R)
INST(xvclo_w,           "xvclo.w",          0,  0x769c0800,0xfffffc00,DF_A_2R)
INST(xvclo_d,           "xvclo.d",          0,  0x769c0c00,0xfffffc00,DF_A_2R)
INST(xvclz_b,           "xvclz.b",          0,  0x769c1000,0xfffffc00,DF_A_2R)
INST(xvclz_h,           "xvclz.h",          0,  0x769c1400,0xfffffc00,DF_A_2R)
INST(xvclz_w,           "xvclz.w",          0,  0x769c1800,0xfffffc00,DF_A_2R)
INST(xvclz_d,           "xvclz.d",          0,  0x769c1c00,0xfffffc00,DF_A_2R)
INST(xvpcnt_b,          "xvpcnt.b",         0,  0x769c2000,0xfffffc00,DF_A_2R)
INST(xvpcnt_h,          "xvpcnt.h",         0,  0x769c2400,0xfffffc00,DF_A_2R)
INST(xvpcnt_w,          "xvpcnt.w",         0,  0x769c2800,0xfffffc00,DF_A_2R)
INST(xvpcnt_d,          "xvpcnt.d",         0,  0x769c2c00,0xfffffc00,DF_A_2R)
INST(xvneg_b,           "xvneg.b",          0,  0x769c3000,0xfffffc00,DF_A_2R)
INST(xvneg_h,           "xvneg.h",          0,  0x769c3400,0xfffffc00,DF_A_2R)
INST(xvneg_w,           "xvneg.w",          0,  0x769c3800,0xfffffc00,DF_A_2R)
INST(xvneg_d,           "xvneg.d",          0,  0x769c3c00,0xfffffc00,DF_A_2R)
INST(xvmskltz_b,        "xvmskltz.b",       0,  0x769c4000,0xfffffc00,DF_A_2R)
INST(xvmskltz_h,        "xvmskltz.h",       0,  0x769c4400,0xfffffc00,DF_A_2R)
INST(xvmskltz_w,        "xvmskltz.w",       0,  0x769c4800,0xfffffc00,DF_A_2R)
INST(xvmskltz_d,        "xvmskltz.d",       0,  0x769c4c00,0xfffffc00,DF_A_2R)
INST(xvmskgez_b,        "xvmskgez.b",       0,  0x769c5000,0xfffffc00,DF_A_2R)
INST(xvmsknz_b,         "xvmsknz.b",        0,  0x769c6000,0xfffffc00,DF_A_2R)
INST(xvflogb_s,         "xvflogb.s",        0,  0x769cc400,0xfffffc00,DF_A_2R)
INST(xvflogb_d,         "xvflogb.d",        0,  0x769cc800,0xfffffc00,DF_A_2R)
INST(xvfclass_s,        "xvfclass.s",       0,  0x769cd400,0xfffffc00,DF_A_2R)
INST(xvfclass_d,        "xvfclass.d",       0,  0x769cd800,0xfffffc00,DF_A_2R)
INST(xvfsqrt_s,         "xvfsqrt.s",        0,  0x769ce400,0xfffffc00,DF_A_2R)
INST(xvfsqrt_d,         "xvfsqrt.d",        0,  0x769ce800,0xfffffc00,DF_A_2R)
INST(xvfrecip_s,        "xvfrecip.s",       0,  0x769cf400,0xfffffc00,DF_A_2R)
INST(xvfrecip_d,        "xvfrecip.d",       0,  0x769cf800,0xfffffc00,DF_A_2R)
INST(xvfrsqrt_s,        "xvfrsqrt.s",       0,  0x769d0400,0xfffffc00,DF_A_2R)
INST(xvfrsqrt_d,        "xvfrsqrt.d",       0,  0x769d0800,0xfffffc00,DF_A_2R)
INST(xvfrint_s,         "xvfrint.s",        0,  0x769d3400,0xfffffc00,DF_A_2R)
INST(xvfrint_d,         "xvfrint.d",        0,  0x769d3800,0xfffffc00,DF_A_2R)
INST(xvfrintrm_s,       "xvfrintrm.s",      0,  0x769d4400,0xfffffc00,DF_A_2R)
INST(xvfrintrm_d,       "xvfrintrm.d",      0,  0x769d4800,0xfffffc00,DF_A_2R)
INST(xvfrintrp_s,       "xvfrintrp.s",      0,  0x769d5400,0xfffffc00,DF_A_2R)
INST(xvfrintrp_d,       "xvfrintrp.d",      0,  0x769d5800,0xfffffc00,DF_A_2R)
INST(xvfrintrz_s,       "xvfrintrz.s",      0,  0x769d6400,0xfffffc00,DF_A_2R)
INST(xvfrintrz_d,       "xvfrintrz.d",      0,  0x769d6800,0xfffffc00,DF_A_2R)
INST(xvfrintrne_s,      "xvfrintrne.s",     0,  0x769d7400,0xfffffc00,DF_A_2R)
INST(xvfrintrne_d,      "xvfrintrne.d",     0,  0x769d7800,0xfffffc00,DF_A_2R)
INST(xvfcvtl_s_h,       "xvfcvtl.s.h",      0,  0x769de800,0xfffffc00,DF_A_2R)
INST(xvfcvth_s_h,       "xvfcvth.s.h",      0,  0x769dec00,0xfffffc00,DF_A_2R)
INST(xvfcvtl_d_s,       "xvfcvtl.d.s",      0,  0x769df000,0xfffffc00,DF_A_2R)
INST(xvfcvth_d_s,       "xvfcvth.d.s",      0,  0x769df400,0xfffffc00,DF_A_2R)
INST(xvffint_s_w,       "xvffint.s.w",      0,  0x769e0000,0xfffffc00,DF_A_2R)
INST(xvffint_s_wu,      "xvffint.s.wu",     0,  0x769e0400,0xfffffc00,DF_A_2R)
INST(xvffint_d_l,       "xvffint.d.l",      0,  0x769e0800,0xfffffc00,DF_A_2R)
INST(xvffint_d_lu,      "xvffint.d.lu",     0,  0x769e0c00,0xfffffc00,DF_A_2R)
INST(xvffintl_d_w,      "xvffintl.d.w",     0,  0x769e1000,0xfffffc00,DF_A_2R)
INST(xvffinth_d_w,      "xvffinth.d.w",     0,  0x769e1400,0xfffffc00,DF_A_2R)
INST(xvftint_w_s,       "xvftint.w.s",      0,  0x769e3000,0xfffffc00,DF_A_2R)
INST(xvftint_l_d,       "xvftint.l.d",      0,  0x769e3400,0xfffffc00,DF_A_2R)
INST(xvftintrm_w_s,     "xvftintrm.w.s",    0,  0x769e3800,0xfffffc00,DF_A_2R)
INST(xvftintrm_l_d,     "xvftintrm.l.d",    0,  0x769e3c00,0xfffffc00,DF_A_2R)
INST(xvftintrp_w_s,     "xvftintrp.w.s",    0,  0x769e4000,0xfffffc00,DF_A_2R)
INST(xvftintrp_l_d,     "xvftintrp.l.d",    0,  0x769e4400,0xfffffc00,DF_A_2R)
INST(xvftintrz_w_s,     "xvftintrz.w.s",    0,  0x769e4800,0xfffffc00,DF_A_2R)
INST(xvftintrz_l_d,     "xvftintrz.l.d",    0,  0x769e4c00,0xfffffc00,DF_A_2R)
INST(xvftintrne_w_s,    "xvftintrne.w.s",   0,  0x769e5000,0xfffffc00,DF_A_2R)
INST(xvftintrne_l_d,    "xvftintrne.l.d",   0,  0x769e5400,0xfffffc00,DF_A_2R)
INST(xvftint_wu_s,      "xvftint.wu.s",     0,  0x769e5800,0xfffffc00,DF_A_2R)
INST(xvftint_lu_d,      "xvftint.lu.d",     0,  0x769e5c00,0xfffffc00,DF_A_2R)
INST(xvftintrz_wu_s,    "xvftintrz.wu.s",   0,  0x769e7000,0xfffffc00,DF_A_2R)
INST(xvftintrz_lu_d,    "xvftintrz.lu.d",   0,  0x769e7400,0xfffffc00,DF_A_2R)
INST(xvftintl_l_s,      "xvftintl.l.s",     0,  0x769e8000,0xfffffc00,DF_A_2R)
INST(xvftinth_l_s,      "xvftinth.l.s",     0,  0x769e8400,0xfffffc00,DF_A_2R)
INST(xvftintrml_l_s,    "xvftintrml.l.s",   0,  0x769e8800,0xfffffc00,DF_A_2R)
INST(xvftintrmh_l_s,    "xvftintrmh.l.s",   0,  0x769e8c00,0xfffffc00,DF_A_2R)
INST(xvftintrpl_l_s,    "xvftintrpl.l.s",   0,  0x769e9000,0xfffffc00,DF_A_2R)
INST(xvftintrph_l_s,    "xvftintrph.l.s",   0,  0x769e9400,0xfffffc00,DF_A_2R)
INST(xvftintrzl_l_s,    "xvftintrzl.l.s",   0,  0x769e9800,0xfffffc00,DF_A_2R)
INST(xvftintrzh_l_s,    "xvftintrzh.l.s",   0,  0x769e9c00,0xfffffc00,DF_A_2R)
INST(xvftintrnel_l_s,   "xvftintrnel.l.s",  0,  0x769ea000,0xfffffc00,DF_A_2R)
INST(xvftintrneh_l_s,   "xvftintrneh.l.s",  0,  0x769ea400,0xfffffc00,DF_A_2R)
INST(xvexth_h_b,        "xvexth.h.b",       0,  0x769ee000,0xfffffc00,DF_A_2R)
INST(xvexth_w_h,        "xvexth.w.h",       0,  0x769ee400,0xfffffc00,DF_A_2R)
INST(xvexth_d_w,        "xvexth.d.w",       0,  0x769ee800,0xfffffc00,DF_A_2R)
INST(xvexth_q_d,        "xvexth.q.d",       0,  0x769eec00,0xfffffc00,DF_A_2R)
INST(xvexth_hu_bu,      "xvexth.hu.bu",     0,  0x769ef000,0xfffffc00,DF_A_2R)
INST(xvexth_wu_hu,      "xvexth.wu.hu",     0,  0x769ef400,0xfffffc00,DF_A_2R)
INST(xvexth_du_wu,      "xvexth.du.wu",     0,  0x769ef800,0xfffffc00,DF_A_2R)
INST(xvexth_qu_du,      "xvexth.qu.du",     0,  0x769efc00,0xfffffc00,DF_A_2R)
INST(vext2xv_h_b,       "vext2xv.h.b",      0,  0x769f1000,0xfffffc00,DF_A_2R)
INST(vext2xv_w_b,       "vext2xv.w.b",      0,  0x769f1400,0xfffffc00,DF_A_2R)
INST(vext2xv_d_b,       "vext2xv.d.b",      0,  0x769f1800,0xfffffc00,DF_A_2R)
INST(vext2xv_w_h,       "vext2xv.w.h",      0,  0x769f1c00,0xfffffc00,DF_A_2R)
INST(vext2xv_d_h,       "vext2xv.d.h",      0,  0x769f2000,0xfffffc00,DF_A_2R)
INST(vext2xv_d_w,       "vext2xv.d.w",      0,  0x769f2400,0xfffffc00,DF_A_2R)
INST(vext2xv_hu_bu,     "vext2xv.hu.bu",    0,  0x769f2800,0xfffffc00,DF_A_2R)
INST(vext2xv_wu_bu,     "vext2xv.wu.bu",    0,  0x769f2c00,0xfffffc00,DF_A_2R)
INST(vext2xv_du_bu,     "vext2xv.du.bu",    0,  0x769f3000,0xfffffc00,DF_A_2R)
INST(vext2xv_wu_hu,     "vext2xv.wu.hu",    0,  0x769f3400,0xfffffc00,DF_A_2R)
INST(vext2xv_du_hu,     "vext2xv.du.hu",    0,  0x769f3800,0xfffffc00,DF_A_2R)
INST(vext2xv_du_wu,     "vext2xv.du.wu",    0,  0x769f3c00,0xfffffc00,DF_A_2R)
INST(xvreplve0_b,       "xvreplve0.b",      0,  0x77070000,0xfffffc00,DF_A_2R)
INST(xvreplve0_h,       "xvreplve0.h",      0,  0x77078000,0xfffffc00,DF_A_2R)
INST(xvreplve0_w,       "xvreplve0.w",      0,  0x7707c000,0xfffffc00,DF_A_2R)
INST(xvreplve0_d,       "xvreplve0.d",      0,  0x7707e000,0xfffffc00,DF_A_2R)
INST(xvreplve0_q,       "xvreplve0.q",      0,  0x7707f000,0xfffffc00,DF_A_2R)
INST(xvextl_q_d,        "xvextl.q.d",       0,  0x77090000,0xfffffc00,DF_A_2R)
INST(xvextl_qu_du,      "xvextl.qu.du",     0,  0x770d0000,0xfffffc00,DF_A_2R)

// R_R_I_I.
// xd, rj, si8, idx
INST(xvstelm_b,         "xvstelm.b",        0,  0x33800000,0xffc00000,DF_A_RG8IX)
INST(xvstelm_h,         "xvstelm.h",        0,  0x33400000,0xffe00000,DF_A_RG8IX)
INST(xvstelm_w,         "xvstelm.w",        0,  0x33200000,0xfff00000,DF_A_RG8IX)
INST(xvstelm_d,         "xvstelm.d",        0,  0x33100000,0xfff80000,DF_A_RG8IX)

// R_R_I.---LASX
// si5
INST(xvseqi_b,          "xvseqi.b",         0,  0x76800000,0xffff8000,DF_A_2R5I) // xd,xj,si5
INST(xvseqi_h,          "xvseqi.h",         0,  0x76808000,0xffff8000,DF_A_2R5I)
INST(xvseqi_w,          "xvseqi.w",         0,  0x76810000,0xffff8000,DF_A_2R5I)
INST(xvseqi_d,          "xvseqi.d",         0,  0x76818000,0xffff8000,DF_A_2R5I)
INST(xvslei_b,          "xvslei.b",         0,  0x76820000,0xffff8000,DF_A_2R5I)
INST(xvslei_h,          "xvslei.h",         0,  0x76828000,0xffff8000,DF_A_2R5I)
INST(xvslei_w,          "xvslei.w",         0,  0x76830000,0xffff8000,DF_A_2R5I)
INST(xvslei_d,          "xvslei.d",         0,  0x76838000,0xffff8000,DF_A_2R5I)
INST(xvslti_b,          "xvslti.b",         0,  0x76860000,0xffff8000,DF_A_2R5I)
INST(xvslti_h,          "xvslti.h",         0,  0x76868000,0xffff8000,DF_A_2R5I)
INST(xvslti_w,          "xvslti.w",         0,  0x76870000,0xffff8000,DF_A_2R5I)
INST(xvslti_d,          "xvslti.d",         0,  0x76878000,0xffff8000,DF_A_2R5I)
INST(xvmaxi_b,          "xvmaxi.b",         0,  0x76900000,0xffff8000,DF_A_2R5I)
INST(xvmaxi_h,          "xvmaxi.h",         0,  0x76908000,0xffff8000,DF_A_2R5I)
INST(xvmaxi_w,          "xvmaxi.w",         0,  0x76910000,0xffff8000,DF_A_2R5I)
INST(xvmaxi_d,          "xvmaxi.d",         0,  0x76918000,0xffff8000,DF_A_2R5I)
INST(xvmini_b,          "xvmini.b",         0,  0x76920000,0xffff8000,DF_A_2R5I)
INST(xvmini_h,          "xvmini.h",         0,  0x76928000,0xffff8000,DF_A_2R5I)
INST(xvmini_w,          "xvmini.w",         0,  0x76930000,0xffff8000,DF_A_2R5I)
INST(xvmini_d,          "xvmini.d",         0,  0x76938000,0xffff8000,DF_A_2R5I)
// si9
INST(xvldrepl_d,        "xvldrepl.d",       0,  0x32100000,0xfff80000,DF_A_RGX) // xd,rj,si9
// si10
INST(xvldrepl_w,        "xvldrepl.w",       0,  0x32200000,0xfff00000,DF_A_RGX) // xd,rj,si10
// si11
INST(xvldrepl_h,        "xvldrepl.h",       0,  0x32400000,0xffe00000,DF_A_RGX) // xd,rj,si11
// si12
INST(xvld,              "xvld",             LD, 0x2c800000,0xffc00000,DF_A_RGX) // xd,rj,si12
INST(xvldrepl_b,        "xvldrepl.b",       0,  0x32800000,0xffc00000,DF_A_RGX)
INST(xvst,              "xvst",             ST, 0x2cc00000,0xffc00000,DF_A_RGX)
// ui1
INST(xvrepl128vei_d,    "xvrepl128vei.d",   0,  0x76f7f000,0xfffff800,DF_A_2RX)  // xd,xj,ui1
// ui2
INST(xvinsve0_d,        "xvinsve0.d",       0,  0x76ffe000,0xfffff000,DF_A_2RX)  // xd,xj,ui2
INST(xvrepl128vei_w,    "xvrepl128vei.w",   0,  0x76f7e000,0xfffff000,DF_A_2RX)  // xd,xj,ui2
INST(xvpickve_d,        "xvpickve.d",       0,  0x7703e000,0xfffff000,DF_A_2RX)  // xd,xj,ui2
INST(xvinsgr2vr_d,      "xvinsgr2vr.d",     0,  0x76ebe000,0xfffff000,DF_A_RGX)  // xd,rj,ui2
INST(xvpickve2gr_d,     "xvpickve2gr.d",    0,  0x76efe000,0xfffff000,DF_A_GRX)  // rd,xj,ui2
INST(xvpickve2gr_du,    "xvpickve2gr.du",   0,  0x76f3e000,0xfffff000,DF_A_GRX)  // rd,xj,ui2
// ui3
INST(xvslli_b,          "xvslli.b",         0,  0x772c2000,0xffffe000,DF_A_2R3IU) // xd,xj,ui3
INST(xvsrli_b,          "xvsrli.b",         0,  0x77302000,0xffffe000,DF_A_2R3IU)
INST(xvsrai_b,          "xvsrai.b",         0,  0x77342000,0xffffe000,DF_A_2R3IU)
INST(xvrotri_b,         "xvrotri.b",        0,  0x76a02000,0xffffe000,DF_A_2R3IU)
INST(xvsrlri_b,         "xvsrlri.b",        0,  0x76a42000,0xffffe000,DF_A_2R3IU)
INST(xvsrari_b,         "xvsrari.b",        0,  0x76a82000,0xffffe000,DF_A_2R3IU)
INST(xvsllwil_h_b,      "xvsllwil.h.b",     0,  0x77082000,0xffffe000,DF_A_2R3IU)
INST(xvsllwil_hu_bu,    "xvsllwil.hu.bu",   0,  0x770c2000,0xffffe000,DF_A_2R3IU)
INST(xvsrlni_b_h,       "xvsrlni.b.h",      0,  0x77404000,0xffffe000,DF_A_2R3IU)
INST(xvsrlrni_b_h,      "xvsrlrni.b.h",     0,  0x77444000,0xffffe000,DF_A_2R3IU)
INST(xvssrlni_b_h,      "xvssrlni.b.h",     0,  0x77484000,0xffffe000,DF_A_2R3IU)
INST(xvssrlni_bu_h,     "xvssrlni.bu.h",    0,  0x774c4000,0xffffe000,DF_A_2R3IU)
INST(xvssrlrni_b_h,     "xvssrlrni.b.h",    0,  0x77504000,0xffffe000,DF_A_2R3IU)
INST(xvssrlrni_bu_h,    "xvssrlrni.bu.h",   0,  0x77544000,0xffffe000,DF_A_2R3IU)
INST(xvsrani_b_h,       "xvsrani.b.h",      0,  0x77584000,0xffffe000,DF_A_2R3IU)
INST(xvsrarni_b_h,      "xvsrarni.b.h",     0,  0x775c4000,0xffffe000,DF_A_2R3IU)
INST(xvssrani_b_h,      "xvssrani.b.h",     0,  0x77604000,0xffffe000,DF_A_2R3IU)
INST(xvssrani_bu_h,     "xvssrani.bu.h",    0,  0x77644000,0xffffe000,DF_A_2R3IU)
INST(xvssrarni_b_h,     "xvssrarni.b.h",    0,  0x77684000,0xffffe000,DF_A_2R3IU)
INST(xvssrarni_bu_h,    "xvssrarni.bu.h",   0,  0x776c4000,0xffffe000,DF_A_2R3IU)
INST(xvinsve0_w,        "xvinsve0.w",       0,  0x76ffc000,0xffffe000,DF_A_2R3IU)
INST(xvrepl128vei_h,    "xvrepl128vei.h",   0,  0x76f7c000,0xffffe000,DF_A_2R3IU)
INST(xvpickve_w,        "xvpickve.w",       0,  0x7703c000,0xffffe000,DF_A_2R3IU)
INST(xvbitclri_b,       "xvbitclri.b",      0,  0x77102000,0xffffe000,DF_A_2R3IU)
INST(xvbitseti_b,       "xvbitseti.b",      0,  0x77142000,0xffffe000,DF_A_2R3IU)
INST(xvbitrevi_b,       "xvbitrevi.b",      0,  0x77182000,0xffffe000,DF_A_2R3IU)
INST(xvinsgr2vr_w,      "xvinsgr2vr.w",     0,  0x76ebc000,0xffffe000,DF_A_RGX)   // xd,rj,ui3
INST(xvpickve2gr_w,     "xvpickve2gr.w",    0,  0x76efc000,0xffffe000,DF_A_GRX)   // rd,xj,ui3
INST(xvpickve2gr_wu,    "xvpickve2gr.wu",   0,  0x76f3c000,0xffffe000,DF_A_GRX)   // rd,xj,ui3
INST(xvsat_b,           "xvsat.b",          0,  0x77242000,0xffffe000,DF_A_2R3IU) // vd,xj,ui3
INST(xvsat_bu,          "xvsat.bu",         0,  0x77282000,0xffffe000,DF_A_2R3IU) // vd,xj,ui3
// ui4
INST(xvslli_h,          "xvslli.h",         0,  0x772c4000,0xffffc000,DF_A_2R4IU) // xd,xj,ui4
INST(xvsrli_h,          "xvsrli.h",         0,  0x77304000,0xffffc000,DF_A_2R4IU)
INST(xvsrai_h,          "xvsrai.h",         0,  0x77344000,0xffffc000,DF_A_2R4IU)
INST(xvsrlri_h,         "xvsrlri.h",        0,  0x76a44000,0xffffc000,DF_A_2R4IU)
INST(xvsrari_h,         "xvsrari.h",        0,  0x76a84000,0xffffc000,DF_A_2R4IU)
INST(xvrotri_h,         "xvrotri.h",        0,  0x76a04000,0xffffc000,DF_A_2R4IU)
INST(xvsllwil_w_h,      "xvsllwil.w.h",     0,  0x77084000,0xffffc000,DF_A_2R4IU)
INST(xvsllwil_wu_hu,    "xvsllwil.wu.hu",   0,  0x770c4000,0xffffc000,DF_A_2R4IU)
INST(xvrepl128vei_b,    "xvrepl128vei.b",   0,  0x76f78000,0xffffc000,DF_A_2R4IU)
INST(xvbitclri_h,       "xvbitclri.h",      0,  0x77104000,0xffffc000,DF_A_2R4IU)
INST(xvbitseti_h,       "xvbitseti.h",      0,  0x77144000,0xffffc000,DF_A_2R4IU)
INST(xvbitrevi_h,       "xvbitrevi.h",      0,  0x77184000,0xffffc000,DF_A_2R4IU)
INST(xvsat_h,           "xvsat.h",          0,  0x77244000,0xffffc000,DF_A_2R4IU)
INST(xvsat_hu,          "xvsat.hu",         0,  0x77284000,0xffffc000,DF_A_2R4IU)
// ui5
INST(xvslei_bu,         "xvslei.bu",        0,  0x76840000,0xffff8000,DF_A_2R5IU) // xd,xj,ui5
INST(xvslei_hu,         "xvslei.hu",        0,  0x76848000,0xffff8000,DF_A_2R5IU)
INST(xvslei_wu,         "xvslei.wu",        0,  0x76850000,0xffff8000,DF_A_2R5IU)
INST(xvslei_du,         "xvslei.du",        0,  0x76858000,0xffff8000,DF_A_2R5IU)
INST(xvslti_bu,         "xvslti.bu",        0,  0x76880000,0xffff8000,DF_A_2R5IU)
INST(xvslti_hu,         "xvslti.hu",        0,  0x76888000,0xffff8000,DF_A_2R5IU)
INST(xvslti_wu,         "xvslti.wu",        0,  0x76890000,0xffff8000,DF_A_2R5IU)
INST(xvslti_du,         "xvslti.du",        0,  0x76898000,0xffff8000,DF_A_2R5IU)
INST(xvaddi_bu,         "xvaddi.bu",        0,  0x768a0000,0xffff8000,DF_A_2R5IU)
INST(xvaddi_hu,         "xvaddi.hu",        0,  0x768a8000,0xffff8000,DF_A_2R5IU)
INST(xvaddi_wu,         "xvaddi.wu",        0,  0x768b0000,0xffff8000,DF_A_2R5IU)
INST(xvaddi_du,         "xvaddi.du",        0,  0x768b8000,0xffff8000,DF_A_2R5IU)
INST(xvsubi_bu,         "xvsubi.bu",        0,  0x768c0000,0xffff8000,DF_A_2R5IU)
INST(xvsubi_hu,         "xvsubi.hu",        0,  0x768c8000,0xffff8000,DF_A_2R5IU)
INST(xvsubi_wu,         "xvsubi.wu",        0,  0x768d0000,0xffff8000,DF_A_2R5IU)
INST(xvsubi_du,         "xvsubi.du",        0,  0x768d8000,0xffff8000,DF_A_2R5IU)
INST(xvmaxi_bu,         "xvmaxi.bu",        0,  0x76940000,0xffff8000,DF_A_2R5IU)
INST(xvmaxi_hu,         "xvmaxi.hu",        0,  0x76948000,0xffff8000,DF_A_2R5IU)
INST(xvmaxi_wu,         "xvmaxi.wu",        0,  0x76950000,0xffff8000,DF_A_2R5IU)
INST(xvmaxi_du,         "xvmaxi.du",        0,  0x76958000,0xffff8000,DF_A_2R5IU)
INST(xvmini_bu,         "xvmini.bu",        0,  0x76960000,0xffff8000,DF_A_2R5IU)
INST(xvmini_hu,         "xvmini.hu",        0,  0x76968000,0xffff8000,DF_A_2R5IU)
INST(xvmini_wu,         "xvmini.wu",        0,  0x76970000,0xffff8000,DF_A_2R5IU)
INST(xvmini_du,         "xvmini.du",        0,  0x76978000,0xffff8000,DF_A_2R5IU)
INST(xvfrstpi_b,        "xvfrstpi.b",       0,  0x769a0000,0xffff8000,DF_A_2R5IU)
INST(xvfrstpi_h,        "xvfrstpi.h",       0,  0x769a8000,0xffff8000,DF_A_2R5IU)
INST(xvslli_w,          "xvslli.w",         0,  0x772c8000,0xffff8000,DF_A_2R5IU)
INST(xvsrli_w,          "xvsrli.w",         0,  0x77308000,0xffff8000,DF_A_2R5IU)
INST(xvsrai_w,          "xvsrai.w",         0,  0x77348000,0xffff8000,DF_A_2R5IU)
INST(xvbsll_v,          "xvbsll.v",         0,  0x768e0000,0xffff8000,DF_A_2R5IU)
INST(xvbsrl_v,          "xvbsrl.v",         0,  0x768e8000,0xffff8000,DF_A_2R5IU)
INST(xvsrlri_w,         "xvsrlri.w",        0,  0x76a48000,0xffff8000,DF_A_2R5IU)
INST(xvsrari_w,         "xvsrari.w",        0,  0x76a88000,0xffff8000,DF_A_2R5IU)
INST(xvrotri_w,         "xvrotri.w",        0,  0x76a08000,0xffff8000,DF_A_2R5IU)
INST(xvsllwil_d_w,      "xvsllwil.d.w",     0,  0x77088000,0xffff8000,DF_A_2R5IU)
INST(xvsllwil_du_wu,    "xvsllwil.du.wu",   0,  0x770c8000,0xffff8000,DF_A_2R5IU)
INST(xvsrlni_h_w,       "xvsrlni.h.w",      0,  0x77408000,0xffff8000,DF_A_2R5IU)
INST(xvsrlrni_h_w,      "xvsrlrni.h.w",     0,  0x77448000,0xffff8000,DF_A_2R5IU)
INST(xvssrlni_h_w,      "xvssrlni.h.w",     0,  0x77488000,0xffff8000,DF_A_2R5IU)
INST(xvssrlni_hu_w,     "xvssrlni.hu.w",    0,  0x774c8000,0xffff8000,DF_A_2R5IU)
INST(xvssrlrni_h_w,     "xvssrlrni.h.w",    0,  0x77508000,0xffff8000,DF_A_2R5IU)
INST(xvssrlrni_hu_w,    "xvssrlrni.hu.w",   0,  0x77548000,0xffff8000,DF_A_2R5IU)
INST(xvsrani_h_w,       "xvsrani.h.w",      0,  0x77588000,0xffff8000,DF_A_2R5IU)
INST(xvsrarni_h_w,      "xvsrarni.h.w",     0,  0x775c8000,0xffff8000,DF_A_2R5IU)
INST(xvssrani_h_w,      "xvssrani.h.w",     0,  0x77608000,0xffff8000,DF_A_2R5IU)
INST(xvssrani_hu_w,     "xvssrani.hu.w",    0,  0x77648000,0xffff8000,DF_A_2R5IU)
INST(xvssrarni_h_w,     "xvssrarni.h.w",    0,  0x77688000,0xffff8000,DF_A_2R5IU)
INST(xvssrarni_hu_w,    "xvssrarni.hu.w",   0,  0x776c8000,0xffff8000,DF_A_2R5IU)
INST(xvbitclri_w,       "xvbitclri.w",      0,  0x77108000,0xffff8000,DF_A_2R5IU)
INST(xvbitseti_w,       "xvbitseti.w",      0,  0x77148000,0xffff8000,DF_A_2R5IU)
INST(xvbitrevi_w,       "xvbitrevi.w",      0,  0x77188000,0xffff8000,DF_A_2R5IU)
INST(xvsat_w,           "xvsat.w",          0,  0x77248000,0xffff8000,DF_A_2R5IU)
INST(xvsat_wu,          "xvsat.wu",         0,  0x77288000,0xffff8000,DF_A_2R5IU)
// ui6
INST(xvslli_d,          "xvslli.d",         0,  0x772d0000,0xffff0000,DF_A_2R6IU)
INST(xvsrli_d,          "xvsrli.d",         0,  0x77310000,0xffff0000,DF_A_2R6IU)
INST(xvsrai_d,          "xvsrai.d",         0,  0x77350000,0xffff0000,DF_A_2R6IU)
INST(xvrotri_d,         "xvrotri.d",        0,  0x76a10000,0xffff0000,DF_A_2R6IU)
INST(xvsrlri_d,         "xvsrlri.d",        0,  0x76a50000,0xffff0000,DF_A_2R6IU)
INST(xvsrari_d,         "xvsrari.d",        0,  0x76a90000,0xffff0000,DF_A_2R6IU)
INST(xvsrlni_w_d,       "xvsrlni.w.d",      0,  0x77410000,0xffff0000,DF_A_2R6IU)
INST(xvsrlrni_w_d,      "xvsrlrni.w.d",     0,  0x77450000,0xffff0000,DF_A_2R6IU)
INST(xvssrlni_w_d,      "xvssrlni.w.d",     0,  0x77490000,0xffff0000,DF_A_2R6IU)
INST(xvssrlni_wu_d,     "xvssrlni.wu.d",    0,  0x774d0000,0xffff0000,DF_A_2R6IU)
INST(xvssrlrni_w_d,     "xvssrlrni.w.d",    0,  0x77510000,0xffff0000,DF_A_2R6IU)
INST(xvssrlrni_wu_d,    "xvssrlrni.wu.d",   0,  0x77550000,0xffff0000,DF_A_2R6IU)
INST(xvsrani_w_d,       "xvsrani.w.d",      0,  0x77590000,0xffff0000,DF_A_2R6IU)
INST(xvsrarni_w_d,      "xvsrarni.w.d",     0,  0x775d0000,0xffff0000,DF_A_2R6IU)
INST(xvssrani_w_d,      "xvssrani.w.d",     0,  0x77610000,0xffff0000,DF_A_2R6IU)
INST(xvssrani_wu_d,     "xvssrani.wu.d",    0,  0x77650000,0xffff0000,DF_A_2R6IU)
INST(xvssrarni_w_d,     "xvssrarni.w.d",    0,  0x77690000,0xffff0000,DF_A_2R6IU)
INST(xvssrarni_wu_d,    "xvssrarni.wu.d",   0,  0x776d0000,0xffff0000,DF_A_2R6IU)
INST(xvbitclri_d,       "xvbitclri.d",      0,  0x77110000,0xffff0000,DF_A_2R6IU)
INST(xvbitseti_d,       "xvbitseti.d",      0,  0x77150000,0xffff0000,DF_A_2R6IU)
INST(xvbitrevi_d,       "xvbitrevi.d",      0,  0x77190000,0xffff0000,DF_A_2R6IU)
INST(xvsat_d,           "xvsat.d",          0,  0x77250000,0xffff0000,DF_A_2R6IU)
INST(xvsat_du,          "xvsat.du",         0,  0x77290000,0xffff0000,DF_A_2R6IU)
// ui7
INST(xvsrlni_d_q,       "xvsrlni.d.q",      0,  0x77420000,0xfffe0000,DF_A_2R7IU)
INST(xvsrlrni_d_q,      "xvsrlrni.d.q",     0,  0x77460000,0xfffe0000,DF_A_2R7IU)
INST(xvssrlni_d_q,      "xvssrlni.d.q",     0,  0x774a0000,0xfffe0000,DF_A_2R7IU)
INST(xvssrlni_du_q,     "xvssrlni.du.q",    0,  0x774e0000,0xfffe0000,DF_A_2R7IU)
INST(xvssrlrni_d_q,     "xvssrlrni.d.q",    0,  0x77520000,0xfffe0000,DF_A_2R7IU)
INST(xvssrlrni_du_q,    "xvssrlrni.du.q",   0,  0x77560000,0xfffe0000,DF_A_2R7IU)
INST(xvsrani_d_q,       "xvsrani.d.q",      0,  0x775a0000,0xfffe0000,DF_A_2R7IU)
INST(xvsrarni_d_q,      "xvsrarni.d.q",     0,  0x775e0000,0xfffe0000,DF_A_2R7IU)
INST(xvssrani_d_q,      "xvssrani.d.q",     0,  0x77620000,0xfffe0000,DF_A_2R7IU)
INST(xvssrani_du_q,     "xvssrani.du.q",    0,  0x77660000,0xfffe0000,DF_A_2R7IU)
INST(xvssrarni_d_q,     "xvssrarni.d.q",    0,  0x776a0000,0xfffe0000,DF_A_2R7IU)
INST(xvssrarni_du_q,    "xvssrarni.du.q",   0,  0x776e0000,0xfffe0000,DF_A_2R7IU)
// ui8
INST(xvextrins_d,       "xvextrins.d",      0,  0x77800000,0xfffc0000,DF_A_2R8IU)
INST(xvextrins_w,       "xvextrins.w",      0,  0x77840000,0xfffc0000,DF_A_2R8IU)
INST(xvextrins_h,       "xvextrins.h",      0,  0x77880000,0xfffc0000,DF_A_2R8IU)
INST(xvextrins_b,       "xvextrins.b",      0,  0x778c0000,0xfffc0000,DF_A_2R8IU)
INST(xvshuf4i_b,        "xvshuf4i.b",       0,  0x77900000,0xfffc0000,DF_A_2R8IU)
INST(xvshuf4i_h,        "xvshuf4i.h",       0,  0x77940000,0xfffc0000,DF_A_2R8IU)
INST(xvshuf4i_w,        "xvshuf4i.w",       0,  0x77980000,0xfffc0000,DF_A_2R8IU)
INST(xvshuf4i_d,        "xvshuf4i.d",       0,  0x779c0000,0xfffc0000,DF_A_2R8IU)
INST(xvbitseli_b,       "xvbitseli.b",      0,  0x77c40000,0xfffc0000,DF_A_2R8IU)
INST(xvandi_b,          "xvandi.b",         0,  0x77d00000,0xfffc0000,DF_A_2R8IU)
INST(xvori_b,           "xvori.b",          0,  0x77d40000,0xfffc0000,DF_A_2R8IU)
INST(xvxori_b,          "xvxori.b",         0,  0x77d80000,0xfffc0000,DF_A_2R8IU)
INST(xvnori_b,          "xvnori.b",         0,  0x77dc0000,0xfffc0000,DF_A_2R8IU)
INST(xvpermi_w,         "xvpermi.w",        0,  0x77e40000,0xfffc0000,DF_A_2R8IU)
INST(xvpermi_d,         "xvpermi.d",        0,  0x77e80000,0xfffc0000,DF_A_2R8IU)
INST(xvpermi_q,         "xvpermi.q",        0,  0x77ec0000,0xfffc0000,DF_A_2R8IU)

// R_I: xd, ui13
INST(xvldi,             "xvldi",            LD, 0x77e00000,0xfffc0000,DF_A_R13IU)
// R_I: cd, xj
INST(xvseteqz_v,        "xvseteqz.v",       0,  0x769c9800,0xfffffc18,DF_A_CR)
INST(xvsetnez_v,        "xvsetnez.v",       0,  0x769c9c00,0xfffffc18,DF_A_CR)
INST(xvsetanyeqz_b,     "xvsetanyeqz.b",    0,  0x769ca000,0xfffffc18,DF_A_CR)
INST(xvsetanyeqz_h,     "xvsetanyeqz.h",    0,  0x769ca400,0xfffffc18,DF_A_CR)
INST(xvsetanyeqz_w,     "xvsetanyeqz.w",    0,  0x769ca800,0xfffffc18,DF_A_CR)
INST(xvsetanyeqz_d,     "xvsetanyeqz.d",    0,  0x769cac00,0xfffffc18,DF_A_CR)
INST(xvsetallnez_b,     "xvsetallnez.b",    0,  0x769cb000,0xfffffc18,DF_A_CR)
INST(xvsetallnez_h,     "xvsetallnez.h",    0,  0x769cb400,0xfffffc18,DF_A_CR)
INST(xvsetallnez_w,     "xvsetallnez.w",    0,  0x769cb800,0xfffffc18,DF_A_CR)
INST(xvsetallnez_d,     "xvsetallnez.d",    0,  0x769cbc00,0xfffffc18,DF_A_CR)
#endif

// clang-format on
/*****************************************************************************/
#undef INST
/*****************************************************************************/
