/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */

/*! \mainpage LCAS - Local Centre Authorization Service
 *
 *  \section intro Introduction
 *
 *  This document describes the LCAS API and the LCAS plugins. Please check the links above.
 *
 *  \section interfaces the LCAS Interfaces
 *
 *  -# The interface to the LCAS credential mapping framework is described in \ref LcasInterface
 *  -# The LCAS plugins should use the LCAS API described in \ref APIforLcasPlugins
 *  -# The interface that the plugins should provide to the LCAS framework is described in
 *     \ref LcasPluginInterface
 *
 *  \section plugins The LCAS plugins
 *  A description of the LCAS plugins can be found
 *  here ...
 *
 *  ... the basic plugins:
 *  -# \ref lcas_userallow.mod
 *  -# \ref lcas_userban.mod
 *  -# \ref lcas_timeslots.mod
 *
 *  ... the voms-aware plugin:
 *  -# \ref lcas_voms.mod
 *
 */

/*!
    \file   lcas.c
    \brief  LCAS - the local centre authorization service.
    \author Martijn Steenbakkers for the EU DataGrid.

    The interface to the LCAS module is composed of:
    -# lcas_init(): To initialize the LCAS module
    -# lcas_get_fabric_authorization(): to place an authorization request
    -# lcas_term(): To cleanly terminate the module
*/

#ifndef LCAS_C
#define LCAS_C

/*****************************************************************************
                            Include header files
******************************************************************************/
#include "lcas_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Programming interface to dynamic linking loader */
#if HAVE_DLFCN_H
#    include <dlfcn.h>
#endif

#ifndef NOGLOBUS
    #include <gssapi.h>
#endif

/* LCAS includes */
#include "lcas_types.h"
#include "_lcas_utils.h"
#include "_lcas_defines.h"
#include "_lcas_log.h"
#include "_lcas_db_read.h"
#include "_lcas.h"
#include "grid_credential_handling/gsi_handling/_lcas_gsi_utils.h"
#include "grid_credential_handling/x509_handling/_lcas_x509_utils.h"


/******************************************************************************
                             Define constants
******************************************************************************/

#ifndef NUL
#define NUL '\0' /*!< NUL character \internal */
#endif

/* LCAS authorization module definitions */
#define MAXAUTHMODS 3 /*!< maximum number of standard authorization modules \internal */
#define MAXPROCS    4 /*!< maximum number of interface symbols in authorization modules \internal */

/*
 * LCAS error codes
 */
#define FAILED_LCAS_USERALLOW   1 /*!< return value of LCAS because of failure of userallow module
                                       (obsolete) */
#define FAILED_LCAS_USERBAN     2 /*!< return value of LCAS because of failure of userban module
	         (obsolete) */
#define FAILED_LCAS_CLOCKCHECK  3 /*!< return value of LCAS because of failure of clockcheck module
	         (obsolete) */
#define FAILED_LCAS_OTHER       4 /*!< return value of LCAS because of general failure */
#define FAILED_LCAS_PLUGIN      5 /*!< return value of LCAS because of failure of plugin
	         authorization module */

#ifndef APIMAJORVERSION
#    define APIMAJORVERSION -1
#endif
#ifndef APIMINORVERSION
#    define APIMINORVERSION -1
#endif
#ifndef APIPATCHVERSION
#    define APIPATCHVERSION -1
#endif

/******************************************************************************
                               Type definitions
******************************************************************************/
/*!
    \enum lcas_proctype_e
    \brief This enumeration type gives the different plugin symbol/function types.
    \internal
*/
enum lcas_proctype_e
{
    INITPROC, /*!< this value corresponds to the plugin initialization function */
    AUTHPROC, /*!< this value corresponds to the plugin authorization request function */
    AUTHPROCX509, /*!< this value corresponds to the plugin authorization request function */
    TERMPROC, /*!< this value corresponds to the plugin termination function */
    ENDOFPROCS /*< this is the last enumeration value */
};

/*!
    \typedef lcas_plugindl_t
    \brief the type definition of the plugin authorization module structure
    \internal
*/
/*!
    \struct lcas_plugindl_s
    \brief the plugin authorization module structure
    \internal
*/
typedef struct lcas_plugindl_s
{
    void *                    handle; /*!< dlopen handle to plugin module */
    lcas_proc_t               procs[MAXPROCS]; /*!< list of handles to interface functions of plugin */
    char                      pluginname[LCAS_MAXPATHLEN+1]; /*!< name of plugin */
    char                      pluginargs[LCAS_MAXARGSTRING+1]; /*!< argument string */
    int                       argc; /*!< number of arguments */
    char *                    argv[LCAS_MAXARGS]; /*!< list of arguments */
    struct lcas_plugindl_s *  next; /*!< pointer to the next plugin in the plugin list */
}
lcas_plugindl_t;

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
static lcas_plugindl_t * PluginInit(lcas_db_entry_t *, lcas_plugindl_t **);
static lcas_proc_t       get_procsymbol(void *, char *);
static int               print_lcas_plugin(int, lcas_plugindl_t *);
static int               parse_args_plugin(const char *, const char *, char **, int *);
static int               clean_plugin_list(lcas_plugindl_t **);

/******************************************************************************
                       Define module specific variables
******************************************************************************/
static lcas_cred_id_t    lcas_cred; /*!< \internal */
static int               lcas_initialized = 0; /*!< \internal */
static char *            lcas_db_file_default = NULL; /*!< \internal */
static char *            lcas_dir         = NULL; /*!< \internal */
static lcas_plugindl_t * plugin_list      = NULL; /*!< \internal */
static lcas_plugindl_t * authmod_list     = NULL; /*!< \internal */
static char *            authmods[MAXAUTHMODS][2] = {
                             {(char *) NULL, (char *) NULL},
                             {(char *) NULL, (char *) NULL},
                             {(char *) NULL, (char *) NULL}
}; /*!< \internal */
/*
static char *            authmods[MAXAUTHMODS][2] =
{
                             "lcas_userallow.mod", "allowed_users.db",
                             "lcas_userban.mod",   "ban_users.db",
                             "lcas_timeslots.mod", "timeslots.db"
                         };
*/

int getMajorVersionNumber() { return (int)(APIMAJORVERSION); }
int getMinorVersionNumber() { return (int)(APIMINORVERSION); }
int getPatchVersionNumber() { return (int)(APIPATCHVERSION); }

/*
 * To be done: set up LCAS as a service and communicate with gatekeeper
 * in an established security context
 */

/******************************************************************************
Function:   lcas_init_and_logfile
Description:
    Select logging type
    Setup logging by providing a file handle or file name, error handling (not yet).
    Read from LCAS database the plugins to be loaded.
    Initialize the plugins

Parameters:
    logfile: name of logfile 
    fp: file handle for logging (from gatekeeper or other previously opened file handle)
        If the file handle is zero, assume that only syslogging is requested
    logtype: type of logging (usrlog and/or syslog)

Returns:
    0: initialization succeeded
    1: initialization failed
******************************************************************************/
int lcas_init_and_logfile(
        char * logfile,
        FILE* fp,
        unsigned short logtype
)
{
    lcas_db_entry_t **  lcas_db_handle=NULL;
    lcas_db_entry_t **  lcas_mod_handle=NULL;
    lcas_db_entry_t *   lcas_mod_db=NULL;
    lcas_db_entry_t     lcas_mod_entry;
    lcas_db_entry_t *   ihandle=NULL;
    lcas_plugindl_t *   plugin_entry=NULL;
    lcas_plugindl_t *   authmod_entry=NULL;
    char *              lcas_db_file=NULL;
    int                 ientry;
    int                 i;

    if (lcas_initialized)
    {
        /* Nothing to do */
        return 0;
    }

    /* set logging file descriptor, for the moment the gatekeeper logfile */
    if (lcas_log_open(logfile,fp,logtype))
        goto fail_lcas_init_and_logfile;
    lcas_log_debug (1,"Initialization LCAS version %s\n",VERSION);

    /* get LCAS (database) home directory and set lcas_db_file */
    lcas_dir = getenv("LCAS_DIR");
    lcas_dir = (lcas_dir ? lcas_dir : getenv("LCAS_ETC_DIR") );
    lcas_dir = (lcas_dir ? lcas_dir : LCAS_ETC_HOME );

    lcas_db_file_default = getenv("LCAS_DB_FILE");
    lcas_db_file_default = (lcas_db_file_default ? lcas_db_file_default : "lcas.db" );

    lcas_db_file=lcas_genfilename(lcas_dir, lcas_db_file_default, NULL);

    /*
     * To be done: obtain the LCAS policy database which is stored within the
     * Configuration Management subsystem
     */

    /* put the STANDARD AUTHORIZATION MODULE info in lcas_db-like structure */
    lcas_mod_entry.next=NULL;
    lcas_mod_handle=&lcas_mod_db;
    for (i=0; i < MAXAUTHMODS; ++i)
    {
        if (authmods[i][0] == NULL) break; /* No standard, default authorization modules */
        *(lcas_mod_entry.pluginname)=NUL;
        *(lcas_mod_entry.pluginargs)=NUL;
        if (authmods[i][0] != NULL)
        {
            strncpy(lcas_mod_entry.pluginname,authmods[i][0],LCAS_MAXPATHLEN);
            (lcas_mod_entry.pluginname)[LCAS_MAXPATHLEN]=NUL;
        }
        if (authmods[i][1] != NULL)
        {
            strncpy(lcas_mod_entry.pluginargs,authmods[i][1],LCAS_MAXARGSTRING);
            (lcas_mod_entry.pluginargs)[LCAS_MAXARGSTRING]=NUL;
        }
        lcas_log_debug(1,"lcas.mod-lcas_init(): creating db structure for authorization module %s\n",
                 authmods[i][0]);
        if (lcas_db_fill_entry(lcas_mod_handle, &lcas_mod_entry) == NULL)
        {
            lcas_log(0,"lcas.mod-lcas_init() error: Cannot create standard authorization module (%d-%s) db structure\n",
                    i,authmods[i][0]);
            goto fail_lcas_init_and_logfile;
        }
    }

    /* Log where the modules will be searched */
    lcas_log_debug(5,"lcas.mod-lcas_init(): LCAS plug-in search paths: $LCAS_MODULE_DIR: \"%s\". The default(compile time) path: \"%s\"\n",
                     getenv("LCAS_MODULES_DIR") ? getenv("LCAS_MODULES_DIR") : "(not set)",
                     LCAS_MOD_HOME);

    /*
     * init the STANDARD AUTHORIZATION MODULES (PluginInit)
     * - open plugins and check the symbols plugin_init and confirm_authorization
     * - run plugin_init
     */
    ientry=0;
    ihandle=*lcas_mod_handle;
    while (ihandle)
    {
        if (strlen(ihandle->pluginname) > 0)
        {
            lcas_log_debug(1,"lcas.mod-lcas_init(): initializing standard module %s (db entry %d)\n",
                     ihandle->pluginname, ientry);
            if ((authmod_entry=PluginInit(ihandle,&authmod_list)) == NULL)
            {
                lcas_log(0,"lcas.mod-lcas_init(): error initializing standard module : %s\n",
                         ihandle->pluginname);
                goto fail_lcas_init_and_logfile;
            }
        }
        ientry++;
        ihandle=ihandle->next;
    }

    /* 
     * read PLUGIN AUTHORIZATION MODULE info from LCAS database and store
     * the info in a lcas_db structure
     */
    lcas_log_debug(1,"lcas.mod-lcas_init(): Reading LCAS database %s\n",
            lcas_db_file);
    lcas_db_handle=lcas_db_read(lcas_db_file);
    if (lcas_db_handle == NULL)
    {
        lcas_log(0,"lcas.mod-lcas_init(): Failed to read LCAS database %s\n",
                 lcas_db_file);
        goto fail_lcas_init_and_logfile;
    }
    /*
     * init the PLUGIN AUTHORIZATION MODULES (PluginInit)
     * - open plugins and check the symbols plugin_init and confirm_authorization
     * - run plugin_init
     */
    ientry=0;
    ihandle=*lcas_db_handle;
    while (ihandle)
    {
        if (strlen(ihandle->pluginname) > 0)
        {
            lcas_log_debug(1,"lcas.mod-lcas_init(): initializing plugin %s (db entry %d)\n",
                     ihandle->pluginname, ientry);
            if ((plugin_entry=PluginInit(ihandle,&plugin_list)) == NULL)
            {
                lcas_log(0,"lcas.mod-lcas_init(): error initializing plugin: %s\n",ihandle->pluginname);
                goto fail_lcas_init_and_logfile;
            }
            /* 
             * Check if plugin module is already registered as standard authorization module,
             * by comparing the complete path names of the plugin and modules
             */
            authmod_entry=authmod_list;
            while (authmod_entry)
            {
                if ( (strncmp(authmod_entry->pluginname,
                              plugin_entry->pluginname,
                              LCAS_MAXPATHLEN) == 0)
                   )
                {
                    lcas_log(0,"lcas.mod-lcas_init() error: plugin %s already registered as\n",
                             plugin_entry->pluginname);
                    lcas_log(0,"    standard authorization module\n");
                    goto fail_lcas_init_and_logfile;
                }
                authmod_entry=authmod_entry->next;
            }
        }
        ientry++;
        ihandle=ihandle->next;
    }

    authmod_entry=authmod_list;
    while (authmod_entry)
    {
        print_lcas_plugin(2,authmod_entry);
        lcas_log_debug(2,"\n");
        authmod_entry=authmod_entry->next;
    }

    plugin_entry=plugin_list;
    while (plugin_entry)
    {
        print_lcas_plugin(2,plugin_entry);
        lcas_log_debug(2,"\n");
        plugin_entry=plugin_entry->next;
    }

    /* clean STANDARD authorization module database structure */
    if(lcas_db_clean_list(&lcas_mod_db))
    {
        lcas_log(0,"lcas.mod-lcas_init() error: could not clean up authmod db structure\n");
        goto fail_lcas_init_and_logfile;
    }
    /* clean PLUGIN authorization module database structure */
    if (lcas_db_clean())
    {
        lcas_log(0,"lcas.mod-lcas_init() error: could not clean up plugin db structure\n");
        goto fail_lcas_init_and_logfile;
    }

    /* success */
    if (lcas_db_file) free(lcas_db_file);
    lcas_initialized++;
    return 0;

 fail_lcas_init_and_logfile:
    /* failure */
    lcas_db_clean_list(&lcas_mod_db);
    lcas_db_clean();
    if (clean_plugin_list(&plugin_list)!=0)
    {
        lcas_log(0,"lmas.mod-lcas_init() error: could not clean up plugin list\n");
    }
    if (lcas_db_file) free(lcas_db_file);

    return 1;
}


/******************************************************************************
Function:   lcas_init_and_log
Description:
    Select logging type
    Setup logging by providing a file handle, error handling (not yet).
    Read from LCAS database the plugins to be loaded.
    Initialize the plugins

Parameters:
    fp: file handle for logging (from gatekeeper or other previously opened file handle)
        If the file handle is zero, assume that only syslogging is requested
    logtype: type of logging (usrlog and/or syslog)

Returns:
    0: initialization succeeded
    1: initialization failed
******************************************************************************/
int lcas_init_and_log(
        FILE* fp,
        unsigned short logtype
)
{
    return lcas_init_and_logfile(NULL, fp, logtype);
}


/******************************************************************************
Function:   lcas_init
Description:
    Initialize LCAS module: 
    setup logging, error handling
    read from LCAS database the plugins to be loaded

Parameters:
    fp: file handle for logging (from gatekeeper)
Returns:
    0: initialization succeeded
    1: initialization failed
******************************************************************************/
int lcas_init(
        FILE* fp
)
{
    /* set logging file descriptor, for the moment the gatekeeper logfile */
    /* if fp == NULL --> syslogging, otherwise only DO_USRLOG */
    if (fp)
    {
        return lcas_init_and_log(fp,DO_USRLOG);
    }
    else
    {
        return lcas_init_and_log(NULL,DO_SYSLOG);
    }
}


/******************************************************************************
Function:   PluginInit
Description:
    Initialize the plugin
    - Create entry in (plugin)list
    - Open plugins and check the symbols plugin_init and confirm_authorization
    - run plugin_init

Parameters:
    db_handle: handle to LCAS db (containing pluginname and pluginargs)
    list:      pointer to list to which (plugin) module has to be added
Returns:
    pointer to plugin structure
    NULL: could not initialize plugin
******************************************************************************/
/*!
    \fn PluginInit(
        lcas_db_entry_t * db_handle,
        lcas_plugindl_t ** list
        )
    \brief Initialize the plugin.

    This function takes a plugin LCAS database entry and performs the following tasks:
    - Create entry in (plugin)list
    - Open the plugins and check the symbols plugin_init and confirm_authorization
    - run plugin_init

    \param db_handle handle to LCAS db (containing pluginname and pluginargs)
    \param list      pointer to plugin structure list to which (plugin) module has to be added

    \return pointer to newly created plugin structure or NULL in case of failure
    \internal
*/
static lcas_plugindl_t * PluginInit(
        lcas_db_entry_t * db_handle,
        lcas_plugindl_t ** list
)
{
    char *                 name;
#if 0
    char *                 names[5]={NULL,NULL,NULL,NULL,NULL};
#endif
    char *                 pname=NULL;
    char *                 args=NULL;
#if 0
    char *                 db;
    char *                 dbs[5]={NULL,NULL,NULL,NULL,NULL};
    char *                 pdb=NULL;
#endif
    void *                 plugin_handle;
    lcas_proc_t            plugin_procs[MAXPROCS] = {NULL};
    lcas_plugindl_t *      pplugin=NULL;
    int                    i;
    int                    retval;

    name = db_handle->pluginname;
#if 0
    db = db_handle->pluginargs;
#endif
    args = db_handle->pluginargs;

    /* Find plugin module */
#if 0
    names[0]=lcas_genfilename(NULL,name,NULL);
    names[1]=lcas_genfilename("modules",name,NULL);
    names[2]=lcas_genfilename(LCAS_MOD_HOME,name,NULL);
    names[3]=lcas_genfilename(LCAS_LIB_HOME,name,NULL);
    names[4]=lcas_genfilename(LCAS_ETC_HOME,name,NULL);
    pname=lcas_getfexist(5,names[0],
                      names[1],names[2],
                      names[3],names[4]
          );
#endif
    pname = lcas_findplugin(name);
    if (pname == NULL)
    {
        lcas_log(0,
            "lcas.mod-PluginInit(): plugin %s not found\n",
            name
        );
        goto fail_PluginInit;
    }

#if 0
    /* Find database */
    if (strlen(db) > 0)
    {
        dbs[0]=lcas_genfilename(NULL,db,NULL);
        dbs[1]=lcas_genfilename("modules",db,NULL);
        dbs[2]=lcas_genfilename(LCAS_ETC_HOME,db,NULL);
        dbs[3]=lcas_genfilename(LCAS_MOD_HOME,db,NULL);
        dbs[4]=lcas_genfilename(LCAS_LIB_HOME,db,NULL);
        pdb=lcas_getfexist(5,dbs[0],
                        dbs[1],dbs[2],
                        dbs[3],dbs[4]
            );
    }
#endif

    /* Try a dlopen() */
    plugin_handle=dlopen(pname,RTLD_NOW);
//    plugin_handle=dlopen(pname,RTLD_NOW|RTLD_GLOBAL);
    if (!plugin_handle)
    {
        lcas_log(0,"lcas.mod-PluginInit(): dlopen error for %s:\n    %s\n",
            pname,dlerror()
        );
        goto fail_PluginInit;
    }

    /* Check symbols */
    for (i=0; i < MAXPROCS; ++i)
    {
        switch (i)
        {
            case INITPROC:
                plugin_procs[i]=get_procsymbol(plugin_handle,"plugin_initialize");
                if (plugin_procs[i] == NULL)
                {
                    lcas_log(0,"lcas.mod-PluginInit(): plugin %s not compliant\n",
                        name);
                    lcas_log(0,"lcas.mod-PluginInit(): missing function \"plugin_initialize()\"\n");
                    goto fail_PluginInit;
                }
                else
                {
                    lcas_log_debug(2,"lcas.mod-PluginInit(): found \"plugin_initialize()\"\n");
                }
                break;
            case AUTHPROC:
                plugin_procs[i]=get_procsymbol(plugin_handle,"plugin_confirm_authorization");
                if (plugin_procs[i] == NULL)
                {
                    lcas_log(0,"lcas.mod-PluginInit(): plugin %s not compliant\n",
                        name);
                    lcas_log(0,"lcas.mod-PluginInit(): missing function \"plugin_confirm_authorization()\"\n");
                    goto fail_PluginInit;
                }
                else
                {
                    lcas_log_debug(2,"lcas.mod-PluginInit(): found \"plugin_confirm_authorization()\"\n");
                }
                break;
            case AUTHPROCX509: /* Actually, this is kind of a hack to be able to work with PEM/X509
                               instead of gss_cred_id_t, not a required symbol */
                plugin_procs[i]=get_procsymbol(plugin_handle,"plugin_confirm_authorization_from_x509");
                if (plugin_procs[i] == NULL)
                {
                    lcas_log_debug(1,"lcas.mod-PluginInit(): plugin %s misses function \"plugin_confirm_authorization_from_X509()\" (not required)\n", name);
                }
                else
                {
                    lcas_log_debug(2,"lcas.mod-PluginInit(): found \"plugin_confirm_authorization_from_x509()\"\n");
                }
                break;
            case TERMPROC:
                plugin_procs[i]=get_procsymbol(plugin_handle,"plugin_terminate");
                if (plugin_procs[i] == NULL)
                {
                    lcas_log(0,"lcas.mod-PluginInit(): plugin %s not compliant\n",
                        name);
                    lcas_log(0,"lcas.mod-PluginInit(): missing function \"plugin_terminate()\"\n");
                    goto fail_PluginInit;
                }
                else
                {
                    lcas_log_debug(2,"lcas.mod-PluginInit(): found \"plugin_terminate()\"\n");
                }
                break;
            default:
                /* do nothing */
                plugin_procs[i]=NULL;
                break;
        }
    }
    /* Make a new entry at end of plugin list */
    if (!*list)
    {
        lcas_log_debug(2,"lcas.mod-PluginInit(): creating first pluginlist entry\n");
        *list=pplugin=(lcas_plugindl_t *)malloc(sizeof(lcas_plugindl_t));
    }
    else
    {
        lcas_log_debug(2,"lcas.mod-PluginInit(): creating new pluginlist entry\n");
        pplugin=*list;
        while (pplugin->next) pplugin=pplugin->next;
        pplugin=pplugin->next=(lcas_plugindl_t *)malloc(sizeof(lcas_plugindl_t));
    }
    if (!pplugin)
    {
        lcas_log(0,"lcas.mod-PluginInit(): error creating new pluginlist entry\n");
        goto fail_PluginInit;
    }
    /* Set plugin list elements */
    pplugin->next=NULL;
    pplugin->handle=plugin_handle;
    for (i=0; i < MAXPROCS; ++i)
        pplugin->procs[i]=plugin_procs[i];
    if (pname != NULL)
    {
        strncpy(pplugin->pluginname,pname,LCAS_MAXPATHLEN);
        (pplugin->pluginname)[LCAS_MAXPATHLEN]=NUL;
    }
    else
        strncpy(pplugin->pluginname,"",LCAS_MAXPATHLEN);

    if (args != NULL)
    {
        strncpy(pplugin->pluginargs,args,LCAS_MAXARGSTRING);
        (pplugin->pluginargs)[LCAS_MAXARGSTRING]=NUL;
    }
    else
        strncpy(pplugin->pluginargs,"",LCAS_MAXARGSTRING);

    /* Parse options */
    if (parse_args_plugin(pname,args,pplugin->argv,&(pplugin->argc)) )
    {
        lcas_log(0,"lcas.mod-PluginInit(): Could not parse arguments for plugin module %s\n",
        pname);
        goto fail_PluginInit;
    }
    for (i=0; i < pplugin->argc; ++i)
        lcas_log_debug(4,"%s arg %d = %s\n",pname,i,(pplugin->argv)[i]);


    /* Run plugin_initialize() */
#if 0
    retval=(*(pplugin->procs[INITPROC]))(pplugin->pluginargs);
#endif
    retval=((lcas_init_proc_t)pplugin->procs[INITPROC])(pplugin->argc,pplugin->argv);
    if (retval != LCAS_MOD_SUCCESS )
    {
        lcas_log(0,"lcas.mod-PluginInit(): \"plugin_initialize()\" failed\n");
        goto fail_PluginInit;
    }
    else
    {
        lcas_log_debug(2,"lcas.mod-PluginInit(): \"plugin_initialize()\" succeeded\n");
    }


    /* free stuff */
    if (pname) { free(pname); pname=NULL; }
#if 0
    for (i=0; i < 5; ++i)
    {
        if (names[i]) free(names[i]);
        if (dbs[i]) free(dbs[i]);
    }
#endif

    return pplugin;

 fail_PluginInit:
    if (pname) { free(pname); pname=NULL; }
#if 0
    for (i=0; i < 5; ++i)
    {
        if (names[i]) free(names[i]);
        if (dbs[i]) free(dbs[i]);
    }
#endif
    return NULL;
}

/******************************************************************************
Function:   parse_args_plugin()
Description:
    Parse the argument string of the plugin and create xargv and xargc

Parameters:
    name: name of the plugin (goes into argv[0])
    argstring: string containing the arguments
    xargv: array of argument strings (has to be freed later)
    xargc: number of arguments
Returns:
    0 on succes
    1 on failure
******************************************************************************/
/*!
    \fn parse_args_plugin(
        const char * name,
        const char * argstring,
        char ** xargv,
        int * xargc
        )
    \brief convert plugin argument string into xargc, xargv

    Parse the argument string of the plugin and create xargv and xargc
    \param name      name of the plugin (goes into xargv[0])
    \param argstring string containing the arguments
    \param xargv     array of argument strings (has to be freed later)
    \param xargc     number of arguments
    \retval 0 succes.
    \retval 1 failure.
    \internal
*/
static int parse_args_plugin(
        const char * name,
        const char * argstring,
        char ** xargv,
        int * xargc
)
{
    int len=0;
    int i;
    int rc;

    /* set modulename */
    len=strlen(name);
    if ( (len > 0) && (len < LCAS_MAXPATHLEN) )
    {
        xargv[0]=(char *)malloc(len+1);
        if (xargv[0] != NULL)
            strncpy(xargv[0],name,len+1);
        else
            return 1;
    }

    *xargc=LCAS_MAXARGS-1;
    if ( (rc=lcas_tokenize(argstring, &xargv[1], xargc, " \t\n")) )
    {
        lcas_log(0,"lcas.mod-parse_args_plugin(): something wrong parsing arguments of %s, rc=%d\n",
                 name, rc);
        (*xargc)++;
        return 1;
    }
    (*xargc)++; /* increase by one for xargv[0] */
    lcas_log_debug(3,"lcas.mod-parse_args_plugin(): Found %d arguments:\n", *xargc);
    for (i=0; i < *xargc; i++)
        lcas_log_debug(3,"lcas.mod-parse_args_plugin(): %d --> %s\n", i, xargv[i]);

    return 0;
}

/******************************************************************************
Function:   get_procsymbol
Description:
    get procedure symbol from dlopen-ed library

Parameters:
    handle:  handle of dynamic library
    symname: name of procedure symbol
Returns:
    handle to procedure symbol
    NULL: Could not get symbol
******************************************************************************/
/*!
    \fn get_procsymbol(
        void * handle,
        char * symname
        )
    \brief get procedure symbol from dlopen-ed library
    \param handle  handle of dynamic library
    \param symname name of procedure symbol
    \return handle to procedure symbol or NUll
    \internal
*/
static lcas_proc_t get_procsymbol(
        void * handle,
        char * symname
)
{
    lcas_proc_t symhandle;
    char *error;

    symhandle=dlsym(handle,symname);
    if ((error = dlerror()) != NULL)
    {
        lcas_log_debug(1,"lcas.mod-get_procsymbol(): dlsym error: %s\n",error);
        return NULL;
    }
    return symhandle;
}

/******************************************************************************
Function:   clean_plugin_list
Description:
    clean (free) the list of plugins and call the plugin termination functions

Parameters:
    list:      pointer to list of plugins which has to be freeed.

Returns:
    1: succes
    0: failure
******************************************************************************/
/*!
    \fn clean_plugin_list(
        lcas_plugindl_t ** list
        )
    \brief clean (free) the list of plugins and call the plugin termination functions
    \param list
    \param list      pointer to list of plugins which has to be freeed.
    \retval 0 succes.
    \retval 1 failure.
    \internal
*/
static int clean_plugin_list(
        lcas_plugindl_t ** list
)
{
    int                           rc;
    lcas_plugindl_t *             plugin_entry=NULL;

    plugin_entry=*list;
    while (plugin_entry)
    {
        lcas_plugindl_t * plugin_next;
        int               i;

        rc = ((lcas_term_proc_t)plugin_entry->procs[TERMPROC])();
        if (rc != LCAS_MOD_SUCCESS)
        {
            lcas_log(0,"lcas.mod-clean_plugin_list(): failed to terminate plugin module %s\n",
                    plugin_entry->pluginname);
        }
        lcas_log_debug(1, "lcas.mod-clean_plugin_list(): plugin module %s terminated\n",
                plugin_entry->pluginname);
        dlclose(plugin_entry->handle);

        plugin_next=plugin_entry->next;
        for (i=0; i < plugin_entry->argc; i++)
        {
            if ((plugin_entry->argv)[i] != NULL)
            {
                lcas_log_debug(3,"Freeing %d - %s\n",i,(plugin_entry->argv)[i]);
                free((plugin_entry->argv)[i]);
            }
        }
        free(plugin_entry);
        plugin_entry=plugin_next;
    }
    *list=plugin_entry=NULL;
    return 0;
}

/******************************************************************************
Function:   print_lcas_plugin
Description:
    print the lcas_plugindl_t structure

Parameters:
    debug_lvl: debugging level
    plugin:    plugin structure
Returns:
    1: succes
    0: failure
******************************************************************************/
/*!
    \fn print_lcas_plugin(
        int debug_lvl,
        lcas_plugindl_t * plugin
        )
    \brief print the lcas_plugindl_t structure
    \param debug_lvl debugging level
    \param plugin    plugin structure
    \retval 0 succes.
    \retval 1 failure.
    \internal
*/
static int print_lcas_plugin(
        int debug_lvl,
        lcas_plugindl_t * plugin
)
{
    int i=0;

    lcas_log_debug(debug_lvl,"\tplugin name                             : %s\n",plugin->pluginname);
    lcas_log_debug(debug_lvl,"\tplugin arguments                        : %s\n",plugin->pluginargs);
    lcas_log_debug(debug_lvl,"\tplugin address                          : %x\n",plugin);
    lcas_log_debug(debug_lvl,"\tplugin size                             : %d\n",sizeof(lcas_plugindl_t));
    lcas_log_debug(debug_lvl,"\tplugin handle                           : %x\n",plugin->handle);
    lcas_log_debug(debug_lvl,"\tplugin_initialize()                     : %x\n",plugin->procs[INITPROC]);
    lcas_log_debug(debug_lvl,"\tplugin_confirm_authorization()          : %x\n",plugin->procs[AUTHPROC]);
    lcas_log_debug(debug_lvl,"\tplugin_confirm_authorization_from_x509(): %x\n",plugin->procs[AUTHPROCX509]);
    lcas_log_debug(debug_lvl,"\tplugin_terminate()                      : %x\n",plugin->procs[TERMPROC]);
    lcas_log_debug(debug_lvl,"\tplugin argc                             : %d\n",plugin->argc);
    for (i=0; i < plugin->argc; i++)
        lcas_log_debug(debug_lvl,"\tplugin argv[%2d]                        : %s\n",i,(plugin->argv)[i]);
    lcas_log_debug(debug_lvl,"\tplugin next                             : %x\n",plugin->next);
    if (plugin->next != NULL)
        lcas_log_debug(debug_lvl,"\tplugin_next                             : %s\n",(plugin->next)->pluginname);
    else
        lcas_log_debug(debug_lvl,"\tplugin_next                             : last plugin in list\n");

    return 1;
}

/******************************************************************************
Function:   lcas_get_fabric_authorization
Description:
    Call LCAS in order to get authorization on the local fabric

Parameters:
    user_cred : user globus credential handle
    request: RSL string
Returns:
    0: authorization succeeded
    1: authorization failed
******************************************************************************/
#if ALLOW_EMPTY_CREDENTIALS
#ifndef NOGLOBUS
int lcas_get_fabric_authorization(
        char * user_dn_tmp,
        gss_cred_id_t user_cred,
        lcas_request_t request
)
{
    return lcas_run_va(LCAS_ARG_GSS_DN, user_dn_tmp, user_cred, request);
}
#else
int lcas_get_fabric_authorization(
        char * user_dn_tmp,
        void * user_cred,
        lcas_request_t request
)
{
    return lcas_run_va(LCAS_ARG_GSS_DN, user_dn_tmp, user_cred, request);
}
#endif /* NOGLOBUS */
#else
#ifndef NOGLOBUS
int lcas_get_fabric_authorization(
        gss_cred_id_t user_cred,
        lcas_request_t request
)
{
    return lcas_run_va(LCAS_ARG_GSS, user_cred, request);
}
#else
int lcas_get_fabric_authorization(
        void * user_cred,
        lcas_request_t request
)
{
    return lcas_run_va(LCAS_ARG_GSS, user_cred, request);
}
#endif /* NOGLOBUS */
#endif /* ALLOW_EMPTY_CREDENTIALS */


/******************************************************************************
Function:   lcas_run_va
Description:
    Call LCAS in order to get authorization on the local fabric

Parameters:
    arg_type:  specifies the type of arguments that will follow
    - arg_type == LCAS_ARG_PEM
        pem_string : the user's (proxy) certificate as a PEM-encoded string
        request: RSL string
    - arg_type == LCAS_ARG_GSS
        user_cred :  the user's credential as a gss_cred_id_t structure
        request: RSL string
    - arg_type == LCAS_ARG_GSS_DN
        user_dn   :  the user's DN
        user_cred :  the user's credential as a gss_cred_id_t structure
        request: RSL string
Returns:
    0: authorization succeeded
    1: authorization failed
******************************************************************************/
int lcas_run_va(
        lcas_argtype_t  arg_type,
        ...
)
{
    va_list             ap;
    char *              pem_string = NULL;
    char *              user_dn = NULL;
    char *              user_dn_tmp = NULL;
    char *              user_dn_new = NULL;
#ifndef NOGLOBUS
    gss_cred_id_t       user_cred = GSS_C_NO_CREDENTIAL;
#else
    void *              user_cred = NULL;
#endif /* NOGLOBUS */
    lcas_request_t      request = NULL;
    int                 rc;
    int                 retval;
    int                 lcas_authorized = 0;
    char *              logstr = "lcas.mod-lcas_run_va()";
    X509 *              px509 = NULL;
    STACK_OF(X509) *    px509_chain = NULL;

    lcas_plugindl_t *   plugin_entry;
    lcas_plugindl_t *   authmod_entry;

    /* Default authorization = no authorization */
    lcas_authorized=0;

    if (lcas_initialized == 0)
    {
        fprintf(stderr,"LCAS has to be initialized first !\n");
        retval = FAILED_LCAS_OTHER;
        goto fail_lcas_run_va;
    }

    // lcas_log_debug(0,"\n");
    lcas_log (2,"LCAS authorization request\n");

    /* Read input variables from va_list */
    va_start(ap, arg_type);
    switch (arg_type)
    {
        case LCAS_ARG_PEM:
            pem_string = va_arg(ap, char *);
            request = va_arg(ap, lcas_request_t);
            lcas_log_debug (5, "%s: got input for LCAS_ARG_PEM execution\n", logstr);
            break;
        case LCAS_ARG_GSS:
#ifndef NOGLOBUS
            user_cred = va_arg(ap, gss_cred_id_t);
#else
            user_cred = va_arg(ap, void *);
#endif /* NOGLOBUS */
            request = va_arg(ap, lcas_request_t);
            logstr = "lcas.mod-lcas_get_fabric_authorization()"; /* Backwards compatibility for logging */
            lcas_log_debug (5, "%s: got input for LCAS_ARG_GSS execution\n", logstr);
            break;
        case LCAS_ARG_GSS_DN:
            user_dn_tmp = va_arg(ap, char *);
#ifndef NOGLOBUS
            user_cred = va_arg(ap, gss_cred_id_t);
#else
            user_cred = va_arg(ap, void *);
#endif /* NOGLOBUS */
            request = va_arg(ap, lcas_request_t);
            lcas_log_debug (5, "%s: got input for LCAS_ARG_GSS_DN execution\n", logstr);
            break;
        default:
            retval = FAILED_LCAS_OTHER;
            lcas_log(0, "%s: Unknown LCAS argument type (arg_type=%d) (failure)\n",
                       logstr, arg_type);
            goto fail_lcas_run_va;
    }
    va_end(ap);

    /*
     * PEM parsing: extract X509 cert + chain + DN
     */
    if (arg_type == LCAS_ARG_PEM)
    {
        lcas_log_debug (2, "%s: Extracting X509 Chain from PEM string\n", logstr);
        if (lcas_pem_string_to_x509_chain(&px509_chain, pem_string) != 0)
        {
            lcas_log(0, "%s: Cannot find certificate chain in pem string(failure)\n", logstr);
            retval = FAILED_LCAS_OTHER;
            goto fail_lcas_run_va;
        }
        if (lcas_pem_string_to_x509(&px509, pem_string) != 0)
        {
            lcas_log(0, "%s: Cannot find (proxy) certificate in pem string (failure)\n", logstr);
            retval = FAILED_LCAS_OTHER;
            goto fail_lcas_run_va;
        }
        if ((user_dn_new = lcas_x509_chain_to_dn(px509, px509_chain)) == NULL)
        {
            lcas_log(0, "%s: Cannot find extract DN from X509 certificate and chain (failure)\n", logstr);
            retval = FAILED_LCAS_OTHER;
            goto fail_lcas_run_va;
        }
        user_dn_tmp = user_dn_new;
        lcas_log_debug(2, "%s: Parsing of PEM string succeeded\n", logstr);
    }

#ifndef NOGLOBUS
    /*
     * GSS parsing: extract X509 cert + chain (+ DN later in lcas_cred_fill)
     */
    if ((arg_type == LCAS_ARG_GSS) || (arg_type == LCAS_ARG_GSS_DN))
    {
        if ( ( px509 = lcas_cred_to_x509(user_cred) ) )
        {
            lcas_log_debug(1,"%s: found X509 struct inside gss credential\n", logstr);
        }
        else
        {
            lcas_log(0,"%s: could not get X509 cred from gss credential!\n", logstr);
            retval = FAILED_LCAS_OTHER;
            goto fail_lcas_run_va;
        }
        if ( ( px509_chain = lcas_cred_to_x509_chain(user_cred) ) )
        {
            lcas_log_debug(1,"%s: found X509 chain inside gss credential\n", logstr);
        }
        else
        {
            lcas_log(0,"%s: could not get X509 chain from gss credential!\n", logstr);
            retval = FAILED_LCAS_OTHER;
            goto fail_lcas_run_va;
        }
        lcas_log_debug(1, "%s: Parsing of gss credential succeeded\n", logstr);
    }
#endif /* NOGLOBUS */

    /*
     * Create lcas credential (checks if dn can be extracted)
     */
    if ( lcas_fill_cred(user_dn_tmp, user_cred, &lcas_cred) != 0)
    {
        lcas_log(0,"%s error: could not create lcas credential, something wrong\n", logstr);
        lcas_log(0,"                                              : with user DN and user credential\n");
        retval = FAILED_LCAS_OTHER;
        goto fail_lcas_run_va;
    }
    user_dn = lcas_get_dn(lcas_cred);
    if (user_dn == NULL)
    {
        lcas_log(0, "%s error: user DN empty\n", logstr);
        retval = FAILED_LCAS_OTHER;
        goto fail_lcas_run_va;
    }

    lcas_log_debug(2, "%s: user is %s\n", logstr, user_dn);

    /*
     *             - Call the STANDARD authorization modules
     *               OBSOLETE!!!
     */
    authmod_entry=authmod_list;
    while (authmod_entry)
    {
        rc = ((lcas_auth_proc_t)authmod_entry->procs[AUTHPROC])(request, lcas_cred);
        if (rc != LCAS_MOD_SUCCESS)
        {
            lcas_log_debug(0,"%s: authorization failed for standard module %s\n",
                    logstr, authmod_entry->pluginname);
            retval = FAILED_LCAS_PLUGIN;
            goto fail_lcas_run_va;
        }
        lcas_log_debug(0, "%s: authorization granted by standard module %s\n",
                logstr, authmod_entry->pluginname);
        lcas_authorized++;

        authmod_entry=authmod_entry->next;
    }

    /*
     *             - Call the PLUGIN authorization modules
     */
    plugin_entry=plugin_list;
    while (plugin_entry)
    {

        /* Try AUTHPROCX509 if available, otherwise fall back to AUTHPROC */
        if (plugin_entry->procs[AUTHPROCX509])
            rc = ((lcas_authx509_proc_t)plugin_entry->procs[AUTHPROCX509])(request, lcas_cred, px509, px509_chain);
        else
            rc = ((lcas_auth_proc_t)plugin_entry->procs[AUTHPROC])(request, lcas_cred);
        if (rc != LCAS_MOD_SUCCESS)
        {
            lcas_log_debug(0,"%s: authorization failed for plugin %s\n",
                    logstr, plugin_entry->pluginname);
            retval = FAILED_LCAS_PLUGIN;
            goto fail_lcas_run_va;
        }
        lcas_log_debug(1, "%s: authorization granted by plugin %s\n",
                logstr, plugin_entry->pluginname);
        lcas_authorized++;

        plugin_entry=plugin_entry->next;
    }

    /*
     * To be done: 
     *             if the user is authorized call FLIDS component and return
     *             the (temporary) certificate.
     */

    /*
     * Check if lcas_authorized > 0
     */
    if ( !(lcas_authorized > 0) )
    {
        lcas_log_debug(0,"%s: No authorization modules were called (check lcas db file)\n", logstr);
        retval = FAILED_LCAS_OTHER;
        goto fail_lcas_run_va;
    }
    else
        lcas_log_debug(2,"%s: %d modules authorized you\n", logstr, lcas_authorized);

    /* success */
    if (user_dn_new) {free(user_dn_new); user_dn_new = NULL;}
    if (px509) X509_free(px509);
    lcas_x509_free_chain (&px509_chain);
    lcas_release_cred(&lcas_cred);
    lcas_log_debug(0,"%s: succeeded\n", logstr);
    return 0;

 fail_lcas_run_va:
    /* failure */
    if (user_dn_new) {free(user_dn_new); user_dn_new = NULL;}
    if (px509) X509_free(px509);
    lcas_x509_free_chain (&px509_chain);
    lcas_release_cred(&lcas_cred);
    lcas_log_debug(0,"%s: failed\n", logstr);
    return retval;
}

/******************************************************************************
Function:   lcas_term
Description:
    Terminate LCAS module: 

Parameters:
Returns:
    0: termination succeeded
    1: termination failed
******************************************************************************/
int lcas_term()
{
    int                           rc;
    lcas_plugindl_t *             plugin_entry;
    lcas_plugindl_t *             authmod_entry;

    // lcas_log_debug(0,"\n");
    lcas_log (1,"Termination LCAS\n");
    /*
     *             - Terminate the STANDARD authorization modules
     *               and clean up the authmod structure
     */
    authmod_entry=authmod_list;
    while (authmod_entry)
    {
        lcas_plugindl_t * authmod_next;
        int               i;

        rc = authmod_entry->procs[TERMPROC]();
        if (rc != LCAS_MOD_SUCCESS)
        {
            lcas_log(0,"lcas.mod-lcas_term(): failed to terminate standard module %s\n",
                    authmod_entry->pluginname);
            return 1;
        }
        lcas_log_debug(1, "lcas.mod-lcas_term(): standard module %s terminated\n",
                authmod_entry->pluginname);
        dlclose(authmod_entry->handle);

        authmod_next=authmod_entry->next;
        for (i=0; i < authmod_entry->argc; i++)
        {
            if ((authmod_entry->argv)[i] != NULL)
            {
                lcas_log_debug(3,"Freeing %d - %s\n",i,(authmod_entry->argv)[i]);
                free((authmod_entry->argv)[i]);
            }
        }
        free(authmod_entry);
        authmod_entry=authmod_next;
    }
    authmod_list=authmod_entry=NULL;

    /*
     *             - Terminate the PLUGIN authorization modules
     *               and clean up the plugin structure
     */
    plugin_entry=plugin_list;
    while (plugin_entry)
    {
        lcas_plugindl_t * plugin_next;
        int               i;

        rc = plugin_entry->procs[TERMPROC]();
        if (rc != LCAS_MOD_SUCCESS)
        {
            lcas_log(0,"lcas.mod-lcas_term(): failed to terminate plugin module %s\n",
                    plugin_entry->pluginname);
            return 1;
        }
        lcas_log_debug(1, "lcas.mod-lcas_term(): plugin module %s terminated\n",
                plugin_entry->pluginname);
        dlclose(plugin_entry->handle);

        plugin_next=plugin_entry->next;
        for (i=0; i < plugin_entry->argc; i++)
        {
            if ((plugin_entry->argv)[i] != NULL)
            {
                lcas_log_debug(3,"Freeing %d - %s\n",i,(plugin_entry->argv)[i]);
                free((plugin_entry->argv)[i]);
            }
        }
        free(plugin_entry);
        plugin_entry=plugin_next;
    }
    plugin_list=plugin_entry=NULL;

    /* close logging (safe if logfile is not owned by LCAS) */
    if (lcas_log_close() != 0)
        return 1;

    lcas_initialized=0;
    return 0;
}

#endif /* LCAS_C */

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcas/src/lcas.c,v $
    $Date: 2010-05-03 10:42:49 $
    $Revision: 2.31 $
    $Author: okoeroo $
******************************************************************************/
