/*	Local_Theater

PIRL CVS ID: Local_Theater.java,v 1.41 2012/04/16 06:04:11 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Conductor.Maestro;

import	PIRL.Conductor.Management;
import	PIRL.Conductor.Processing_Listener;
import	PIRL.Conductor.Processing_Event;
import	PIRL.Messenger.*;
import	PIRL.Configuration.Configuration;
import	PIRL.Configuration.Configuration_Exception;
import	PIRL.PVL.Value;
import	PIRL.PVL.PVL_Exception;
import	PIRL.Utilities.Styled_Multiwriter;
import	PIRL.Utilities.Suspendable_Styled_Writer;

import	java.io.Writer;
import	java.io.IOException;
import	java.net.ConnectException;
import	java.net.MulticastSocket;
import	java.net.DatagramPacket;
import	java.net.InetAddress;
import	java.util.concurrent.BlockingQueue;
import	java.util.concurrent.LinkedBlockingQueue;
import	java.util.Vector;


/**	A <i>Local_Theater</i> is the local Conductor management side of a
	{@link Theater}.
<p>
	A Local_Theater is used by a {@link PIRL.Conductor.Conductor} to
	provide network distributed clients access to its {@link
	PIRL.Conductor.Management} interface via a {@link Stage_Manager}
	proxy server.
<p>
	A Local_Theater implements the {@link Message_Delivered_Listener}
	interface. Each {@link Message} delivered via a Stage_Manager {@link
	Messenger} is examined for an {@link Message#Action() action}
	parameter that determines how the remaining content will be mapped
	to a Management method of the Conductor bound to the Local_Theater,
	and any return values used to assemble the appropriate {@link
	Message#Reply_To(Message) reply} sent back.
<p>
	A Local_Theater also implements the {@link Processing_Listener}
	interface that receives {@link Processing_Event} notifications from
	the Conductor bound to the Local_Theater. The {@link
	PIRL.Conductor.Processing_Changes} of the event are used to assemble
	the appropriate Message that is sent to all clients that have
	registered, via a Message with a {@link
	Theater#ADD_PROCESSING_LISTENER_ACTION}, an interest in the
	Conductor's processing events. In addition, any clients that have
	requested, via a Message with a {@link
	Theater#ADD_LOG_WRITER_ACTION}, to receive the Conductor processing
	log stream will have the stream sent via Messages.
<p>
	@author		Bradford Castalia - UA/PIRL
	@version	1.41
*/
public class Local_Theater
	extends
		Theater
	implements
		Message_Delivered_Listener,
		Processing_Listener
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor.Maestro.Local_Theater (1.41 2012/04/16 06:04:11)";


private Management
	The_Management;

private String
	Theater_Key						= null;

/**	The name of the Theater Message protocol processing Thread.
*/
public static final String
	ASSISTANT_NAME					= "Local_Theater Assistant";

private Assistant
	Assistant						= null;
private final BlockingQueue<Message_Delivered_Event>
	Tasks = new LinkedBlockingQueue<Message_Delivered_Event> ();

/**	The default port on which to receive the Hello datagram.
*/
public static final int
	DEFAULT_HELLO_PORT				= Dispatcher.DEFAULT_HELLO_PORT;
private int
	Hello_Port						= DEFAULT_HELLO_PORT;

/**	The default private multicast address to use in the Hello datagram.
*/
public static final String
	DEFAULT_HELLO_ADDRESS			= Dispatcher.DEFAULT_HELLO_ADDRESS;
private String
	Hello_Address					= DEFAULT_HELLO_ADDRESS;

/**	The content of the Hello broadcast datagram message.
*/
public static final String
	HELLO_MESSAGE					= Dispatcher.HELLO_MESSAGE;

private Listen_for_Hello
	Hello_Listener					= null;
private IOException
	Hello_Listener_Exception		= null;

private boolean
	Auto_Open						= true;
//	Delay (seconds) to allow the Stage_Manager to be ready for connections.
private static final int
	AUTO_OPEN_DELAY					= 4;


private Vector<Value>
	Processing_Listener_Routes	= new Vector<Value> ();

private Styled_Multiwriter
	Log_Writer					= new Styled_Multiwriter ();

/**	The Theater Message protcol error statistics.
*/
protected int
	Undeliverable_Messages		= 0,
	NACK_Messages				= 0,
	Unrecognized_Messages		= 0;


// Debug control.
private static final int
	DEBUG_OFF					= 0,
	DEBUG_CONNECTION			= 1 << 1,
	DEBUG_MESSAGES				= 1 << 2,
	DEBUG_PROCESSING_LISTENER	= 1 << 3,
	DEBUG_LOG_WRITER			= 1 << 4,
	DEBUG_HELLO_LISTENER		= 1 << 5,
	DEBUG_ALL					= -1,

	DEBUG						= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct a Local_Theater for a Conductor Management interface.
<p>
	@param	management	The Conductor Management interface to use with
		the Theater Message protocol.
	@throws	IllegalArgumentException	If the management is null.
*/
public Local_Theater
	(
	Management	management
	)
{
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		(">-< Local_Theater");
if ((The_Management = management) == null)
	throw new IllegalArgumentException (ID + NL
		+ "Can't construct without Management.");
}

//	A Local_Theater can not operate without Management.
private Local_Theater ()
{}

/*==============================================================================
	Accessors
*/
/**	Set the key to be used when {@link #Open(String) open}ing a Stage_Manager
	connection during {@link #Auto_Open(boolean) auto-open} operations.
<p>
	@param	key	The Stage_Manager authentication key.
	@return	This Local_Theater.
*/
public Local_Theater Key
	(
	String	key
	)
{
Theater_Key = key;
return this;
}

/**	Test if auto-open has been enabled.
<p>
	@return	true if auto-open is enabled; false otherwise.
	@see	#Auto_Open(boolean)
*/
public boolean	Auto_Open ()
{return Auto_Open;}

/**	Enable or disable auto-open mode.
<p>
	The current {@link #Hello_Listener_Port() "Hello" listener port}
	and {@link #Hello_Listener_Address() "Hello" listener address} will
	be used.
<p>
	@param	enable	If true, auto-open is enabled; otherwise it is disabled.
	@return	This Local_Theater.
	@see	#Auto_Open(boolean, int, String)
*/
public Local_Theater Auto_Open
	(
	boolean	enable
	)
{return Auto_Open (enable, Hello_Port, Hello_Address);}

/**	Enable or disable auto-open mode.
<p>
	When auto-open has been enabled this Local_Theater will attempt to
	stay open once it has been told to {@link #Open(String) open}.
	Failure to open will result in a Thread being started that will
	{@link #Listening_for_Hello() listen} for a {@link #HELLO_MESSAGE}
	broadcast on a specified port of a network multicast address. When
	the message is received listening stops, an attempt is made to open
	this Local_Theater again, and the Thread dies. In addition, if a
	{@link Theater#DONE_ACTION} Message is received after this
	Local_Theater has been opened another attempt to open will be made.
<p>
	<b>N.B.</b>: Enabling auto-open after an open has failed or this
	Local_Theater is done will not cause an open to be tried again.
<p>
	@param	enable	If true, auto-open is enabled; otherwise it is disabled.
	@param	port	The port number to use when listening for a "Hello"
		broadcast. If less than or equal to zero the {@link
		#DEFAULT_HELLO_PORT} will be used.
	@param	address	The network multicast address to use when listening
		for a "Hello" broadcast. If null the {@link
		#DEFAULT_HELLO_ADDRESS} will be used.
	@return	This Local_Theater.
*/
public Local_Theater Auto_Open
	(
	boolean	enable,
	int		port,
	String	address
	)
{
Auto_Open = enable;

if (port <= 0)
	Hello_Port = DEFAULT_HELLO_PORT;
else
	Hello_Port = port;

if (address == null ||
	address.length () == 0)
	Hello_Address = DEFAULT_HELLO_ADDRESS;
else
	Hello_Address = address;

return this;
}

/**	Get the port number that will be used to listen for an
	{@link #Auto_Open(boolean, int, String) auto-open} "Hello" broadcast.
<p>
	@return	The port number on which to listen for an auto-open "Hello"
		broadcast.
*/
public int Hello_Listener_Port ()
{return Hello_Port;}

/**	Get the network multicast address that will be used to listen for an
	{@link #Auto_Open(boolean, int, String) auto-open} "Hello" broadcast.
<p>
	@return	The network multicast address String on which to listen for an
		auto-open "Hello" broadcast.
*/
public String Hello_Listener_Address ()
{return Hello_Address;}

/**	Test if this Local_Theater is listening for a {@link
	#Auto_Open(boolean, int, String) auto-open} "Hello" broadcast.
<p>
	@return	true if a Thread is running that is listening for a "Hello"
		broadcast; false otherwise.
*/
public boolean Listening_for_Hello ()
{return Hello_Listener == null;}

/**	Get any exception that occured in an {@link #Auto_Open(boolean, int,
	String) auto-open} "Hello" broadcast listener Thread.
<p>
	The exception may be from either the "Hello" broadcast listener
	socket or the {@link #Open(String) open} operation. In the former
	case no attempt will be made to open this Local_Theater again to
	prevent an endless loop of auto-open broadcast listener socket
	errors.
<p>
	@return	The most recent IOException that occurred in a "Hello"
		broadcast listener Thread. This will be null if no exception has
		occured.
*/
public IOException Hello_Listener_Exception ()
{return Hello_Listener_Exception;}

/**	Get the number of undeliverable Theater protocol Messages that
	have been encountered.
<p>
	@return	The number of undeliverable Theater protocol Messages that
		have been encountered.
*/
public int Undeliverable_Messages ()
{return Undeliverable_Messages;}

/**	Get the number of NACK (negative acknowledgment) Messages that have
	been received.
<p>
	@return	The number of NACK Messages that have been received.
*/
public int NACK_Messages ()
{return NACK_Messages;}

/**	Get the number of Messages encountered that contain content not
	recognized by the Theater protocol.
<p>
	@return	The number of unrecognized Messages that have been
		encountered.
*/
public int Unrecognized_Messages ()
{return Unrecognized_Messages;}

/**	Get a description of this Local_Theater.
<p>
	The description is a line containing the Local_Theater {@link #ID}
	followed by the base Theater {@link Theater#toString() description}.
<p>
	@return	The Local_Theater/Theater description String.
*/
public String toString ()
{
return ID + NL
	+ super.toString ();
}

/*==============================================================================
	Stage_Manager Connection
*/
/**	Open this Local_Theater.
<p>
	If this Local_Theater is not {@link Theater#Opened() opened} a {@link
	Theater#Open(Message) new connection} is attempted to the
	Stage_Manager port on the localhost using the {@link
	#Listener_Identity() identity} of the Management object provided at
	construction time supplemented by the specified authentication key.
	<b>N.B.</b>: The effective authenticiation key for use by auto-open
	operations is reset to the specified key only if the latter is
	non-null.
<p>
	Once the connection to the Stage_Manager has been established this
	Local_Theater object is set as the {@link
	Theater#Employer(Message_Delivered_Listener) employer} of the
	Messenger used to {@link #Message_Delivered(Message_Delivered_Event)
	deliver Messages} from the Stage_Manager. Then this Messenger is told
	to begin {@link Messenger#Listen_for_Messages() listening for
	messages}.
<p>
	If the Stage_Manager connection failed to open and {@link
	#Auto_Open(boolean, int, String) auto-open} has been enabled a Thread
	is started that will {@link #Listening_for_Hello() listen} for a
	{@link #HELLO_MESSAGE} broadcast on a specified port of a network
	multicast address indicating that a Stage_Manager is available to
	attempt the connection again. The exception that resulted from
	failing to open the connection is always thrown regardless of whether
	auto-open has been enabled.
<p>
	<b>N.B.</b>: If this Local_Theater has already been {@link
	Theater#Opened() opened} nothing is done. In addition, if an auto-open
	{@link #Listening_for_Hello() "Hello" listener} Thread is active
	nothing is done even if this Local_Theater is not currently open; this
	prevents a open race condition.
<p>
	@param	key	The Stage_Manager {@link #KEY_PARAMETER_NAME} parameter
		value String. <b>N.B.</b>: If this key is null
	@throws	IOException	If a connection could not be established to the
		Stage_Manager. This will be a Theater_Protocol_Exception if there
		was a problem with any protocol Message content, including if the
		{@link #Listener_Identity() identity} from the Management
		interface object bound to this Local_Theater when it was
		constructed does not contain a valid PVL structure.
	@see	#Key(String)
*/
public void Open
	(
	String	key
	)
	throws IOException
{
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		(">>> Local_Theater.Open");
if (! Opened ())
	{
	if (Hello_Listener != null)
		{
		if ((DEBUG & DEBUG_CONNECTION) != 0)
			System.out.println
				("    Hello_Listener is already running." + NL
				+"<<< Local_Theater.Open");
		return;
		}

	if (key != null)
		Theater_Key = key;
	Message
		identity = null;
	try {identity = new Message (Listener_Identity ());}
	catch (PVL_Exception exception)
		{
		throw new Theater_Protocol_Exception (ID + NL
			+ "Invalid Management identity -" + NL
			+ Listener_Identity (),
			Theater_Protocol_Exception.INVALID_MESSAGE,
			exception);
		}
	identity.Set (KEY_PARAMETER_NAME, key);

	try {super.Open (identity);}
	catch (IOException exception)
		{
		if ((DEBUG & DEBUG_CONNECTION) != 0)
			System.out.println
				("    Unable to open the Theater." + NL
				+ exception + NL
				+"    Hello_Listener: " + Hello_Listener);
		if (Auto_Open)
			{
			//	Start a Hello_Listener Thread.
			if ((DEBUG & DEBUG_CONNECTION) != 0)
				System.out.println
					("    Starting a Listen_for_Hello thread");
			Hello_Listener = new Listen_for_Hello ();
			Hello_Listener.start ();
			}
		throw exception;
		}

	//	Start asychronous message listening.
	Employer (this);
	Listen_for_Messages ();
	}
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("<<< Local_Theater.Open: " + Opened ());
}

/**	Close this Theater.
<p>
	If this Theater is not {@link #Opened() open} nothing is done and
	false is returned.
<p>
	The log writer is {@link Management#Remove_Log_Writer(Writer)
	removed} from the Theater Management and it is {@link
	Styled_Multiwriter#Remove_All() cleared}. <b>N.B.</b>: The Writers
	contained in the log writer are not closed.
<p>
	This object is {@link
	Management#Remove_Processing_Listener(Processing_Listener) removed}
	from the Management and its lists of Messenger routes to receive
	Processing_Event notifications is cleared.
<p>
	The parent Theater is then closed.
<p>
	@return	true if this Theater was open at the time the method
		was called; false if the Theater was already closed.
*/
public synchronized boolean Close ()
{
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		(">>> Local_Theater.Close");
if (! Opened ())
	{
	if ((DEBUG & DEBUG_CONNECTION) != 0)
		System.out.println
			("    Already closed" + NL
			+"<<< Local_Theater.Close: false");
	return false;
	}
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("    Closing and clearing local Log_Writer");
//	Close down the log writers.
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("    Removing the Log_Writer from The_Management");
The_Management.Remove_Log_Writer (Log_Writer);
Log_Writer.Remove_All ();

//	Remove the Messenger route from the processing listeners list.
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("    Removing this Processing_Listener from The_Management");
The_Management.Remove_Processing_Listener (this);
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("    Clearing the Processing_Listener_Routes");
synchronized (Processing_Listener_Routes)
	{Processing_Listener_Routes.clear ();}

//	Close the Theater.
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("    Close this Theater");
boolean
	closed = super.Close ();
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("<<< Local_Theater.Close: " + closed);
return closed;
}

/*==============================================================================
	Hello_Listener
*/
private class Listen_for_Hello 
	extends Thread
{
public void run ()
{
if ((DEBUG & DEBUG_HELLO_LISTENER) != 0)
	System.out.println
		("==> Listen_for_Hello: Thread starting" + NL
		+"    Port - " + Hello_Port + NL
		+"    Address - " + Hello_Address);
IOException
	hello_listener_exception = null;
MulticastSocket
	socket = null;
InetAddress
	address = null;
try
	{
	address = InetAddress.getByName (Hello_Address);
	socket = new MulticastSocket (Hello_Port);
	socket.joinGroup (address);
	}
catch (IOException exception)
	{
	if ((DEBUG & DEBUG_HELLO_LISTENER) != 0)
		System.out.println
			("==> Listen_for_Hello: Unable to establish the socket -" + NL
			+ exception);
	hello_listener_exception = exception;
	}

if (hello_listener_exception == null)
	{
	byte[]
		buffer = new byte[32];
	DatagramPacket
		packet = new DatagramPacket (buffer, buffer.length);

	try
		{
		socket.receive (packet);
		if ((DEBUG & DEBUG_HELLO_LISTENER) != 0)
			System.out.println
				("==> Listen_for_Hello: \""
					+ new String (packet.getData (), 0, packet.getLength ())
					+ "\" message received");
		}
	catch (IOException exception) 
		{
		if ((DEBUG & DEBUG_HELLO_LISTENER) != 0)
			System.out.println
				("==> Listen_for_Hello: socket.receive exception -" + NL
				+ exception);
		hello_listener_exception = exception;
		}
	}

Hello_Listener = null;
if (socket != null)
	{
	try
		{
		socket.leaveGroup (address);
		socket.close ();
		}
	catch (IOException exception) {}
	}

if (hello_listener_exception == null &&
	Auto_Open)
	{
	//	Wait for the Stage_Manager to be ready to accept connections.
	try {sleep (AUTO_OPEN_DELAY * 1000);}
	catch (InterruptedException exception) {}

	if ((DEBUG & DEBUG_HELLO_LISTENER) != 0)
		System.out.println
			("==> Listen_for_Hello: Attempting to open the theater");
	try {Open (Theater_Key);}
	catch (ConnectException exception)
		{
		//	Another Listen_for_Hello thread may be run.
		if ((DEBUG & DEBUG_HELLO_LISTENER) != 0)
			System.out.println
				("==> Listen_for_Hello: theater opening failed -" + NL
				+ exception);
		}
	catch (IOException exception)
		{
		if ((DEBUG & DEBUG_HELLO_LISTENER) != 0)
			System.out.println
				("==> Listen_for_Hello: theater opening failed -" + NL
				+ exception);
		hello_listener_exception = exception;
		}
	}
Hello_Listener_Exception = hello_listener_exception;
if ((DEBUG & DEBUG_HELLO_LISTENER) != 0)
	System.out.println
		("<== Listen_for_Hello");
}

}	//	Listen_for_Hello

/*==============================================================================
	Messages
*/
//	Message_Delivered_Listener
/**	Get the identity of the Management object.
<p>
	@return	A Message containing the identity of the Management object
		provided at construction time.
*/
public Message Listener_Identity ()
{return The_Management.Identity ();}

//	Message_Delivered_Listener
/**	A message is delivered from the Stage_Manager.
<p>
	The event containing the delivered Message, and a reference to the
	Messenger that delivered it, is put on a LinkedBlockingQueue. The the
	Local_Theater Assistant Thread is started, if it is not already
	running. The Assistant will take events from the queue (FIFO) and
	process them according to the Theater Message protocol that conveys
	the Conductor Management interface information.
<p>
	Of particular interest is the handling of Messages with the {@link
	Theater#DONE_ACTION}: After {@link #Close() closing} this Local_Theater,
	if {@link #Auto_Open(boolean, int, String) auto-open} has been enabled
	an attempt is made to {@link #Open(String) open} the Local_Theater
	again using the current {@link #Key(String) key}.
<p>
	@param	event	The Message_Delivered_Event containing the Message.
	@see	PIRL.Conductor.Management
*/
public void Message_Delivered
	(
	Message_Delivered_Event	event
	)
{
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		(">>> Local_Theater.Message_Delivered: " + event.Message.Action () + NL
		+"    From -" + NL
		+ (Messenger)event.getSource () + NL
		+ ((Messenger)event.getSource ()).Identity () + NL
		+"    Message -" + NL
		+ event.Message.Routing () + NL
		+ event.Message);
try {Tasks.put (event);}
catch (InterruptedException exception)
	{
	Undeliverable_Messages++;
	try {Send_Message
		(Message.NACK ()
			.Add (ORIGINAL_MESSAGE_PARAMETER_NAME, event.Message)
			.Reply_To (event.Message));}
	catch (Exception e) {}
	}
if (Assistant == null)
	{
	Assistant = new Assistant ();
	Assistant.start ();
	}
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		("<<< Local_Theater.Message_Delivered");
}

/*------------------------------------------------------------------------------
	Assistant
*/
private class Assistant
	extends Thread
{
public Assistant ()
{super (ASSISTANT_NAME);}


public void run ()
{
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		("==> Local_Theater.Assistant");
Message_Delivered_Event
	event;
Messenger
	messenger;
String
	string;
Value
	value;

while (true)
	{
	try {event = Tasks.take ();}
	catch (InterruptedException exception) {break;}
	messenger = (Messenger)event.getSource ();
	if ((DEBUG & DEBUG_MESSAGES) != 0)
		System.out.println
			(NL
			+"--> Local_Theater.Assistant: " + event.Message.Action () + NL
			+"    From -" + NL
			+ messenger + NL
			+ messenger.Identity () + NL
			+"    Message -" + NL
			+ event.Message.Routing () + NL
			+ event.Message);

	switch (Action_Code (event.Message.Action ()))
		{
		case IDENTITY_CODE:
			try {Send_Message
				(The_Management.Identity ()
				.Reply_To (event.Message));}
			catch (Exception e) {}
			break;

		case START_CODE:
			The_Management.Start ();
			break;

		case PROCESSING_STATE_CODE:
			try {Send_Message
				(event.Message
				.Set (VALUE_PARAMETER_NAME,
					String.valueOf (The_Management.Processing_State ()))
				.Reply_To (event.Message));}
			catch (Exception e) {}
			break;

		case STOP_CODE:
			The_Management.Stop ();
			break;

		case CONFIGURATION_CODE:
			//	Remove any parameters with conflicting names.
			while (event.Message.Remove (CONFIGURATION_PARAMETER_NAME) != null);
			Configuration
				configuration = The_Management.Configuration ();
			if (configuration != null)
				event.Message
					.Set (NAME_PARAMETER_NAME,
						configuration.Source ())
					.Add (CONFIGURATION_PARAMETER_NAME,
						configuration);
			else
				event.Message.Set (EXPLANATION_PARAMETER_NAME,
					"No Configuration could be obtained.");
			try {Send_Message
				(event.Message
				.Reply_To (event.Message));}
			catch (Exception e) {}
			break;

		case SOURCES_CODE:
			try {Send_Message
				(event.Message
				.Set (TABLE_PARAMETER_NAME,
					The_Management.Sources ())
				.Reply_To (event.Message));}
			catch (Exception e) {}
			break;

		case PROCEDURES_CODE:
			try {Send_Message
				(event.Message
				.Set (TABLE_PARAMETER_NAME,
					The_Management.Procedures ())
				.Reply_To (event.Message));}
			catch (Exception e) {}
			break;

		case CONDUCTOR_STATE_CODE:
			try {Send_Message
					(Processing_Changes
						(CONDUCTOR_STATE_ACTION, The_Management.State ())
					.Reply_To (event.Message));}
			catch (Exception e) {}
			break;

		case ADD_PROCESSING_LISTENER_CODE:
			Add_Processing_Listener_Route (event);
			break;

		case REMOVE_PROCESSING_LISTENER_CODE:
			Remove_Processing_Listener_Route (event.Message.Route_From ());
			break;

		case ADD_LOG_WRITER_CODE:
			Add_Log_Writer_Route (event.Message.Route_From ());
			break;

		case REMOVE_LOG_WRITER_CODE:
			Remove_Log_Writer_Route (event.Message.Route_From ());
			break;

		case ENABLE_LOG_WRITER_CODE:
			if ((string = event.Message.Get (VALUE_PARAMETER_NAME)) != null)
				Enable_Log_Writer_Route (event.Message.Route_From (),
					string.equals ("true"));
			break;

		case GET_POLL_INTERVAL_CODE:
			try {Send_Message
				(event.Message
				.Set (VALUE_PARAMETER_NAME,
					The_Management.Poll_Interval ())
				.Reply_To (event.Message));}
			catch (Exception e) {}
			break;

		case SET_POLL_INTERVAL_CODE:
			if ((value = event.Message.Value_of (VALUE_PARAMETER_NAME)) != null)
				{
				try {The_Management.Poll_Interval ((int)value.long_Data ());}
				catch (PVL_Exception exception)
					{
					// The Value is not an integer.
					try {Send_Message
						(NACK ("Invalid " + SET_POLL_INTERVAL_ACTION + ' '
							+ VALUE_PARAMETER_NAME + ": " + value.Description (),
							exception));}
					catch (Exception e) {}
					}
				}
			break;

		case GET_RESOLVER_DEFAULT_VALUE_CODE:
			if ((string = The_Management.Resolver_Default_Value ()) == null)
				string = "null";
			try {Send_Message (event.Message
					.Set (VALUE_PARAMETER_NAME,
						string)
					.Reply_To (event.Message));}
			catch (Exception e) {}
			break;

		case SET_RESOLVER_DEFAULT_VALUE_CODE:
			if ((string = event.Message.Get (VALUE_PARAMETER_NAME)) != null)
				{
				if (string.equalsIgnoreCase ("null"))
					string = null;
				The_Management.Resolver_Default_Value (string);
				}
			break;

		case GET_STOP_ON_FAILURE_CODE:
			try {Send_Message
				(event.Message
				.Set (VALUE_PARAMETER_NAME,
					The_Management.Stop_on_Failure ())
				.Reply_To (event.Message));}
			catch (Exception e) {}
			break;

		case SET_STOP_ON_FAILURE_CODE:
			if ((value = event.Message.Value_of (VALUE_PARAMETER_NAME)) != null)
				{
				try {The_Management.Stop_on_Failure ((int)value.long_Data ());}
				catch (PVL_Exception exception)
					{
					// The Value is not an integer.
					try {Send_Message
						(NACK ("Invalid " + SET_STOP_ON_FAILURE_ACTION + ' '
							+ VALUE_PARAMETER_NAME + ": " + value.Description (),
							exception));}
					catch (Exception e) {}
					}
				}
			break;

		case SEQUENTIAL_FAILURES_CODE:
			try {Send_Message
				(event.Message
				.Set (VALUE_PARAMETER_NAME,
					The_Management.Sequential_Failures ())
				.Reply_To (event.Message));}
			catch (Exception e) {}
			break;

		case RESET_SEQUENTIAL_FAILURES_CODE:
			The_Management.Reset_Sequential_Failures ();
			break;

		case PROCESSING_EXCEPTION_CODE:
			Exception
				processing_exception = The_Management.Processing_Exception ();
			try {Send_Message
				(event.Message
				.Set (VALUE_PARAMETER_NAME,
					((processing_exception == null) ? "null" :
					processing_exception.toString ()))
				.Reply_To (event.Message));}
			catch (Exception e) {}
			break;


		case QUIT_CODE:
			The_Management.Quit ();
			break;

		//	Basic Messages:

		case MESSENGERS_REPORT_CODE:
			//	Ignored.
			break;

		case DONE_CODE:
			if ((DEBUG & DEBUG_CONNECTION) != 0)
				System.out.println
					("==> Local_Theater.Message_Delivered: Done");
			Close ();

			if (Auto_Open)
				{
				//	Let the dust settle.
				try {sleep (AUTO_OPEN_DELAY * 1000);}
				catch (InterruptedException exception) {}

				if ((DEBUG & DEBUG_CONNECTION) != 0)
					System.out.println
						("    Re-opening (Auto_Open) the Theater");
				try {Open (Theater_Key);}
				catch (IOException exception) {}
				}
			if ((DEBUG & DEBUG_CONNECTION) != 0)
				System.out.println
					("<== Local_Theater.Message_Delivered: Done");
			break;

		case UNDELIVERABLE_CODE:
			Undeliverable_Messages++;
			/*
				The problem is knowing when to take action to prevent
				constantly repeating message problems.

			Remove_Log_Writer_Route (event.Message.Route_From ());
			Remove_Processing_Listener_Route (event.Message.Route_From ());
			*/
			break;

		case NACK_CODE:
			//!!!	Error handling or reporting?
			NACK_Messages++;
			break;

		default:
			Unrecognized_Messages++;
			break;
		}
	if ((DEBUG & DEBUG_MESSAGES) != 0)
		System.out.println
			(NL
			+"<-- Local_Theater.Assistant");
	}
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		("<== Local_Theater.Assistant quit");
Assistant = null;
}
}	//	Assistant class.

/*==============================================================================
	Processing_Listener
*/
/**	The Processing_Listener interface implementation.
<p>
	This method is used by the Conductor Management interface.
<p>
	@param	event	The Conductor Management Processing_Event that has
		occurred.
*/
public void Processing_Event_Occurred
	(
	Processing_Event	event
	)
{
if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
	System.out.println
		(">>> Local_Theater.Processing_Event_Occurred:" + NL
		+ event.Changes);

Message
	message =
		Processing_Changes (PROCESSING_CHANGES_ACTION, event.Changes);
if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
	System.out.println
		("    Message -" + NL
		+ message);

Vector<Value>
	remove_routes = new Vector<Value> ();
int
	count;
synchronized (Processing_Listener_Routes)
	{
	Value
		route;
	int
		index = Processing_Listener_Routes.size ();
	if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
		System.out.println
			("    " + index + " Processing_Listener_Routes -");
	Notify_Listener:
	while (--index >= 0)
		{
		route = Processing_Listener_Routes.get (index);
		count = remove_routes.size ();
		while (--count >= 0)
			if (route == remove_routes.get (count))
				continue Notify_Listener;

		if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
			System.out.println
				("      " + route.Description ());
		try {Send_Message (message.Route_To (route));}
		catch (Exception exception)
			{
			/*
				Either the Route_To Value is invalid (shouldn't happen)
				or the message could not be sent (IOException).
				In either case the route must be removed from the
				Processing_Listener_Routes list.
			*/
			remove_routes.add (route);
			}
		}

	}
count = remove_routes.size ();
if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
	System.out.println
		("    " + count + " routes to remove -");
while (--count >= 0)
	Remove_Processing_Listener_Route (remove_routes.get (count));
remove_routes = null;
if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
	System.out.println
		("<<< Local_Theater.Processing_Event_Occurred");
}


private void Add_Processing_Listener_Route
	(
	Message_Delivered_Event	event
	)
{
Value
	route = event.Message.Route_From ();
if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
	System.out.println
		(">>> Local_Theater.Add_Processing_Listener_Route: "
			+ route.Description ());
boolean
	report = false;
synchronized (Processing_Listener_Routes)
	{
	if (Processing_Listener_Routes.isEmpty ())
		{
		if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
			System.out.println
				("    Processing_Listener_Routes is empty");
		Processing_Listener_Routes.add (route);
		/*
			Note that the Conductor will generate a Processing_Event
			that reports the current Conductor state.

			Since this is the first listener route it will be the
			only one to receive this report, as intended.
		*/
		The_Management.Add_Processing_Listener (this);
		}
	else
		{
		int
			index = Processing_Listener_Routes.size ();
		while (--index >= 0)
			{
			if ((Processing_Listener_Routes.get (index)).equals (route))
				{
				if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
					System.out.println
						("    Processing_Listener_Routes contains route" + NL
						+"<<< Local_Theater.Add_Processing_Listener_Route");
				return;
				}
			}

		Processing_Listener_Routes.add (route);
		report = true;
		}
	}

if (report)
	{
	//	Send the current Conductor state Processing_Changes to the new listener.
	Message
		message =
			Processing_Changes
				(PROCESSING_CHANGES_ACTION, The_Management.State ());
	if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
		System.out.println
			("    Processing_Changes report -" + NL
			+ message);
	try {Send_Message (message.Reply_To (event.Message));}
	catch (Exception exception)
		{
		/*
			Either the Route_To Value is invalid (shouldn't happen)
			or the message could not be sent (IOException).
			In either case the route must be removed from the
			Processing_Listener_Routes list.
		*/
		Remove_Processing_Listener_Route (route);
		}
	}
if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
	System.out.println
		("<<< Local_Theater.Add_Processing_Listener_Route");
}


private void Remove_Processing_Listener_Route
	(
	Value	route
	)
{
if (route == null)
	return;
if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
	System.out.println
		("<<< Local_Theater.Remove_Processing_Listener_Route : "
			+ route.Description ());

synchronized (Processing_Listener_Routes)
	{
	if (Processing_Listener_Routes.isEmpty ())
		return;

	int
		index = Processing_Listener_Routes.size ();
	while (--index >= 0)
		{
		if ((Processing_Listener_Routes.get (index)).equals (route))
			{
			Processing_Listener_Routes.remove (index);
			break;
			}
		}
	if (Processing_Listener_Routes.isEmpty ())
		The_Management.Remove_Processing_Listener (this);
	if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
		System.out.println
			("    " + Processing_Listener_Routes.size ()
				+ " Processing_Listener_Routes remaining" + NL
			+"<<< Local_Theater.Remove_Processing_Listener_Route");
	}
}

/*==============================================================================
	Log_Writer
*/
private void Add_Log_Writer_Route
	(
	Value	route
	)
{
if ((DEBUG & DEBUG_LOG_WRITER) != 0)
	System.out.println
		(">>> Local_Theater.Add_Log_Writer_Route: "
			+ route.Description ());
if (Log_Writer.Is_Empty ())
	{
	if ((DEBUG & DEBUG_LOG_WRITER) != 0)
		System.out.println
			("    Log_Writer is empty");
	Log_Writer.Add (new Messenger_Styled_Writer (Messenger (), route));
	The_Management.Add_Log_Writer (Log_Writer);
	}
else
if (Log_Writer_Contains (route) == null)
	Log_Writer.Add (new Messenger_Styled_Writer (Messenger (), route));
if ((DEBUG & DEBUG_LOG_WRITER) != 0)
	System.out.println
		("    " + Log_Writer.Writers ().size () + " Log_Writer entries" + NL
		+"<<< Local_Theater.Add_Log_Writer_Route");
}


private boolean Remove_Log_Writer_Route
	(
	Value	route
	)
{
if (route == null ||
	Log_Writer.Is_Empty ())
	return false;
if ((DEBUG & DEBUG_LOG_WRITER) != 0)
	System.out.println
		(">>> Local_Theater.Remove_Log_Writer_Route: "
			+ route.Description ());

Suspendable_Styled_Writer
	writer = Log_Writer_Contains (route);
if (writer != null)
	{
	Log_Writer.Remove (writer.Writer ());
	if (Log_Writer.Is_Empty ())
		{
		if ((DEBUG & DEBUG_LOG_WRITER) != 0)
			System.out.println
				("    Log_Writer is empty");
		The_Management.Remove_Log_Writer (Log_Writer);
		}
	}
if ((DEBUG & DEBUG_LOG_WRITER) != 0)
	System.out.println
		("<<< Local_Theater.Remove_Log_Writer_Route: " + (writer != null));
return writer != null;
}


private void Enable_Log_Writer_Route
	(
	Value	route,
	boolean	enable
	)
{
if (Log_Writer.Is_Empty ())
	return;

if ((DEBUG & DEBUG_LOG_WRITER) != 0)
	System.out.println
		(">>> Local_Theater.Enable_Log_Writer_Route: "
			+ (enable ? "En" : "Dis") + "able " + route.Description ());
Suspendable_Styled_Writer
	writer = Log_Writer_Contains (route);
if (writer != null)
	Log_Writer.Suspend (writer.Writer (), enable);
if ((DEBUG & DEBUG_LOG_WRITER) != 0)
	System.out.println
		("<<< Local_Theater.Enable_Log_Writer_Route");
}


private Suspendable_Styled_Writer Log_Writer_Contains
	(
	Value	route
	)
{
if ((DEBUG & DEBUG_LOG_WRITER) != 0)
	System.out.println
		(">>> Local_Theater.Log_Writer_Contains: "
			+ route.Description ());
synchronized (Log_Writer)
	{
	Vector
		writers = Log_Writer.Writers ();
	Suspendable_Styled_Writer
		writer;
	int
		index = writers.size ();
	while (--index >= 0)
		{
		writer = (Suspendable_Styled_Writer)writers.get (index);
		if (((Messenger_Styled_Writer)writer.Writer ()).Route ().equals (route))
			{
			if ((DEBUG & DEBUG_LOG_WRITER) != 0)
				System.out.println
					("<<< Local_Theater.Log_Writer_Contains: " + writer);
			return writer;
			}
		}
	}
if ((DEBUG & DEBUG_LOG_WRITER) != 0)
	System.out.println
		("<<< Local_Theater.Log_Writer_Contains: null");
return null;
}


}
